;;; -*- coding: japanese-shift-jis -*-
;;; emacspeak-ja-tbl.el:
;;;     This module defines an explanatory-reading dictionary of 2-byte
;;;     Japanese characters and provides some functions to access it.
;;;
;;; $Id: emacspeak-ja-tbl.el,v 1.1.2.4 2002/05/04 16:17:48 inoue Exp $

;;{{{ Require
(eval-when-compile
  (require 'cl))
;;}}}
;;{{{ Character Replacing for Japanese string.
(defconst dtk-fix-characters-regexp-ja
  "[][{}<>\\|`#()opuvij]"
  "regexp of characters which need to be fixed in Japanese context.")

(defvar dtk-replace-characters-table-ja nil
  "Replacement table for Japanese output.
All characters `dtk-bracket-regexp-ja' matches should be included.")

(setq dtk-replace-characters-table-ja
      '(("(" . "JbRqL")
	(")" . "JbRgW")
	("[" . "_CJbRqL")
	("]" . "_CJbRgW")
	("{" . "`[JbRqL")
	("}" . "`[JbRgW")
	("`" . "ANTO[u")
	("#" . "V[v")
	("\\" . "G}[N")
	("|" . "^eZ")
	("<" . "V[i")
	(">" . "_Ci")
	("i" . "JbR")
	("j" . "JbRgW")
	("u" . "JMqL")
	("v" . "JMgW")))
;;}}}
;;{{{ character-to-speech-table-ja
;; dtk-character-to-speech-table is a table maps characters less than 256
;; to its speech string. For Japanese TTS, we first inherit the table,
;; and change it in some part.
(defvar dtk-character-to-speech-table-ja
  "Character-to-speech table for Japanese TTS.")

(defvar emacspeak-m17n-ja-char-speech-list
  "assoc list from char to modify its speech string and the speech string.")

(setq emacspeak-m17n-ja-char-speech-list
      '(
	(28 . "control backslash")
	(29 . "control `[JbRgW")
	(30 . "control caret")
	(31 . "control underscore")
	(33 . "J^t")
	(34 . "NI[e[V")
	(35 . "V[v")
	(36 . "h")
	(40 . "JbRqL")
	(41 . "JbRgW")
	(45 . "}CiX")
	(46 . "sIh")
	(60 . "V[i")
	(62 . "_Ci")
	(63 . "NGX`")
	(91 . "_CJbRqL")
	(93 . "_CJbRgW")
	(123 . "`[JbRqL")
	(125 . "`[JbRgW")
	))

(defun emacspeak-m17n-ja-setup-character-to-speech-table ()
  "Setup character-to-speech-table for Japanese speech output.
It covers only the same range as original dtk-character-to-speech-table,
but some characters can be redefined."
  (setq dtk-character-to-speech-table-ja
	(copy-sequence dtk-character-to-speech-table))
  ;; Fix some characters
  (mapcar
   (function (lambda (x)
	       (aset dtk-character-to-speech-table-ja
		     (car x) (cdr x))))
   emacspeak-m17n-ja-char-speech-list))
;;}}}
;;{{{
;; Auditory-disp-table-ja and its access functions
;; This table has 5 fields:
;; 1) Key: Kanji or other 2-byte Japanese char ""
;; 2) Value 1: read one char according to a cursor movement ""
;; 3) Value 2: explain Kanji to identify it while Kana-Kanji translation ""
;; 4) Value 3: almost same as 3) but explain Kanji in detail.
;; 5) Symbol represents type of the character.
;; The table provided with this package is based on GrassRoots,
;; a Japanese screen reader for DOS
;; developed by Jun Ishikawa <ishikawa@u-shizuoka-ken.ac.jp>
(defvar auditory-display-table-ja nil
  "char-table from a Japanese character and corresponding explanatory readings")

(defun emacspeak-m17n-ja-setup-auditory-display-table ()
  "Setup auditory-display-table for Japanese."
  (put 'auditory-display-table 'char-table-extra-slots 6)
  (setq auditory-display-table-ja
	(make-char-table 'auditory-display-table))
(mapcar
 (function
  (lambda (dictionary)
    (aset auditory-display-table-ja
          (string-to-char (first dictionary)) ; Key
          (vector                             ; Values
                  (second dictionary)
                  (third dictionary)
                  (fourth dictionary)
		  (fifth dictionary)))))
'( ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 1byte characters
("" "}" "}" "}" katakana)
("" "JMqL" "JMqL" "JMqL" katakana)
("" "JMgW" "JMgW" "JMgW" katakana)
("" "e" "e" "e" katakana)
("" "iJe" "iJe" "iJe" katakana)
("" "" "Im@I" "Im@I" katakana)
("" "@" "ATqm@A" "ATqm@A" katakana)
("" "B" "Cnm@C" "Cnm@C" katakana)
("" "D" "EGmm@E" "EGmm@E" katakana)
("" "F" "GCSm@G" "GCSm@G" katakana)
("" "H" "IITJm@I" "IITJm@I" katakana)
("" "" "}g @" "}gm@" katakana)
("" "" "~m@" "~m@" katakana)
("" "" "Vmm@" "Vmm@" katakana)
("" "b" "cJm@c" "cJm@c" katakana)
("" "`[I" "`[I" "`[I" katakana)
("" "A" "ATqm@A" "ATqm@A" katakana)
("" "C" "Cnm@C" "Cnm@C" katakana)
("" "E" "EGmm@E" "EGmm@E" katakana)
("" "G" "GCSm@G" "GCSm@G" katakana)
("" "I" "IITJm@I" "IITJm@I" katakana)
("" "J" "JZm@J" "JZm@J" katakana)
("" "L" "Lbem@L" "Lbem@L" katakana)
("" "N" "Num@N" "Num@N" katakana)
("" "P" "PVLm@P" "PVLm@P" katakana)
("" "R" "Rhm@R" "Rhm@R" katakana)
("" "T" "TNm@T" "TNm@T" katakana)
("" "V" "Vum@V" "Vum@V" katakana)
("" "X" "XYm@X" "XYm@X" katakana)
("" "Z" "ZJCm@Z" "ZJCm@Z" katakana)
("" "\" "\om@\" "\om@\" katakana)
("" "^" "^oRm@^" "^oRm@^" katakana)
("" "`" "`hm@`" "`hm@`" katakana)
("" "c" "cJm@c" "cJm@c" katakana)
("" "e" "eK~m@e" "eK~m@e" katakana)
("" "g" "g[L[m@g" "g[L[m@g" katakana)
("" "i" "iSm@i" "iSm@i" katakana)
("" "j" "jzm@j" "jzm@j" katakana)
("" "k" "k}Ym@k" "k}Ym@k" katakana)
("" "l" "lY~m@l" "lY~m@l" katakana)
("" "m" "mnm@m" "mnm@m" katakana)
("" "n" "nKLm@n" "nKLm@n" katakana)
("" "q" "qR[Lm@q" "qR[Lm@q" katakana)
("" "t" "tWTm@t" "tWTm@t" katakana)
("" "w" "wCm@w" "wCm@w" katakana)
("" "z" "zPm@z" "zPm@z" katakana)
("" "}" "}b`m@}" "}b`m@}" katakana)
("" "~" "~JTm@~" "~JTm@~" katakana)
("" "" "Zm@" "Zm@" katakana)
("" "" "CWm@" "CWm@" katakana)
("" "" "~Wm@" "~Wm@" katakana)
("" "" "}gm@" "}gm@" katakana)
("" "" "~m@" "~m@" katakana)
("" "" "Vmm@" "Vmm@" katakana)
("" "" "WIm@" "WIm@" katakana)
("" "" "Sm@" "Sm@" katakana)
("" "" "XCm@" "XCm@" katakana)
("" "" "Qm@" "Qm@" katakana)
("" "" "[}m@" "[}m@" katakana)
("" "" "rm@" "rm@" katakana)
("" "" "IV}Cm@" "IV}Cm@" katakana)
("" "_Ne" "_Ne" "_Ne" katakana)
("" "n_Ne" "n_Ne" "n_Ne" katakana)
;; 2byte characters
("@" "Xy[X" "Xy[X" "Xy[X" ja-other)
("A" "e" "e" "e" ja-other)
("B" "}" "}" "}" ja-other)
("C" "J}" "J}" "J}" ja-other)
("D" "sIh" "sIh" "sIh" ja-other)
("E" "iJe" "iJe" "iJe" ja-other)
("F" "R" "R" "R" ja-other)
("G" "Z~R" "Z~R" "Z~R" ja-other)
("H" "Mt" "Mt" "Mt" ja-other)
("I" "J^t" "J^t" "J^t" ja-other)
("J" "_Ne" "_Ne" "_Ne" ja-other)
("K" "n_Ne" "n_Ne" "n_Ne" ja-other)
("L" "ANTeBM[" "ANTeBM[" "ANTeBM[" ja-other)
("M" "ANTO[u" "ANTO[u" "ANTO[u" ja-other)
("N" "EEg" "EEg" "EEg" ja-other)
("O" "xLW[" "xLW[" "xLW[" ja-other)
("P" "I[o[C" "I[o[C" "I[o[C" ja-other)
("Q" "A_[" "A_[" "A_[" ja-other)
("R" "J^JiNJGV" "J^JiNJGV" "J^JiNJGV" ja-other)
("S" "J^Ji_NeNJGV" "J^Ji_NeNJGV" "J^Ji_NeNJGV" ja-other)
("T" "qKiNJGV" "qKiNJGV" "qKiNJGV" ja-other)
("U" "qKi_NeNJGV" "qKi_NeNJGV" "qKi_NeNJGV" ja-other)
("V" "IiWN" "IiWN" "IiWN" ja-other)
("W" "h[W[" "h[W[" "h[W[" ja-other)
("X" "NJGVLS[" "NJGVLS[" "NJGVLS[" ja-other)
("Y" "V" "V" "V" ja-other)
("Z" "JX[W[" "JX[W[" "JX[W[" ja-other)
("[" "`[I" "`[I" "`[I" ja-katakana)
("\" "_bV" "_bV" "_bV" ja-other)
("]" "nCt" "nCt" "nCt" ja-other)
("^" "XbV" "XbV" "XbV" ja-other)
("_" "obNXbV" "obNXbV" "obNXbV" ja-other)
("`" "i~Z" "i~Z" "i~Z" ja-other)
("a" "wCR[" "wCR[" "wCR[" ja-other)
("b" "^eZ" "^eZ" "^eZ" ja-other)
("c" "Te" "Te" "Te" ja-other)
("d" "je" "je" "je" ja-other)
("e" "q_C[t" "q_C[t" "q_C[t" ja-other)
("f" "~MC[t" "~MC[t" "~MC[t" ja-other)
("g" "q_jW[C[t" "q_jW[C[t" "q_jW[C[t" ja-other)
("h" "~MjW[C[t" "~MjW[C[t" "~MjW[C[t" ja-other)
("i" "JbRqL" "JbRqL" "JbRqL" ja-other)
("j" "JbRgW" "JbRgW" "JbRgW" ja-other)
("k" "LbR[" "LbR[" "LbR[" ja-other)
("l" "gWLbR[" "gWLbR[" "gWLbR[" ja-other)
("m" "_CJbRqL" "_CJbRqL" "_CJbRqL" ja-other)
("n" "_CJbRgW" "_CJbRgW" "_CJbRgW" ja-other)
("o" "`[JbRqL" "`[JbRqL" "`[JbRqL" ja-other)
("p" "`[JbRgW" "`[JbRgW" "`[JbRgW" ja-other)
("q" "}JbR" "}JbR" "}JbR" ja-other)
("r" "gW}JbR" "gW}JbR" "gW}JbR" ja-other)
("s" "jW[}JbR" "jW[}JbR" "jW[}JbR" ja-other)
("t" "gWjW[}JbR" "gWjW[}JbR" "gWjW[}JbR" ja-other)
("u" "JMqL" "JMqL" "JMqL" ja-other)
("v" "JMgW" "JMgW" "JMgW" ja-other)
("w" "jW[JM" "jW[JM" "jW[JM" ja-other)
("x" "jW[JMgW" "jW[JMgW" "jW[JMgW" ja-other)
("y" "X~cL" "X~cL" "X~cL" ja-other)
("z" "X~cLgW" "X~cLgW" "X~cLgW" ja-other)
("{" "vX" "vX" "vX" ja-other)
("|" "}CiX" "}CiX" "}CiX" ja-other)
("}" "vX}CiX" "vX}CiX" "vX}CiX" ja-other)
("~" "JP" "JP" "JP" ja-other)
("" "" "" "" ja-other)
("" "CR[" "CR[" "CR[" ja-other)
("" "mbgCR[" "mbgCR[" "mbgCR[" ja-other)
("" "V[i" "V[i" "V[i" ja-other)
("" "_Ci" "_Ci" "_Ci" ja-other)
("" "V[iCR[" "V[iCR[" "V[iCR[" ja-other)
("" "_CiCR[" "_CiCR[" "_CiCR[" ja-other)
("" "Q_C" "Q_C" "Q_C" ja-other)
("" "Gj" "Gj" "Gj" ja-other)
("" "_ZC" "_ZC" "_ZC" ja-other)
("" "WZC" "WZC" "WZC" ja-other)
("" "h" "h" "h" ja-other)
("" "t" "t" "t" ja-other)
("" "r[" "r[" "r[" ja-other)
("" "hV[" "hV[" "hV[" ja-other)
("" "G}[N" "G}[N" "G}[N" ja-other)
("" "h" "h" "h" ja-other)
("" "Zg" "Zg" "Zg" ja-other)
("" "|h" "|h" "|h" ja-other)
("" "p[Zg" "p[Zg" "p[Zg" ja-other)
("" "V[v" "V[v" "V[v" ja-other)
("" "ApTh" "ApTh" "ApTh" ja-other)
("" "AX^XN" "AX^XN" "AX^XN" ja-other)
("" "Abg" "Abg" "Abg" ja-other)
("" "ZNV" "ZNV" "ZNV" ja-other)
("" "V{V" "V{V" "V{V" ja-other)
("" "N{V" "N{V" "N{V" ja-other)
("" "V}" "V}" "V}" ja-other)
("" "N}" "N}" "N}" ja-other)
("" "jW[}" "jW[}" "jW[}" ja-other)
("" "qVK^" "qVK^" "qVK^" ja-other)
("" "NqVK^" "NqVK^" "NqVK^" ja-other)
("" "VJN" "VJN" "VJN" ja-other)
("" "NVJN" "NVJN" "NVJN" ja-other)
("" "TJN" "TJN" "TJN" ja-other)
("" "NTJN" "NTJN" "NTJN" ja-other)
("" "MNTJN" "MNTJN" "MNTJN" ja-other)
("" "MNNTJN" "MNNTJN" "MNNTJN" ja-other)
("" "RWV" "RWV" "RWV" ja-other)
("" "[r" "[r" "[r" ja-other)
("" "~MWV" "~MWV" "~MWV" ja-other)
("" "q_WV" "q_WV" "q_WV" ja-other)
("" "EGWV" "EGWV" "EGWV" ja-other)
("" "V^WV" "V^WV" "V^WV" ja-other)
("" "NMLS[" "NMLS[" "NMLS[" ja-other)
("" "]NX" "]NX" "]NX" ja-other)
("" "ggVetN" "ggVetN" "ggVetN" ja-other)
("" "uuV[S[" "uuV[S[" "uuV[S[" ja-other)
("" "uuItN" "uuItN" "uuItN" ja-other)
("" "VtuV[S[" "VtuV[S[" "VtuV[S[" ja-other)
("" "VuItN" "VuItN" "VuItN" ja-other)
("" "KbyCV[S[" "KbyCV[S[" "KbyCV[S[" ja-other)
("" "L[c[V[S[" "L[c[V[S[" "L[c[V[S[" ja-other)
("" "S[ZcLS[" "S[ZcLS[" "S[ZcLS[" ja-other)
("" "ZcLS[" "ZcLS[" "ZcLS[" ja-other)
("" "qeCLS[" "qeCLS[" "qeCLS[" ja-other)
("" "S[CLS[" "S[CLS[" "S[CLS[" ja-other)
("" "h[`LS[" "h[`LS[" "h[`LS[" ja-other)
("" "tc[QeCV" "tc[QeCV" "tc[QeCV" ja-other)
("" "\UCQeCV" "\UCQeCV" "\UCQeCV" ja-other)
("" "JNLS[" "JNLS[" "JNLS[" ja-other)
("" "XC`N" "XC`N" "XC`N" ja-other)
("" "RLS[" "RLS[" "RLS[" ja-other)
("" "EhfB[" "EhfB[" "EhfB[" ja-other)
("" "iuLS[" "iuLS[" "iuLS[" ja-other)
("" "cljqgVCLS[" "cljqgVCLS[" "cljqgVCLS[" ja-other)
("" "zghqgVC" "zghqgVC" "zghqgVC" ja-other)
("" "qW[j`CTCLS[" "qW[j`CTCLS[" "qW[j`CTCLS[" ja-other)
("" "qW[jI[LCLS[" "qW[jI[LCLS[" "qW[jI[LCLS[" ja-other)
("" "RLS[" "RLS[" "RLS[" ja-other)
("" "\[WLS[" "\[WLS[" "\[WLS[" ja-other)
("" "qCLS[" "qCLS[" "qCLS[" ja-other)
("" "i[ioLS[" "i[ioLS[" "i[ioLS[" ja-other)
("" "ZLuLS[" "ZLuLS[" "ZLuLS[" ja-other)
("" "jW[ZLuLS[" "jW[ZLuLS[" "jW[ZLuLS[" ja-other)
("" "IOXg[" "IOXg[" "IOXg[" ja-other)
("" "p[~" "p[~" "p[~" ja-other)
("" "V[v" "V[v" "V[v" ja-other)
("" "tbg" "tbg" "tbg" ja-other)
("" "Iv" "Iv" "Iv" ja-other)
("" "_K[" "_K[" "_K[" ja-other)
("" "_u_K[" "_u_K[" "_u_K[" ja-other)
("" "_NLS[" "_NLS[" "_NLS[" ja-other)
("O" "[" "XEWm [" "XEWm [" ja-number)
("P" "C`" "XEWm C`" "XEWm C`" ja-number)
("Q" "[" "XEWm [" "XEWm [" ja-number)
("R" "T" "XEWm T" "XEWm T" ja-number)
("S" "" "XEWm " "XEWm " ja-number)
("T" "S" "XEWm S" "XEWm S" ja-number)
("U" "N" "XEWm N" "XEWm N" ja-number)
("V" "ii" "XEWm ii" "XEWm ii" ja-number)
("W" "n`" "XEWm n`" "XEWm n`" ja-number)
("X" "L[" "XEWm L[" "XEWm L[" ja-number)
("`" "G[" "At@" "At@" ja-alpha)
("a" "r[" "u{[" "u{[" ja-alpha)
("b" "V[" "`[[" "`[[" ja-alpha)
("c" "f[" "f^" "f^" ja-alpha)
("d" "C[" "GR[" "GR[" ja-alpha)
("e" "Gt" "tHbNXgbg" "tHbNXgbg" ja-alpha)
("f" "W[" "St" "St" ja-alpha)
("g" "GC`" "ze" "ze" ja-alpha)
("h" "AC" "CfBA" "CfBA" ja-alpha)
("i" "WF[" "WGbg" "WGbg" ja-alpha)
("j" "P[" "L" "L" ja-alpha)
("k" "G" "[}" "[}" ja-alpha)
("l" "G" "}CN" "}CN" ja-alpha)
("m" "Gk" "mxo[" "mxo[" ja-alpha)
("n" "I[" "IXJ[" "IXJ[" ja-alpha)
("o" "s[" "pp" "pp" ja-alpha)
("p" "L[" "PxbN" "PxbN" ja-alpha)
("q" "A[" "~I" "~I" ja-alpha)
("r" "GX" "VG" "VG" ja-alpha)
("s" "eB[" "^S" "^S" ja-alpha)
("t" "[" "jtH[" "jtH[" ja-alpha)
("u" "uC" "rN^[" "rN^[" ja-alpha)
("v" "_u[" "ECXL[" "ECXL[" ja-alpha)
("w" "GbNX" "GbNXC" "GbNXC" ja-alpha)
("x" "C" "L[" "L[" ja-alpha)
("y" "[bg" "Y" "Y" ja-alpha)
("" "G[" "At@" "At@" ja-alpha)
("" "r[" "u{[" "u{[" ja-alpha)
("" "V[" "`[[" "`[[" ja-alpha)
("" "f[" "f^" "f^" ja-alpha)
("" "C[" "GR[" "GR[" ja-alpha)
("" "Gt" "tHbNXgbg" "tHbNXgbg" ja-alpha)
("" "W[" "St" "St" ja-alpha)
("" "GC`" "ze" "ze" ja-alpha)
("" "AC" "CfBA" "CfBA" ja-alpha)
("" "WF[" "WGbg" "WGbg" ja-alpha)
("" "P[" "L" "L" ja-alpha)
("" "G" "[}" "[}" ja-alpha)
("" "G" "}CN" "}CN" ja-alpha)
("" "Gk" "mxo[" "mxo[" ja-alpha)
("" "I[" "IXJ[" "IXJ[" ja-alpha)
("" "s[" "pp" "pp" ja-alpha)
("" "L[" "PxbN" "PxbN" ja-alpha)
("" "A[" "~I" "~I" ja-alpha)
("" "GX" "VG" "VG" ja-alpha)
("" "eB[" "^S" "^S" ja-alpha)
("" "[" "jtH[" "jtH[" ja-alpha)
("" "uC" "rN^[" "rN^[" ja-alpha)
("" "_u[" "ECXL[" "ECXL[" ja-alpha)
("" "GbNX" "GbNXC" "GbNXC" ja-alpha)
("" "C" "L[" "L[" ja-alpha)
("" "[bg" "Y" "Y" ja-alpha)
("" "A" "ATqm A" "ATqm A" ja-hiragana)
("" "A" "ATqm A" "ATqm A" ja-hiragana)
("" "C" "Cnm C" "Cnm C" ja-hiragana)
("" "C" "Cnm C" "Cnm C" ja-hiragana)
("" "E" "EGmm E" "EGmm E" ja-hiragana)
("" "E" "EGmm E" "EGmm E" ja-hiragana)
("" "G" "GCSm G" "GCSm G" ja-hiragana)
("" "G" "GCSm G" "GCSm G" ja-hiragana)
("" "I" "IITJm I" "IITJm I" ja-hiragana)
("" "I" "IITJm I" "IITJm I" ja-hiragana)
("" "J" "JZm J" "JZm J" ja-hiragana)
("" "K" "JZm Jj _Ne" "JZm Jj _Ne" ja-hiragana)
("" "L" "Lbem L" "Lbem L" ja-hiragana)
("" "M" "Lbem Lj _Ne" "Lbem Lj _Ne" ja-hiragana)
("" "N" "Num N" "Num N" ja-hiragana)
("" "O" "Num Nj _Ne" "Num Nj _Ne" ja-hiragana)
("" "P" "PVLm P" "PVLm P" ja-hiragana)
("" "Q" "PVLm Pj _Ne" "PVLm Pj _Ne" ja-hiragana)
("" "R" "Rhm R" "Rhm R" ja-hiragana)
("" "S" "Rhm Rj _Ne" "Rhm Rj _Ne" ja-hiragana)
("" "T" "TNm T" "TNm T" ja-hiragana)
("" "U" "TNm Tj _Ne" "TNm Tj _Ne" ja-hiragana)
("" "V" "Vum V" "Vum V" ja-hiragana)
("" "W" "Vum Vj _Ne" "Vum Vj _Ne" ja-hiragana)
("" "X" "XYm X" "XYm X" ja-hiragana)
("" "Y" "XYm Xj _Ne" "XYm Xj _Ne" ja-hiragana)
("" "Z" "ZJCm Z" "ZJCm Z" ja-hiragana)
("" "[" "ZJCm Zj _Ne" "ZJCm Zj _Ne" ja-hiragana)
("" "\" "\om \" "\om \" ja-hiragana)
("" "]" "\om \j _Ne" "\om \j _Ne" ja-hiragana)
("" "^" "^oRm ^" "^oRm ^" ja-hiragana)
("" "_" "^oRm ^j _Ne" "^oRm ^j _Ne" ja-hiragana)
("" "`" "`hm `" "`hm `" ja-hiragana)
("" "a" "`hm `j _Ne" "`hm `j _Ne" ja-hiragana)
("" "c" "cJm c" "cJm c" ja-hiragana)
("" "c" "cJm c" "cJm c" ja-hiragana)
("" "d" "cJm c" "cJm c" ja-hiragana)
("" "e" "eK~m e" "eK~m e" ja-hiragana)
("" "f" "eK~m ej _Ne" "eK~m ej _Ne" ja-hiragana)
("" "g" "gELEm g" "gELEm g" ja-hiragana)
("" "h" "gELEm gj _Ne" "gELEm gj _Ne" ja-hiragana)
("" "i" "iSm i" "iSm i" ja-hiragana)
("" "j" "jzm j" "jzm j" ja-hiragana)
("" "k" "k}Ym k" "k}Ym k" ja-hiragana)
("" "l" "lY~m l" "lY~m l" ja-hiragana)
("" "m" "mnm m" "mnm m" ja-hiragana)
("" "n" "nKLm n" "nKLm n" ja-hiragana)
("" "o" "nKLm nj _Ne" "nKLm nj _Ne" ja-hiragana)
("" "p" "nKLm nj n_Ne" "nKLm nj n_Ne" ja-hiragana)
("" "q" "qRELm q" "qRELm q" ja-hiragana)
("" "r" "qRELm qj _Ne" "qRELm qj _Ne" ja-hiragana)
("" "s" "qRELm qj n_Ne" "qRELm qj n_Ne" ja-hiragana)
("" "t" "tWTm t" "tWTm t" ja-hiragana)
("" "u" "tWTm tj _Ne" "tWTm tj _Ne" ja-hiragana)
("" "v" "tWTm tj n_Ne" "tWTm tj n_Ne" ja-hiragana)
("" "w" "wCm w" "wCm w" ja-hiragana)
("" "x" "wCm wj _Ne" "wCm wj _Ne" ja-hiragana)
("" "y" "wCm wj n_Ne" "wCm wj n_Ne" ja-hiragana)
("" "z" "zPm z" "zPm z" ja-hiragana)
("" "{" "zPm zj _Ne" "zPm zj _Ne" ja-hiragana)
("" "|" "zPm zj n_Ne" "zPm zj n_Ne" ja-hiragana)
("" "}" "}b`m }" "}b`m }" ja-hiragana)
("" "~" "~JTm ~" "~JTm ~" ja-hiragana)
("" "" "Zm " "Zm " ja-hiragana)
("" "" "CWm " "CWm " ja-hiragana)
("" "" "~Wm " "~Wm " ja-hiragana)
("" "" "}gm " "}gm " ja-hiragana)
("" "" "}gm " "}gm " ja-hiragana)
("" "" "~m " "~m " ja-hiragana)
("" "" "~m " "~m " ja-hiragana)
("" "" "Vmm " "Vmm " ja-hiragana)
("" "" "Vmm " "Vmm " ja-hiragana)
("" "" "WIm " "WIm " ja-hiragana)
("" "" "Sm " "Sm " ja-hiragana)
("" "" "XCm " "XCm " ja-hiragana)
("" "" "Qm " "Qm " ja-hiragana)
("" "" "[}m " "[}m " ja-hiragana)
("" "" "rm " "rm " ja-hiragana)
("" "" "rm " "rm " ja-hiragana)
("" "C" "C" "C" ja-hiragana)
("" "G" "G" "G" ja-hiragana)
("" "" "IV}Cm " "IV}Cm " ja-hiragana)
("" "" "IV}Cm " "IV}Cm " ja-hiragana)
("@" "A" "ATqm A" "ATqm A" ja-katakana)
("A" "A" "ATqm A" "ATqm A" ja-katakana)
("B" "C" "Cnm C" "Cnm C" ja-katakana)
("C" "C" "Cnm C" "Cnm C" ja-katakana)
("D" "E" "EGmm E" "EGmm E" ja-katakana)
("E" "E" "EGmm E" "EGmm E" ja-katakana)
("F" "G" "GCSm G" "GCSm G" ja-katakana)
("G" "G" "GCSm G" "GCSm G" ja-katakana)
("H" "I" "IITJm I" "IITJm I" ja-katakana)
("I" "I" "IITJm I" "IITJm I" ja-katakana)
("J" "J" "JZm J" "JZm J" ja-katakana)
("K" "K" "JZm Jj _Ne" "JZm Jj _Ne" ja-katakana)
("L" "L" "Lbem L" "Lbem L" ja-katakana)
("M" "M" "Lbem Lj _Ne" "Lbem Lj _Ne" ja-katakana)
("N" "N" "Num N" "Num N" ja-katakana)
("O" "O" "Num Nj _Ne" "Num Nj _Ne" ja-katakana)
("P" "P" "PVLm P" "PVLm P" ja-katakana)
("Q" "Q" "PVLm Pj _Ne" "PVLm Pj _Ne" ja-katakana)
("R" "R" "Rhm R" "Rhm R" ja-katakana)
("S" "S" "Rhm Rj _Ne" "Rhm Rj _Ne" ja-katakana)
("T" "T" "TNm T" "TNm T" ja-katakana)
("U" "U" "TNm Tj _Ne" "TNm Tj _Ne" ja-katakana)
("V" "V" "Vum V" "Vum V" ja-katakana)
("W" "W" "Vum Vj _Ne" "Vum Vj _Ne" ja-katakana)
("X" "X" "XYm X" "XYm X" ja-katakana)
("Y" "Y" "XYm Xj _Ne" "XYm Xj _Ne" ja-katakana)
("Z" "Z" "ZJCm Z" "ZJCm Z" ja-katakana)
("[" "[" "ZJCm Zj _Ne" "ZJCm Zj _Ne" ja-katakana)
("\" "\" "\om \" "\om \" ja-katakana)
("]" "]" "\om \j _Ne" "\om \j _Ne" ja-katakana)
("^" "^" "^oRm ^" "^oRm ^" ja-katakana)
("_" "_" "^oRm ^j _Ne" "^oRm ^j _Ne" ja-katakana)
("`" "`" "`hm `" "`hm `" ja-katakana)
("a" "a" "`hm `j _Ne" "`hm `j _Ne)" ja-katakana)
("b" "c" "cJm c" "cJm c" ja-katakana)
("c" "c" "cJm c" "cJm c" ja-katakana)
("d" "d" "cJm c" "cJm c" ja-katakana)
("e" "e" "eK~m e" "eK~m e" ja-katakana)
("f" "f" "eK~m ej _Ne" "eK~m ej _Ne" ja-katakana)
("g" "g" "gELEm g" "gELEm g" ja-katakana)
("h" "h" "gELEm gj _Ne" "gELEm gj _Ne" ja-katakana)
("i" "i" "iSm i" "iSm i" ja-katakana)
("j" "j" "jzm j" "jzm j" ja-katakana)
("k" "k" "k}Ym k" "k}Ym k" ja-katakana)
("l" "l" "lY~m l" "lY~m l" ja-katakana)
("m" "m" "mnm m" "mnm m" ja-katakana)
("n" "n" "nKLm n" "nKLm n" ja-katakana)
("o" "o" "nKLm nj _Ne" "nKLm nj _Ne" ja-katakana)
("p" "p" "nKLm nj n_Ne" "nKLm nj n_Ne" ja-katakana)
("q" "q" "qRELm q" "qRELm q" ja-katakana)
("r" "r" "qRELm qj _Ne" "qRELm qj _Ne" ja-katakana)
("s" "s" "qRELm qj n_Ne" "qRELm qj n_Ne" ja-katakana)
("t" "t" "tWTm t" "tWTm t" ja-katakana)
("u" "u" "tWTm tj _Ne" "tWTm tj _Ne" ja-katakana)
("v" "v" "tWTm tj n_Ne" "tWTm tj n_Ne" ja-katakana)
("w" "w" "wCm w" "wCm w" ja-katakana)
("x" "x" "wCm wj _Ne" "wCm wj _Ne" ja-katakana)
("y" "y" "wCm wj n_Ne" "wCm wj n_Ne" ja-katakana)
("z" "z" "zPm z" "zPm z" ja-katakana)
("{" "{" "zPm zj _Ne" "zPm zj _Ne" ja-katakana)
("|" "|" "zPm zj n_Ne" "zPm zj n_Ne" ja-katakana)
("}" "}" "}b`m }" "}b`m }" ja-katakana)
("~" "~" "~JTm ~" "~JTm ~" ja-katakana)
("" "" "Zm " "Zm " ja-katakana)
("" "" "CWm " "CWm " ja-katakana)
("" "" "~Wm " "~Wm " ja-katakana)
("" "" "}gm " "}gm " ja-katakana)
("" "" "}gm " "}gm " ja-katakana)
("" "" "~m " "~m " ja-katakana)
("" "" "~m " "~m " ja-katakana)
("" "" "Vmm " "Vmm " ja-katakana)
("" "" "Vmm " "Vmm " ja-katakana)
("" "" "WIm " "WIm " ja-katakana)
("" "" "Sm " "Sm " ja-katakana)
("" "" "XCm " "XCm " ja-katakana)
("" "" "Qm " "Qm " ja-katakana)
("" "" "[}m " "[}m " ja-katakana)
("" "" "rm " "rm " ja-katakana)
("" "" "" "" ja-katakana)
("" "" "IV}Cm " "IV}Cm " ja-katakana)
("" "" "" "" ja-katakana)
("" "J" "J" "J" ja-katakana)
("" "J" "J" "J" ja-katakana)
("" "IIWAt@" "IIWAt@" "IIWAt@" ja-greek)
("" "IIWx[^" "IIWx[^" "IIWx[^" ja-greek)
("" "IIWK}" "IIWK}" "IIWK}" ja-greek)
("" "IIWf^" "IIWf^" "IIWf^" ja-greek)
("" "IIWCvV" "IIWCvV" "IIWCvV" ja-greek)
("" "IIWWF[^" "IIWWF[^" "IIWWF[^" ja-greek)
("" "IIWC[^" "IIWC[^" "IIWC[^" ja-greek)
("" "IIWV[^" "IIWV[^" "IIWV[^" ja-greek)
("" "IIWCI[^" "IIWCI[^" "IIWCI[^" ja-greek)
("" "IIWJbp" "IIWJbp" "IIWJbp" ja-greek)
("" "IIW_" "IIW_" "IIW_" ja-greek)
("" "IIW~[" "IIW~[" "IIW~[" ja-greek)
("" "IIWj[" "IIWj[" "IIWj[" ja-greek)
("" "IIWNV[" "IIWNV[" "IIWNV[" ja-greek)
("" "IIWI~N" "IIWI~N" "IIWI~N" ja-greek)
("" "IIWpC" "IIWpC" "IIWpC" ja-greek)
("" "IIW[" "IIW[" "IIW[" ja-greek)
("" "IIWVO}" "IIWVO}" "IIWVO}" ja-greek)
("" "IIW^E" "IIW^E" "IIW^E" ja-greek)
("" "IIWEvV" "IIWEvV" "IIWEvV" ja-greek)
("" "IIWtF[" "IIWtF[" "IIWtF[" ja-greek)
("" "IIWL[" "IIWL[" "IIWL[" ja-greek)
("" "IIWvV[" "IIWvV[" "IIWvV[" ja-greek)
("" "IIWIK" "IIWIK" "IIWIK" ja-greek)
("" "At@" "At@" "At@" ja-greek)
("" "x[^" "x[^" "x[^" ja-greek)
("" "K}" "K}" "K}" ja-greek)
("" "f^" "f^" "f^" ja-greek)
("" "CvV" "CvV" "CvV" ja-greek)
("" "WF[^" "WF[^" "WF[^" ja-greek)
("" "C[^" "C[^" "C[^" ja-greek)
("" "V[^" "V[^" "V[^" ja-greek)
("" "CI[^" "CI[^" "CI[^" ja-greek)
("" "Jbp" "Jbp" "Jbp" ja-greek)
("" "_" "_" "_" ja-greek)
("" "~[" "~[" "~[" ja-greek)
("" "j[" "j[" "j[" ja-greek)
("" "NV[" "NV[" "NV[" ja-greek)
("" "I~N" "I~N" "I~N" ja-greek)
("" "pC" "pC" "pC" ja-greek)
("" "[" "[" "[" ja-greek)
("" "VO}" "VO}" "VO}" ja-greek)
("" "^E" "^E" "^E" ja-greek)
("" "EvV" "EvV" "EvV" ja-greek)
("" "tF[" "tF[" "tF[" ja-greek)
("" "L[" "L[" "L[" ja-greek)
("" "vV[" "vV[" "vV[" ja-greek)
("" "IK" "IK" "IK" ja-greek)
("@" "A[" "A[" "A[" ja-cyrillic)
("A" "x[" "x[" "x[" ja-cyrillic)
("B" "uG[" "uG[" "uG[" ja-cyrillic)
("C" "Q[" "Q[" "Q[" ja-cyrillic)
("D" "f[" "f[" "f[" ja-cyrillic)
("E" "CG[" "CG[" "CG[" ja-cyrillic)
("F" "[" "[" "[" ja-cyrillic)
("G" "WG[" "WG[" "WG[" ja-cyrillic)
("H" "[[" "[[" "[[" ja-cyrillic)
("I" "C[" "C[" "C[" ja-cyrillic)
("J" "CN[gJ" "CN[gJ" "CN[gJ" ja-cyrillic)
("K" "J[" "J[" "J[" ja-cyrillic)
("L" "G" "G" "G" ja-cyrillic)
("M" "G" "G" "G" ja-cyrillic)
("N" "Gk" "Gk" "Gk" ja-cyrillic)
("O" "I[" "I[" "I[" ja-cyrillic)
("P" "y[S" "y[S" "y[S" ja-cyrillic)
("Q" "GS" "GS" "GS" ja-cyrillic)
("R" "GXS" "GXS" "GXS" ja-cyrillic)
("S" "e[S" "e[S" "e[S" ja-cyrillic)
("T" "E[S" "E[S" "E[S" ja-cyrillic)
("U" "GtS" "GtS" "GtS" ja-cyrillic)
("V" "n[S" "n[S" "n[S" ja-cyrillic)
("W" "cG[" "cG[" "cG[" ja-cyrillic)
("X" "`F[" "`F[" "`F[" ja-cyrillic)
("Y" "V[" "V[" "V[" ja-cyrillic)
("Z" "VV[" "VV[" "VV[" ja-cyrillic)
("[" "hr[hCG" "hr[hCG" "hr[hCG" ja-cyrillic)
("\" "EC[" "EC[" "EC[" ja-cyrillic)
("]" "CG~[tL[" "CG~[tL[" "CG~[tL[" ja-cyrillic)
("^" "G[" "G[" "G[" ja-cyrillic)
("_" "[" "[" "[" ja-cyrillic)
("`" "[" "[" "[" ja-cyrillic)
("p" "A[S" "A[S" "A[S" ja-cyrillic)
("q" "x[S" "x[S" "x[S" ja-cyrillic)
("r" "uG[S" "uG[S" "uG[S" ja-cyrillic)
("s" "Q[S" "Q[S" "Q[S" ja-cyrillic)
("t" "f[S" "f[S" "f[S" ja-cyrillic)
("u" "CG[S" "CG[S" "CG[S" ja-cyrillic)
("v" "[S" "[S" "[S" ja-cyrillic)
("w" "WG[S" "WG[S" "WG[S" ja-cyrillic)
("x" "[[S" "[[S" "[[S" ja-cyrillic)
("y" "C[S" "C[S" "C[S" ja-cyrillic)
("z" "CN[gJS" "CN[gJS" "CN[gJS" ja-cyrillic)
("{" "J[S" "J[S" "J[S" ja-cyrillic)
("|" "GS" "GS" "GS" ja-cyrillic)
("}" "GS" "GS" "GS" ja-cyrillic)
("~" "GkS" "GkS" "GkS" ja-cyrillic)
("" "I[S" "I[S" "I[S" ja-cyrillic)
("" "y[S" "y[S" "y[S" ja-cyrillic)
("" "GS" "GS" "GS" ja-cyrillic)
("" "GXS" "GXS" "GXS" ja-cyrillic)
("" "e[S" "e[S" "e[S" ja-cyrillic)
("" "E[S" "E[S" "E[S" ja-cyrillic)
("" "GtS" "GtS" "GtS" ja-cyrillic)
("" "n[S" "n[S" "n[S" ja-cyrillic)
("" "cG[S" "cG[S" "cG[S" ja-cyrillic)
("" "`F[S" "`F[S" "`F[S" ja-cyrillic)
("" "V[S" "V[S" "V[S" ja-cyrillic)
("" "VV[S" "VV[S" "VV[S" ja-cyrillic)
("" "hr[hCiYi[NjS" "hr[hCiYi[NjS" "hr[hCiYi[NjS" ja-cyrillic)
("" "EC[S" "EC[S" "EC[S" ja-cyrillic)
("" "~[tL[S" "~[tL[S" "~[tL[S" ja-cyrillic)
("" "G[S" "G[S" "G[S" ja-cyrillic)
("" "[S" "[S" "[S" ja-cyrillic)
("" "[S" "[S" "[S" ja-cyrillic)
("" "PCZ" "PCZ" "PCZ" ja-other)
("" "PCZ" "PCZ" "PCZ" ja-other)
("" "~Mz\V^z\" "~Mz\V^z\" "~Mz\V^z\" ja-other)
("" "q_z\V^z\" "q_z\V^z\" "q_z\V^z\" ja-other)
("" "q_z\EGz\" "q_z\EGz\" "q_z\EGz\" ja-other)
("" "~Mz\EGz\" "~Mz\EGz\" "~Mz\EGz\" ja-other)
("" "~Mz\^ez\" "~Mz\^ez\" "~Mz\^ez\" ja-other)
("" "Rz\V^z\" "Rz\V^z\" "Rz\V^z\" ja-other)
("" "q_z\^ez\" "q_z\^ez\" "q_z\^ez\" ja-other)
("" "Rz\EGz\" "Rz\EGz\" "Rz\EGz\" ja-other)
("" "Rz\^ez\" "Rz\^ez\" "Rz\^ez\" ja-other)
("" "~Mtg^etg" "~Mtg^etg" "~Mtg^etg" ja-other)
("" "RtgV^tg" "RtgV^tg" "RtgV^tg" ja-other)
("" "q_tg^etg" "q_tg^etg" "q_tg^etg" ja-other)
("" "RtgEGtg" "RtgEGtg" "RtgEGtg" ja-other)
("" "Rtg^etg" "Rtg^etg" "Rtg^etg" ja-other)
("" "~Mz\^etg" "~Mz\^etg" "~Mz\^etg" ja-other)
("" "RtgV^z\" "RtgV^z\" "RtgV^z\" ja-other)
("" "q_z\^etg" "q_z\^etg" "q_z\^etg" ja-other)
("" "RtgEGz\" "RtgEGz\" "RtgEGz\" ja-other)
("" "Rtg^ez\" "Rtg^ez\" "Rtg^ez\" ja-other)
("" "~Mtg^ez\" "~Mtg^ez\" "~Mtg^ez\" ja-other)
("" "Rz\V^tg" "Rz\V^tg" "Rz\V^tg" ja-other)
("" "q_tg^ez\" "q_tg^ez\" "q_tg^ez\" ja-other)
("" "Rz\EGtg" "Rz\EGtg" "Rz\EGtg" ja-other)
("" "Rz\^etg" "Rz\^etg" "Rz\^etg" ja-other)
("@" "}C`" "}C`" "}C`" ja-other)
("A" "}j" "}j" "}j" ja-other)
("B" "}T" "}T" "}T" ja-other)
("C" "}" "}" "}" ja-other)
("D" "}S" "}S" "}S" ja-other)
("E" "}N" "}N" "}N" ja-other)
("F" "}ii" "}ii" "}ii" ja-other)
("G" "}n`" "}n`" "}n`" ja-other)
("H" "}L[" "}L[" "}L[" ja-other)
("I" "}W[" "}W[" "}W[" ja-other)
("J" "}W[C`" "}W[C`" "}W[C`" ja-other)
("K" "}W[j" "}W[j" "}W[j" ja-other)
("L" "}W[T" "}W[T" "}W[T" ja-other)
("M" "}W[" "}W[" "}W[" ja-other)
("N" "}W[S" "}W[S" "}W[S" ja-other)
("O" "}W[N" "}W[N" "}W[N" ja-other)
("P" "}W[ii" "}W[ii" "}W[ii" ja-other)
("Q" "}W[n`" "}W[n`" "}W[n`" ja-other)
("R" "}W[L[" "}W[L[" "}W[L[" ja-other)
("S" "}jW[" "}jW[" "}jW[" ja-other)
("T" "C`" "[}XEWm C`" "[}XEWm C`" ja-other)
("U" "j[" "[}XEWm j[" "[}XEWm j[" ja-other)
("V" "T" "[}XEWm T" "[}XEWm T" ja-other)
("W" "" "[}XEWm " "[}XEWm " ja-other)
("X" "S" "[}XEWm S" "[}XEWm S" ja-other)
("Y" "N" "[}XEWm N" "[}XEWm N" ja-other)
("Z" "ii" "[}XEWm ii" "[}XEWm ii" ja-other)
("[" "n`" "[}XEWm n`" "[}XEWm n`" ja-other)
("\" "L[" "[}XEWm L[" "[}XEWm L[" ja-other)
("]" "W[" "[}XEWm W[" "[}XEWm W[" ja-other)
("_" "~" "~" "~" ja-other)
("`" "L" "L" "L" ja-other)
("a" "Z`" "Z`" "Z`" ja-other)
("b" "[g" "[g" "[g" ja-other)
("c" "O" "O" "O" ja-other)
("d" "g" "g" "g" ja-other)
("e" "A[" "A[" "A[" ja-other)
("f" "wN^[" "wN^[" "wN^[" ja-other)
("g" "bg" "bg" "bg" ja-other)
("h" "bg" "bg" "bg" ja-other)
("i" "J[" "J[" "J[" ja-other)
("j" "h" "h" "h" ja-other)
("k" "Zg" "Zg" "Zg" ja-other)
("l" "p[Zg" "p[Zg" "p[Zg" ja-other)
("m" "~o[" "~o[" "~o[" ja-other)
("n" "y[W" "y[W" "y[W" ja-other)
("o" "~[g" "~[g" "~[g" ja-other)
("p" "Z`[g" "Z`[g" "Z`[g" ja-other)
("q" "L[g" "L[g" "L[g" ja-other)
("r" "~O" "~O" "~O" ja-other)
("s" "LO" "LO" "LO" ja-other)
("t" "V[V[" "V[V[" "V[V[" ja-other)
("u" "wCz[[g" "wCz[[g" "wCz[[g" ja-other)
("~" "wCZC" "wCZC" "wCZC" ja-other)
("" "io[" "io[" "io[" ja-other)
("" "JuVLKCV" "JuVLKCV" "JuVLKCV" ja-other)
("" "foSE" "foSE" "foSE" ja-other)
("" "}miJjEG" "}miJjEG" "}miJjEG" ja-other)
("" "}miJj`E" "}miJj`E" "}miJj`E" ja-other)
("" "}miJjV^" "}miJjV^" "}miJjV^" ja-other)
("" "}miJjq_" "}miJjq_" "}miJjq_" ja-other)
("" "}miJj~M" "}miJj~M" "}miJj~M" ja-other)
("" "JuVLKCV" "JuVLKCV" "JuVLKCV" ja-other)
("" "EQKCV" "EQKCV" "EQKCV" ja-other)
("" "_CqE" "_CqE" "_CqE" ja-other)
("" "CW" "CW" "CW" ja-other)
("" "^CV[" "^CV[" "^CV[" ja-other)
("" "V[" "V[" "V[" ja-other)
("" "LW}^g[S[" "LW}^g[S[" "LW}^g[S[" ja-other)
("" "S[h[" "S[h[" "S[h[" ja-other)
("" "ZLuLS[" "ZLuLS[" "ZLuLS[" ja-other)
("" "V[JCZLu" "V[JCZLu" "V[JCZLu" ja-other)
("" "VO}" "VO}" "VO}" ja-other)
("" "wCz[R" "wCz[R" "wCz[R" ja-other)
("" "XC`N" "XC`N" "XC`N" ja-other)
("" "JNh" "JNh" "JNh" ja-other)
("" "`bJN" "`bJN" "`bJN" ja-other)
("" "TJNPC" "TJNPC" "TJNPC" ja-other)
("" "i[io" "i[io" "i[io" ja-other)
("" "Lbv" "Lbv" "Lbv" ja-other)
("" "Jbv" "Jbv" "Jbv" ja-other)
("" "A" "AWAm A" "AWAm A" ja-kanji)
("" "A" "A[g Xm A" "A[g Xm A" ja-kanji)
("" "A" "rWm A" "rWm A" ja-kanji)
("" "A" "A~_ucm A" "A~_ucm A" ja-kanji)
("" "AC" "Am AC" "Am AC" ja-kanji)
("" "AC" "ACWm AC" "ACWm AC" ja-kanji)
("" "AC" "ACTcm AC" "ACTcm AC" ja-kanji)
("" "AC" "ACm AC" "ACm AC" ja-kanji)
("" "z[" "z[`Nm AE" "z[`Nm AE" ja-kanji)
("" "L" "nim AIC" "nim AIC" ja-kanji)
("" "Z" "AJCm AJl" "AJCm AJl" ja-kanji)
("" "L" "Jm AL" "Jm AL" ja-kanji)
("" "AN" "Cm AN" "Cm AN" ja-kanji)
("" "AN" "jMm AN" "jMm AN" ja-kanji)
("" "AN" "AcCm AN" "AcCm AN" ja-kanji)
("" "LN" "LNm ATq" "LNm ATq" ja-kanji)
("" "AV" "AVQm AV" "AVQm AV" ja-kanji)
("" "AV" "VNucm AV" "VNucm AV" ja-kanji)
("" "\[" "TJim AW" "TJim AW" ja-kanji)
("" "V" "Lm AYT" "Lm AYT" ja-kanji)
("" "Ac" "AcNm Ac" "AcNm Ac" ja-kanji)
("" "Ac" "Om Ac" "Om Ac" ja-kanji)
("" "L[" "AcJEm L[" "AcJEm L[" ja-kanji)
("" "Ae" "Aeim Ae" "Aeim Ae" ja-kanji)
("" "Al" "AlSm Al" "AlSm Al" ja-kanji)
("" "{[" "R`[m Au" "R`[m Au" ja-kanji)
("" "C" "^xmm A" "^xmm A" ja-kanji)
("" "P" "Pm A" "Pm A" ja-kanji)
("" "[" "AMkm A" "AMkm A" ja-kanji)
("" "A" "TJim A" "TJim A" ja-kanji)
("" "N" "Nm AC" "Nm AC" ja-kanji)
("" "A" "]Nm A" "]Nm A" ja-kanji)
("" "AZ" "Lmm AZ" "Lmm AZ" ja-kanji)
("" "A" "XCm A" "XCm A" ja-kanji)
("" "A" "CIm A" "CIm A" ja-kanji)
("" "A" "A}m A" "A}m A" ja-kanji)
("" "A" "NCm A" "NCm A" ja-kanji)
("" "A" "AiCm A" "AiCm A" ja-kanji)
("" "A" "~m A" "~m A" ja-kanji)
("" "A" "Nm A" "Nm A" ja-kanji)
("" "L[" "N_mm AY" "N_mm AY" ja-kanji)
("" "C" "bem C" "bem C" ja-kanji)
("" "C" "CYng[m C" "CYng[m C" ja-kanji)
("" "C" "NCm C" "NCm C" ja-kanji)
("" "C" "m C" "m C" ja-kanji)
("" "C" "GCm C" "GCm C" ja-kanji)
("" "C" "JRm C" "JRm C" ja-kanji)
("" "C" "GrXm C" "GrXm C" ja-kanji)
("" "C" "_lm C" "_lm C" ja-kanji)
("" "C" "IhXm C" "IhXm C" ja-kanji)
("" "C" "V[Cm C" "V[Cm C" ja-kanji)
("" "C" "IEm C" "IEm C" ja-kanji)
("" "C" "C~m C" "C~m C" ja-kanji)
("" "C" "iOTm C" "iOTm C" ja-kanji)
("" "C" "TVCm C" "TVCm C" ja-kanji)
("" "C" "CXm C" "CXm C" ja-kanji)
("" "C" "^m C" "^m C" ja-kanji)
("" "C" "JVR}m C" "JVR}m C" ja-kanji)
("" "C" "Rgim C" "Rgim C" ja-kanji)
("" "C" "EcXm C" "EcXm C" ja-kanji)
("" "C" "CWCVm C" "CWCVm C" ja-kanji)
("" "C" "RCgm C" "RCgm C" ja-kanji)
("" "C" "CuNm C" "CuNm C" ja-kanji)
("" "C" "iGm C" "iGm C" ja-kanji)
("" "C" "Rm C" "Rm C" ja-kanji)
("" "C" "Cm C" "Cm C" ja-kanji)
("" "C" "`KEm C" "`KEm C" ja-kanji)
("" "C" "CVm C" "CVm C" ja-kanji)
("" "C" "CVm C" "CVm C" ja-kanji)
("" "C" "Chm C" "Chm C" ja-kanji)
("" "KC" "W[jVm C" "W[jVm C" ja-kanji)
("" "CL" "`CLm CL" "`CLm CL" ja-kanji)
("" "CN" "\_cm CN" "\_cm CN" ja-kanji)
("" "CN" "tNCNm CN" "tNCNm CN" ja-kanji)
("" "L" "C\xm C\" "C\xm C\" ja-kanji)
("" "C`" "JX[Wm C`" "JX[Wm C`" ja-kanji)
("" "C`" "CLm V}m C`" "CLm V}m C`" ja-kanji)
("" "Cc" "Atm Cc" "Atm Cc" ja-kanji)
("" "Cc" "mKXm Cc" "mKXm Cc" ja-kanji)
("" "g[" "Cizm Cl" "Cizm Cl" ja-kanji)
("" "V" "gQm Co" "gQm Co" ja-kanji)
("" "C" "Tc}Cm C" "Tc}Cm C" ja-kanji)
("" "WN" "TJim CV" "TJim CV" ja-kanji)
("" "C" "Xm C" "Xm C" ja-kanji)
("" "C" "VVm C" "VVm C" ja-kanji)
("" "C" "mhm C" "mhm C" ja-kanji)
("" "C" "eCm C" "eCm C" ja-kanji)
("" "C" "QCm C" "QCm C" ja-kanji)
("" "C" "RCm C" "RCm C" ja-kanji)
("" "C" "qNm C" "qNm C" ja-kanji)
("" "C" "mm C" "mm C" ja-kanji)
("" "C" "~_m C" "~_m C" ja-kanji)
("" "C" "^lm C" "^lm C" ja-kanji)
("" "C" "RJQm C" "RJQm C" ja-kanji)
("@" "C" "r[Cm C" "r[Cm C" ja-kanji)
("A" "C" "JQm C" "JQm C" ja-kanji)
("B" "C" "JNXm C" "JNXm C" ja-kanji)
("C" "C" "Cm C" "Cm C" ja-kanji)
("D" "C`" "iKTm C`" "iKTm C`" ja-kanji)
("E" "E" "~Mm E" "~Mm E" ja-kanji)
("F" "E" "E`[m E" "E`[m E" ja-kanji)
("G" "E" "JXm E" "JXm E" ja-kanji)
("H" "E" "nlm E" "nlm E" ja-kanji)
("I" "E" "EGm E" "EGm E" ja-kanji)
("J" "E" "Am E" "Am E" ja-kanji)
("K" "{[" "W[jVm E" "W[jVm E" ja-kanji)
("L" "eC" "gm E" "gm E" ja-kanji)
("M" "L" "Lm EJKE" "Lm EJKE" ja-kanji)
("N" "`[" "W[jVm EV" "W[jVm EV" ja-kanji)
("O" "^C" "^Cm EX" "^Cm EX" ja-kanji)
("P" "L[" "L[m EX" "L[m EX" ja-kanji)
("Q" "J" "Jm EY" "Jm EY" ja-kanji)
("R" "L" "Lm E\" "Lm E\" ja-kanji)
("S" "oC" "oCm E^" "oCm E^" ja-kanji)
("T" "Ec" "[Ecm Ec" "[Ecm Ec" ja-kanji)
("U" "Ec" "Cm Ec" "Cm Ec" ja-kanji)
("V" "}" "}m EiM" "}m EiM" ja-kanji)
("W" "{" "{m Eo" "{m Eo" ja-kanji)
("X" "L[" "L[m E}" "L[m E}" ja-kanji)
("Y" "E" "zm E" "zm E" ja-kanji)
("Z" "J" "Jm E" "Jm E" ja-kanji)
("[" "W" "Wm E[" "Wm E[" ja-kanji)
("\" "\" "\m ET" "\m ET" ja-kanji)
("]" "E" "CEm E" "CEm E" ja-kanji)
("^" "E" "nRum E" "nRum E" ja-kanji)
("_" "E" "Nm E" "Nm E" ja-kanji)
("`" "G" "jm GS}" "jm GS}" ja-kanji)
("a" "W" "Wm GT" "Wm GT" ja-kanji)
("b" "GC" "qGCUm GC" "qGCUm GC" ja-kanji)
("c" "GC" "Cgim GC" "Cgim GC" ja-kanji)
("d" "GC" "~hSm GC" "~hSm GC" ja-kanji)
("e" "GC" "JQm GC" "JQm GC" ja-kanji)
("f" "GC" "Ecm GC" "Ecm GC" ja-kanji)
("g" "GC" "qNm GC" "qNm GC" ja-kanji)
("h" "GC" "TJGm GC" "TJGm GC" ja-kanji)
("i" "GC" "iKCm GC" "iKCm GC" ja-kanji)
("j" "GC" "IOm GC" "IOm GC" ja-kanji)
("k" "GC" "m GC" "m GC" ja-kanji)
("l" "GC" "XCV[m GC" "XCV[m GC" ja-kanji)
("m" "GC" "~`m GC" "~`m GC" ja-kanji)
("n" "GC" "mMm GC" "mMm GC" ja-kanji)
("o" "GC" "mMm ]NW" "mMm ]NW" ja-kanji)
("p" "GC" "GCSm GC" "GCSm GC" ja-kanji)
("q" "GC" "}m GC" "}m GC" ja-kanji)
("r" "GC" "m GC" "m GC" ja-kanji)
("s" "GC" "XhCm GC" "XhCm GC" ja-kanji)
("t" "GL" "GL^Cm GL" "GL^Cm GL" ja-kanji)
("u" "GL" "GLr[m GL" "GLr[m GL" ja-kanji)
("v" "GL" "GLm GL" "GLm GL" ja-kanji)
("w" "GL" "eCVom GL" "eCVom GL" ja-kanji)
("x" "Gc" "Rum Gc" "Rum Gc" ja-kanji)
("y" "Gc" "GbPm Gc" "GbPm Gc" ja-kanji)
("z" "Gc" "RXm Gc" "RXm Gc" ja-kanji)
("{" "Gc" "Gcm Gc" "Gcm Gc" ja-kanji)
("|" "J" "LmGm L" "LmGm L" ja-kanji)
("}" "G" "Cg[m G" "Cg[m G" ja-kanji)
("~" "G" "}m G" "}m G" ja-kanji)
("" "G" "\mm G" "\mm G" ja-kanji)
("" "G" "ZLm G" "ZLm G" ja-kanji)
("" "G" "A}~m G" "A}~m G" ja-kanji)
("" "G" "E^Qm G" "E^Qm G" ja-kanji)
("" "G" "mrm G" "mrm G" ja-kanji)
("" "G" "Em G" "Em G" ja-kanji)
("" "G" "IIEm G" "IIEm G" ja-kanji)
("" "G" "I[Gm G" "I[Gm G" ja-kanji)
("" "G" "\[m G" "\[m G" ja-kanji)
("" "G" "GQLm G" "GQLm G" ja-kanji)
("" "G" "zm Im G" "zm Im G" ja-kanji)
("" "G" "JG_}m G" "JG_}m G" ja-kanji)
("" "G" "Pm G" "Pm G" ja-kanji)
("" "G" "com G" "com G" ja-kanji)
("" "G" "Tm G" "Tm G" ja-kanji)
("" "G" "[cGm G" "[cGm G" ja-kanji)
("" "G" "cm G" "cm G" ja-kanji)
("" "G" "R[WGm G" "R[WGm G" ja-kanji)
("" "G" "NTm \mm G" "NTm \mm G" ja-kanji)
("" "G" "gICm G" "gICm G" ja-kanji)
("" "G" "i}m G" "i}m G" ja-kanji)
("" "G" "IVhm G" "IVhm G" ja-kanji)
("" "G" "VIm G" "VIm G" ja-kanji)
("" "I" "IPm I" "IPm I" ja-kanji)
("" "I" "L^iCm I" "L^iCm I" ja-kanji)
("" "IC" "ICCm IC" "ICCm IC" ja-kanji)
("" "I[" "I[gcm I[" "I[gcm I[" ja-kanji)
("" "I[" "`[I[m I[" "`[I[m I[" ja-kanji)
("" "I[" "INm I[" "INm I[" ja-kanji)
("" "I[" "Nm I[" "Nm I[" ja-kanji)
("" "I[" "R^Gm I[" "R^Gm I[" ja-kanji)
("" "I[" "IXm I[" "IXm I[" ja-kanji)
("" "I[" "TJm I[" "TJm I[" ja-kanji)
("" "I[" "Rm I[" "Rm I[" ja-kanji)
("" "I[" "ZCI[m I[" "ZCI[m I[" ja-kanji)
("" "I[" "iOm I[" "iOm I[" ja-kanji)
("" "I[" "I[T}m I[" "I[T}m I[" ja-kanji)
("" "I[" "ILim I[" "ILim I[" ja-kanji)
("" "I[" "tX}m I[" "tX}m I[" ja-kanji)
("" "I[" "EOCXm I[" "EOCXm I[" ja-kanji)
("" "I[" "Jm I[" "Jm I[" ja-kanji)
("" "I[" "LCm I[" "LCm I[" ja-kanji)
("" "IJ" "IJ}m IJ" "IJ}m IJ" ja-kanji)
("" "`[" "`[m IL" "`[m IL" ja-kanji)
("" "e" "NTm IM" "NTm IM" ja-kanji)
("" "IN" "JX[Wm IN" "JX[Wm IN" ja-kanji)
("" "" "m IN" "m IN" ja-kanji)
("" "IN" "LINm IN" "LINm IN" ja-kanji)
("" "IN" "INr[m IN" "INr[m IN" ja-kanji)
("" "[" "eIPm IP" "eIPm IP" ja-kanji)
("" "{" "IXXm IX" "IXXm IX" ja-kanji)
("" "Ic" "R[Icm Ic" "R[Icm Ic" ja-kanji)
("" "G" "rim I" "rim I" ja-kanji)
("" "IV" "Vm IX" "Vm IX" ja-kanji)
("" "I" "IWm I" "IWm I" ja-kanji)
("" "I" "A^^Jm I" "A^^Jm I" ja-kanji)
("" "I" "I_Jm I" "I_Jm I" ja-kanji)
("" "I" "Igm I" "Igm I" ja-kanji)
("" "J" "V^m J" "V^m J" ja-kanji)
("" "J" "oPm J" "oPm J" ja-kanji)
("" "J" "Jm J" "Jm J" ja-kanji)
("" "i" "ijm J" "ijm J" ja-kanji)
("" "J" "IgMoiVm J" "IgMoiVm J" ja-kanji)
("" "J" "A^Cm J" "A^Cm J" ja-kanji)
("" "J" "JWm J" "JWm J" ja-kanji)
("" "J" "NGm J" "NGm J" ja-kanji)
("" "J" "LJm J" "LJm J" ja-kanji)
("" "J" "JtNm J" "JtNm J" ja-kanji)
("" "J" "icm J" "icm J" ja-kanji)
("" "J" "m J" "m J" ja-kanji)
("" "J" "CGm J" "CGm J" ja-kanji)
("" "J" "XNiCm J" "XNiCm J" ja-kanji)
("" "J" "Jm J" "Jm J" ja-kanji)
("" "J" "q}m J" "q}m J" ja-kanji)
("" "J" "n^Xm J" "n^Xm J" ja-kanji)
("" "J" "JPm J" "JPm J" ja-kanji)
("" "J" "E^m J" "E^m J" ja-kanji)
("" "J" "Jm J" "Jm J" ja-kanji)
("" "J" "qm J" "qm J" ja-kanji)
("" "J" "Vm Em J" "Vm Em J" ja-kanji)
("" "J" "UCm J" "UCm J" ja-kanji)
("" "J" "RNcm J" "RNcm J" ja-kanji)
("" "J" "JZOm J" "JZOm J" ja-kanji)
("" "J" "JW[KLm J" "JW[KLm J" ja-kanji)
("" "J" "nim J" "nim J" ja-kanji)
("" "J" "SCm J" "SCm J" ja-kanji)
("" "J" "iXm J" "iXm J" ja-kanji)
("" "J" "jcm J" "jcm J" ja-kanji)
("" "J" "`[Jm J" "`[Jm J" ja-kanji)
("" "J" "JVm J" "JVm J" ja-kanji)
("" "J" "Grm J" "Grm J" ja-kanji)
("" "J" "J_Cm J" "J_Cm J" ja-kanji)
("" "J" "J}VCm J" "J}VCm J" ja-kanji)
("" "J" "Jcm J" "Jcm J" ja-kanji)
("" "J" "VJm J" "VJm J" ja-kanji)
("" "J" "XMm J" "XMm J" ja-kanji)
("" "J" "JX~m J" "JX~m J" ja-kanji)
("" "J" "Vm J" "Vm J" ja-kanji)
("" "K" "jJm K" "jJm K" ja-kanji)
("" "K" "KrTm K" "KrTm K" ja-kanji)
("" "K" "m K" "m K" ja-kanji)
("" "K" "Lom K" "Lom K" ja-kanji)
("" "K" "Km K" "Km K" ja-kanji)
("" "K" "tXm K" "tXm K" ja-kanji)
("" "K" "m K" "m K" ja-kanji)
("" "K" "Vm K" "Vm K" ja-kanji)
("" "K" "VNKm K" "VNKm K" ja-kanji)
("" "K" "~rJm K" "~rJm K" ja-kanji)
("" "K" "KLm K" "KLm K" ja-kanji)
("" "K" "JSm K" "JSm K" ja-kanji)
("" "JC" "JC]Gm JC" "JC]Gm JC" ja-kanji)
("" "JC" "AEm JC" "AEm JC" ja-kanji)
("" "JC" "gNm JC" "gNm JC" ja-kanji)
("" "JC" "}m JC" "}m JC" ja-kanji)
("" "JC" "J^}m JC" "J^}m JC" ja-kanji)
("" "JC" "RXm JC" "RXm JC" ja-kanji)
("" "JC" "Om JC" "Om JC" ja-kanji)
("" "JC" "RRCm JC" "RRCm JC" ja-kanji)
("" "JC" "AVCm JC" "AVCm JC" ja-kanji)
("" "JC" "NCm JC" "NCm JC" ja-kanji)
("" "JC" "JCJCm JC" "JCJCm JC" ja-kanji)
("" "JC" "icJVCm JC" "icJVCm JC" ja-kanji)
("" "JC" "C}Vm JC" "C}Vm JC" ja-kanji)
("" "JC" "[JCm JC" "[JCm JC" ja-kanji)
("" "JC" "A^m JC" "A^m JC" ja-kanji)
("@" "JC" "TLKPm JC" "TLKPm JC" ja-kanji)
("A" "JC" "cSm JC" "cSm JC" ja-kanji)
("B" "JC" "LJCm JC" "LJCm JC" ja-kanji)
("C" "JC" "E~m JC" "E~m JC" ja-kanji)
("D" "JC" "nCm JC" "nCm JC" ja-kanji)
("E" "JC" "ZJCm JC" "ZJCm JC" ja-kanji)
("F" "JC" "~im JC" "~im JC" ja-kanji)
("G" "JC" "Gm JC" "Gm JC" ja-kanji)
("H" "JC" "AN^m JC" "AN^m JC" ja-kanji)
("I" "JC" "h[ucm Jj" "h[ucm Jj" ja-kanji)
("J" "JC" "qNm JC" "qNm JC" ja-kanji)
("K" "JC" "JC_m JC" "JC_m JC" ja-kanji)
("L" "JC" "JCKm JC" "JCKm JC" ja-kanji)
("M" "KC" "KCZm KC" "KCZm KC" ja-kanji)
("N" "KC" "_KCm KC" "_KCm KC" ja-kanji)
("O" "KC" "\gm KC" "\gm KC" ja-kanji)
("P" "KC" "ZLm KC" "ZLm KC" ja-kanji)
("Q" "KC" "KChNm KC" "KChNm KC" ja-kanji)
("R" "KC" "KPm KC" "KPm KC" ja-kanji)
("S" "KC" "tKCm KC" "tKCm KC" ja-kanji)
("T" "KC" "IIlm KC" "IIlm KC" ja-kanji)
("U" "KC" "nem KC" "nem KC" ja-kanji)
("V" "KC" "KCVm KC" "KCVm KC" ja-kanji)
("W" "KC" "P_Vm KC" "P_Vm KC" ja-kanji)
("X" "KC" "}`m KC" "}`m KC" ja-kanji)
("Y" "KC" "g[KCm KC" "g[KCm KC" ja-kanji)
("Z" "KC" "Cm KC" "Cm KC" ja-kanji)
("[" "KC" "KCRcm KC" "KCRcm KC" ja-kanji)
("\" "JC" "iKTm JC" "iKTm JC" ja-kanji)
("]" "PC" "PCm JI" "PCm JI" ja-kanji)
("^" "A" "h[ucm JG" "h[ucm JG" ja-kanji)
("_" "G" "JLlm JL" "JLlm JL" ja-kanji)
("`" "V" "N_mm JL" "N_mm JL" ja-kanji)
("a" "C" "JCm JL" "JCm JL" ja-kanji)
("b" "R[" "JMom JM" "JMom JM" ja-kanji)
("c" "JN" "JNCcm L[W" "JNCcm L[W" ja-kanji)
("d" "JN" "CJNXm JN" "CJNXm JN" ja-kanji)
("e" "JN" "ImImm JN" "Im Imm JN" ja-kanji)
("f" "JN" "Nm JN" "Nm JN" ja-kanji)
("g" "JN" "JN_Cm JN" "JN_Cm JN" ja-kanji)
("h" "JN" "JNnm JN" "JNnm JN" ja-kanji)
("i" "JN" "JNQm JN" "JNQm JN" ja-kanji)
("j" "JN" "QVJNm JN" "QVJNm JN" ja-kanji)
("k" "JN" "Jm JN" "Jm JN" ja-kanji)
("l" "JN" "Gm JN" "Gm JN" ja-kanji)
("m" "JN" "^VJm JN" "^VJm JN" ja-kanji)
("n" "JN" "V[JNm JN" "V[JNm JN" ja-kanji)
("o" "JN" "I{Gm JN" "I{Gm JN" ja-kanji)
("p" "JN" "cmm JN" "cmm JN" ja-kanji)
("q" "JN" "IhXm JN" "IhXm JN" ja-kanji)
("r" "JN" "Nxm JN" "Nxm JN" ja-kanji)
("s" "JN" "JNm JN" "JNm JN" ja-kanji)
("t" "JN" "JN[m JN" "JN[m JN" ja-kanji)
("u" "JN" "w_em JN" "w_em JN" ja-kanji)
("v" "JN" "Jm JN" "Jm JN" ja-kanji)
("w" "KN" "}ium KN" "}ium KN" ja-kanji)
("x" "KN" "^Pm KN" "^Pm KN" ja-kanji)
("y" "KN" "^m VCm KN" "^m VCm KN" ja-kanji)
("z" "KN" "q^Cm KN" "q^Cm KN" ja-kanji)
("{" "KN" "ASm KN" "ASm KN" ja-kanji)
("|" "J" "JPSGm JP" "JPSGm JP" ja-kanji)
("}" "[" "[m JT" "[m JT" ja-kanji)
("~" "P" "JVm Lm JV" "JVm Lm JV" ja-kanji)
("" "L[" "JVnm JV" "JVnm JV" ja-kanji)
("" "r" "JW{[m JW" "JW{[m JW" ja-kanji)
("" "V[" "TJim JWJ" "TJim JWJ" ja-kanji)
("" "J^" "jCK^m J^" "jCK^m J^" ja-kanji)
("" "Jc" "m Jc" "m Jc" ja-kanji)
("" "Jc" "JbTCm Jc" "JbTCm Jc" ja-kanji)
("" "Jc" "JbR[m Jc" "JbR[m Jc" ja-kanji)
("" "Jc" "\[Jcm Jc" "\[Jcm Jc" ja-kanji)
("" "Jc" "Jch[m Jc" "Jch[m Jc" ja-kanji)
("" "Jc" "JNm Jc" "JNm Jc" ja-kanji)
("" "Jc" "iJm Jc" "iJm Jc" ja-kanji)
("" "Jc" "Jbg[m Jc" "Jbg[m Jc" ja-kanji)
("" "Jc" "JbVNm Jc" "JbVNm Jc" ja-kanji)
("" "Jc" "JJcm Jc" "JJcm Jc" ja-kanji)
("" "\" "Jc}^m Jc" "Jc}^m Jc" ja-kanji)
("" "P" "TJim JcI" "TJim JcI" ja-kanji)
("" "L[" "lKCK JiE" "lKCK JiE" ja-kanji)
("" "J" "nim uVm Jo" "nim uVm Jo" ja-kanji)
("" "J" "Lm Jo" "Lm Jo" ja-kanji)
("" "z[" "Jm Jo" "Jm Jo" ja-kanji)
("" "Ju" "JuVLm Ju" "JuVLm Ju" ja-kanji)
("" "g[" "Cm Jug" "Cm Jug" ja-kanji)
("" "\[" "CVm J}h" "CVm J}h" ja-kanji)
("" "z" "NTm K}" "NTm K}" ja-kanji)
("" "t" "ixJ}m J}" "ixJ}m J}" ja-kanji)
("" "" "J}Nrm J}" "J}Nrm J}" ja-kanji)
("" "R[" "nf Jm J" "nf Jm J" ja-kanji)
("" "I[" "gm J" "gm J" ja-kanji)
("" "nN" "JVm ]NW" "JVm ]NW" ja-kanji)
("" "`" "{[m J" "{[m J" ja-kanji)
("" "P" "Pm J" "Pm J" ja-kanji)
("" "VN" "CKm J" "CKm J" ja-kanji)
("" "KC" "KCm J" "KCm J" ja-kanji)
("" "KC" "Jm RNW" "Jm RNW" ja-kanji)
("" "K" "Km J" "Km J" ja-kanji)
("" "J" "JNm J" "JNm J" ja-kanji)
("" "J" "JKNm J" "JKNm J" ja-kanji)
("" "J" "Jm J" "Jm J" ja-kanji)
("" "J" "TCm J" "TCm J" ja-kanji)
("" "J" "jbJVm J" "jbJVm J" ja-kanji)
("" "J" "Jxm J" "Jxm J" ja-kanji)
("" "J" "XXm J" "XXm J" ja-kanji)
("" "J" "}Nm J" "}Nm J" ja-kanji)
("" "J" "um J" "um J" ja-kanji)
("" "J" "^Gm J" "^Gm J" ja-kanji)
("" "J" "JV}VCm J" "JV}VCm J" ja-kanji)
("" "J" "J[m J" "J[m J" ja-kanji)
("" "J" "Jm J" "Jm J" ja-kanji)
("" "J" "J_Cm J" "J_Cm J" ja-kanji)
("" "J" "zXm J" "zXm J" ja-kanji)
("" "J" "~Lm J" "~Lm J" ja-kanji)
("" "J" "YEm J" "YEm J" ja-kanji)
("" "J" "JWZCm J" "JWZCm J" ja-kanji)
("" "J" "im J" "im J" ja-kanji)
("" "J" "CJm J" "CJm J" ja-kanji)
("" "J" "JGm J" "JGm J" ja-kanji)
("" "J" "[Jm J" "[Jm J" ja-kanji)
("" "J" "~Jm J" "~Jm J" ja-kanji)
("" "J" "Jm J" "Jm J" ja-kanji)
("" "J" "qcMm J" "qcMm J" ja-kanji)
("" "J" "bJm J" "bJm J" ja-kanji)
("" "J" "Rum J" "Rum J" ja-kanji)
("" "J" "AZm J" "AZm J" ja-kanji)
("" "J" "JWm J" "JWm J" ja-kanji)
("" "J" "^jm J" "^jm J" ja-kanji)
("" "J" "\\Om J" "\\Om J" ja-kanji)
("" "J" "JL[m J" "JL[m J" ja-kanji)
("" "J" "A}Cm J" "A}Cm J" ja-kanji)
("" "J" "JSNm J" "JSNm J" ja-kanji)
("" "J" "JStm J" "JStm J" ja-kanji)
("" "J" "TIm J" "TIm J" ja-kanji)
("" "J" "N_m J" "N_m J" ja-kanji)
("" "J" "J^m J" "J^m J" ja-kanji)
("" "J" "Cm J" "Cm J" ja-kanji)
("" "J" "Jdm J" "Jdm J" ja-kanji)
("" "J" "eK~m J" "eK~m J" ja-kanji)
("" "J" "Lm J" "Lm J" ja-kanji)
("" "J" "OJm J" "OJm J" ja-kanji)
("" "J" "CkVm J" "CkVm J" ja-kanji)
("" "J" "~m J" "~m J" ja-kanji)
("" "J" "CTm J" "CTm J" ja-kanji)
("" "J" "ckNm J" "ckNm J" ja-kanji)
("" "J" "JLm J" "JLm J" ja-kanji)
("" "J" "Jxcm J" "Jxcm J" ja-kanji)
("" "J" "AC_m J" "AC_m J" ja-kanji)
("" "J" "q}m J" "q}m J" ja-kanji)
("" "J" "ZLm J" "ZLm J" ja-kanji)
("" "J" "I`Cm J" "I`Cm J" ja-kanji)
("" "J" "JRNm J" "JRNm J" ja-kanji)
("" "J" "J^m J" "J^m J" ja-kanji)
("" "J" "J^m ]NW" "J^m ]NW" ja-kanji)
("" "K" "}m K" "}m K" ja-kanji)
("" "K" "tNm K" "tNm K" ja-kanji)
("" "K" "LVm K" "LVm K" ja-kanji)
("" "K" "KNcm K" "KNcm K" ja-kanji)
("" "K" "KOm K" "KOm K" ja-kanji)
("" "K" "r[Lm K" "r[Lm K" ja-kanji)
("" "K" "}iRm K" "}iRm K" ja-kanji)
("" "K" "Cm K" "Cm K" ja-kanji)
("" "K" "eA\um K" "eA\um K" ja-kanji)
("" "K" "jZm K" "jZm K" ja-kanji)
("" "K" "gm K" "gm K" ja-kanji)
("" "K" "J^Nim K" "J^Nim K" ja-kanji)
("" "K" "JIm K" "JIm K" ja-kanji)
("" "K" "lKEm K" "lKEm K" ja-kanji)
("" "L" "N_em L" "N_em L" ja-kanji)
("" "L" "JuLm L" "JuLm L" ja-kanji)
("" "L" "AuiCm L" "AuiCm L" ja-kanji)
("" "L" "Rum L" "Rum L" ja-kanji)
("" "L" "Ecm L" "Ecm L" ja-kanji)
("" "L" "gm L" "gm L" ja-kanji)
("" "L" "L~[m L" "L~[m L" ja-kanji)
("" "L" "EVCm L" "EVCm L" ja-kanji)
("" "L" "m L" "m L" ja-kanji)
("" "L" "uLem L" "uLem L" ja-kanji)
("" "L" "L{[m L" "L{[m L" ja-kanji)
("" "L" "CNm L" "CNm L" ja-kanji)
("" "L" "Cm L" "Cm L" ja-kanji)
("" "L" "tEm L" "tEm L" ja-kanji)
("" "L" "cNGm L" "cNGm L" ja-kanji)
("" "L" "n^m L" "n^m L" ja-kanji)
("" "L" "Xfjm L" "Xfjm L" ja-kanji)
("" "L" "L^Cm L" "L^Cm L" ja-kanji)
("" "L" "V[Mm L" "V[Mm L" ja-kanji)
("" "L" "Xem L" "Xem L" ja-kanji)
("@" "L" "LJCm L" "LJCm L" ja-kanji)
("A" "L" "JGm L" "JGm L" ja-kanji)
("B" "L" "L[g Xm L" "L[g Xm L" ja-kanji)
("C" "L" "N[Lm L" "N[Lm L" ja-kanji)
("D" "L" "LJVm L" "LJVm L" ja-kanji)
("E" "L" "LL `z[m L" "LL `z[m L" ja-kanji)
("F" "L" "Cmm L" "Cmm L" ja-kanji)
("G" "L" "LZcm L" "LZcm L" ja-kanji)
("H" "L" "}m L" "}m L" ja-kanji)
("I" "L" "jWbZCLm L" "jWbZCLm L" ja-kanji)
("J" "L" "RLfm L" "RLfm L" ja-kanji)
("K" "L" "L\Nm L" "L\Nm L" ja-kanji)
("L" "L" "VXm L" "VXm L" ja-kanji)
("M" "L" "g[gCm L" "g[gCm L" ja-kanji)
("N" "L" "ILm L" "ILm L" ja-kanji)
("O" "L" "Lh[m L" "Lh[m L" ja-kanji)
("P" "L" "JKNm L" "JKNm L" ja-kanji)
("Q" "L" "EGm L" "EGm L" ja-kanji)
("R" "L" "LoZm L" "LoZm L" ja-kanji)
("S" "L" "Ijm L" "Ijm L" ja-kanji)
("T" "L" "Jm L" "Jm L" ja-kanji)
("U" "M" "Ccm M" "Ccm M" ja-kanji)
("V" "M" "MVLm M" "MVLm M" ja-kanji)
("W" "M" "QCMm M" "QCMm M" ja-kanji)
("X" "M" "xMm M" "xMm M" ja-kanji)
("Y" "M" "^m M" "^m M" ja-kanji)
("Z" "M" "Um M" "Um M" ja-kanji)
("[" "M" "MRum M" "MRum M" ja-kanji)
("\" "M" "AUNm M" "AUNm M" ja-kanji)
("]" "M" "MZCm M" "MZCm M" ja-kanji)
("^" "M" "E^KEm M" "E^KEm M" ja-kanji)
("_" "M" "MIV[Wm M" "MIV[Wm M" ja-kanji)
("`" "M" "Mm M" "Mm M" ja-kanji)
("a" "M" "Vm A" "Vm A" ja-kanji)
("b" "M" "V~m M" "V~m M" ja-kanji)
("c" "M" "MJCm M" "MJCm M" ja-kanji)
("d" "LN" "XNEm LN" "XNEm LN" ja-kanji)
("e" "LN" "nim LN" "nim LN" ja-kanji)
("f" "LN" "P}m LN" "P}m LN" ja-kanji)
("g" "L`" "_CL`m L`" "_CL`m L`" ja-kanji)
("h" "Lc" "hm Lc" "hm Lc" ja-kanji)
("i" "Lc" "LcGm Lc" "LcGm Lc" ja-kanji)
("j" "Lc" "LL[m Lc" "LL[m Lc" ja-kanji)
("k" "Lc" "^`oim Lc" "^`oim Lc" ja-kanji)
("l" "Lc" "cm Lc" "cm Lc" ja-kanji)
("m" "`" "Lk^m `" "Lk^m `" ja-kanji)
("n" "V" "LldJm Ll" "LldJm Ll" ja-kanji)
("o" "V" "g[Lrm Lr" "g[Lrm Lr" ja-kanji)
("p" "LN" "{[LNm LN" "{[LNm LN" ja-kanji)
("q" "LN" "ILNT}m LN" "ILNT}m LN" ja-kanji)
("r" "LN" "AVm LN" "AVm LN" ja-kanji)
("s" "MN" "VC^Qm MN" "VC^Qm MN" ja-kanji)
("t" "MN" "MNem MN" "MNem MN" ja-kanji)
("u" "L[" "IJm L[" "IJm L[" ja-kanji)
("v" "L[" "qTVCm L[" "qTVCm L[" ja-kanji)
("w" "L[" "A_m L[" "A_m L[" ja-kanji)
("x" "L[" "Xm L[" "Xm L[" ja-kanji)
("y" "L[" "Ium L[" "Ium L[" ja-kanji)
("z" "L[" "X[m L[" "X[m L[" ja-kanji)
("{" "L[" "~m L[" "~m L[" ja-kanji)
("|" "L[" "~m L[" "~m L[" ja-kanji)
("}" "L[" "C\Om L[" "C\Om L[" ja-kanji)
("~" "L[" "XNEm L[" "XNEm L[" ja-kanji)
("" "L[" "N`m L[" "N`m L[" ja-kanji)
("" "L[" "gm L[" "gm L[" ja-kanji)
("" "L[" "Nm L[" "Nm L[" ja-kanji)
("" "L[" "iNm L[" "iNm L[" ja-kanji)
("" "L[" "IL[m L[" "IL[m L[" ja-kanji)
("" "L[" "^}m L[" "^}m L[" ja-kanji)
("" "L[" "Lm L[" "Lm L[" ja-kanji)
("" "L[" "L}m L[" "L}m L[" ja-kanji)
("" "L[" "ICm L[" "ICm L[" ja-kanji)
("" "L[" "O[hm L[" "O[hm L[" ja-kanji)
("" "L[" "^_Xm L[" "^_Xm L[" ja-kanji)
("" "L[" "L[[m L[" "L[[m L[" ja-kanji)
("" "L[" "tCm L[" "tCm L[" ja-kanji)
("" "M[" "EVm M[" "EVm M[" ja-kanji)
("" "L" "Tm L" "Tm L" ja-kanji)
("" "L" "Cm L" "Cm L" ja-kanji)
("" "L" "LWm L" "LWm L" ja-kanji)
("" "L" "Rom L" "Rom L" ja-kanji)
("" "L" "RLm L" "RLm L" ja-kanji)
("" "L" "AQm L" "AQm L" ja-kanji)
("" "L" "ALm L" "ALm L" ja-kanji)
("" "L" "LGCm L" "LGCm L" ja-kanji)
("" "L" "Xm L" "Xm L" ja-kanji)
("" "L" "Lm L" "Lm L" ja-kanji)
("" "L" "mRMm L" "mRMm L" ja-kanji)
("" "M" "MR[m M" "MR[m M" ja-kanji)
("" "M" "ghm M" "ghm M" ja-kanji)
("" "M" "TJim M" "TJim M" ja-kanji)
("" "L[" "gIm L[" "gIm L[" ja-kanji)
("" "L[" "L[lm L[" "L[lm L[" ja-kanji)
("" "L[" "L[gm L[" "L[gm L[" ja-kanji)
("" "L[" "\iGm L[" "\iGm L[" ja-kanji)
("" "L[" "jL[m L[" "jL[m L[" ja-kanji)
("" "L[" "JL[m L[" "JL[m L[" ja-kanji)
("" "L[" "L[{[m L[m L[W" "L[{[m L[m L[W" ja-kanji)
("" "L[" "L\Em L[" "L\Em L[" ja-kanji)
("" "L[" "gm L[" "gm L[" ja-kanji)
("" "L[" "L[{[m L[" "L[{[m L[" ja-kanji)
("" "L[" "L[Nm L[" "L[Nm L[" ja-kanji)
("" "L[" "^_Xm L[" "^_Xm L[" ja-kanji)
("" "L[" "NL[m L[" "NL[m L[" ja-kanji)
("" "L[" "TPum L[" "TPum L[" ja-kanji)
("" "L[" "L[{Nm L[" "L[{Nm L[" ja-kanji)
("" "L[" "TJCm L[" "TJCm L[" ja-kanji)
("" "L[" "nU}m L[" "nU}m L[" ja-kanji)
("" "L[" "cCm L[" "cCm L[" ja-kanji)
("" "L[" "WL[Wcm L[" "WL[Wcm L[" ja-kanji)
("" "L[" "IrGm L[" "IrGm L[" ja-kanji)
("" "L[" "I\m L[" "I\m L[" ja-kanji)
("" "L[" "EEVCm L[" "EEVCm L[" ja-kanji)
("" "L[" "nTm L[" "nTm L[" ja-kanji)
("" "L[" "IVGm L[" "IVGm L[" ja-kanji)
("" "L[" "nVm L[" "nVm L[" ja-kanji)
("" "L[" "W[L[m L[" "W[L[m L[" ja-kanji)
("" "L[" "NEm L[" "NEm L[" ja-kanji)
("" "L[" "Z}Cm L[" "Z}Cm L[" ja-kanji)
("" "L[" "^m L[" "^m L[" ja-kanji)
("" "L[" "lm L[" "lm L[" ja-kanji)
("" "L[" "IrJXm L[" "IrJXm L[" ja-kanji)
("" "L[" "IRm L[" "IRm L[" ja-kanji)
("" "L[" "\om L[" "\om L[" ja-kanji)
("" "L[" "RL[m L[" "RL[m L[" ja-kanji)
("" "L[" "JK~m L[" "JK~m L[" ja-kanji)
("" "L[" "qrNm L[" "qrNm L[" ja-kanji)
("" "L[" "L[Gm L[" "L[Gm L[" ja-kanji)
("" "L[" "IhNm L[" "IhNm L[" ja-kanji)
("" "M[" "AIOm M[" "AIOm M[" ja-kanji)
("" "M[" "Rm M[" "Rm M[" ja-kanji)
("" "M[" "M[Vm M[" "M[Vm M[" ja-kanji)
("" "M[" "AJcLm M[" "AJcLm M[" ja-kanji)
("" "M[" "Um M[" "Um M[" ja-kanji)
("" "LN" "[rLNm LN" "[rLNm LN" ja-kanji)
("" "LN" "}Km LN" "}Km LN" ja-kanji)
("" "LN" "Lm LN" "Lm LN" ja-kanji)
("" "MN" "^}m MN" "^}m MN" ja-kanji)
("" "g[" "VNucm L" "VNucm L" ja-kanji)
("" "L[g" "iKTm L" "iKTm L" ja-kanji)
("" "L" "YJm L" "YJm L" ja-kanji)
("" "L" "cgm L" "cgm L" ja-kanji)
("" "L" "wCLm L" "wCLm L" ja-kanji)
("" "L" "L`Nm L" "L`Nm L" ja-kanji)
("" "L" "jVLm L" "jVLm L" ja-kanji)
("" "L" "ITm L" "ITm L" ja-kanji)
("" "L" "Rum L" "Rum L" ja-kanji)
("" "L" "LeCm L" "LeCm L" ja-kanji)
("" "L" "Rgm L" "Rgm L" ja-kanji)
("" "L" "LVm L" "LVm L" ja-kanji)
("" "L" "gm L" "gm L" ja-kanji)
("" "L" "XWm L" "XWm L" ja-kanji)
("" "L" "L`[m L" "L`[m L" ja-kanji)
("" "L" "Zm L" "Zm L" ja-kanji)
("" "L" "oCLm L" "oCLm L" ja-kanji)
("" "L" "GAVm L" "GAVm L" ja-kanji)
("" "L" "Gm L" "Gm L" ja-kanji)
("" "L" "ccVfm L" "ccVfm L" ja-kanji)
("" "L" "`JCm L" "`JCm L" ja-kanji)
("" "L" "Jlm L" "Jlm L" ja-kanji)
("" "M" "M~m M" "M~m M" ja-kanji)
("" "M" "MCm M" "MCm M" ja-kanji)
("" "L[" "JX[Wm N" "JX[Wm N" ja-kanji)
("" "N" "Num N" "Num N" ja-kanji)
("" "N" "nCNm N" "nCNm N" ja-kanji)
("" "N" "NJNZCm N" "NJNZCm N" ja-kanji)
("" "N" "eOm N" "eOm N" ja-kanji)
("" "N" "NCVm N" "NCVm N" ja-kanji)
("" "N" "NPCm N" "NPCm N" ja-kanji)
("" "N" "NVm N" "NVm N" ja-kanji)
("" "N" "J_m N" "J_m N" ja-kanji)
("" "N" "JPm N" "JPm N" ja-kanji)
("" "N" "E}j IJm N" "E}j IJm N" ja-kanji)
("" "N" "R}m N" "R}m N" ja-kanji)
("" "O" "h[Om O" "h[Om O" ja-kanji)
("" "O" "IJm O" "IJm O" ja-kanji)
("" "O" "OrWm O" "OrWm O" ja-kanji)
("" "N[" "N`wm N[" "N`wm N[" ja-kanji)
("" "N[" "\m N[" "\m N[" ja-kanji)
("" "O[" "O[[m O[" "O[[m O[" ja-kanji)
("" "O[" "O[m O[" "O[m O[" ja-kanji)
("" "O[" "\[O[m O[" "\[O[m O[" ja-kanji)
("" "O[" "X~m O[" "X~m O[" ja-kanji)
("" "Z" "NVUVm NV" "NVUVm NV" ja-kanji)
("" "Zc" "NVPYm NV" "NVPYm NV" ja-kanji)
("" "Z" "NVm NV" "NVm NV" ja-kanji)
("" "NY" "NYJSm NY" "NYJSm NY" ja-kanji)
("" "Nc" "NcWNm Nc" "NcWNm Nc" ja-kanji)
("@" "Nc" "zm Nc" "zm Nc" ja-kanji)
("A" "Nc" "h[Ncm Nc" "h[Ncm Nc" ja-kanji)
("B" "g[" "g[m Nc" "g[m Nc" ja-kanji)
("C" "J" "NcV^m Nc" "NcV^m Nc" ja-kanji)
("D" "q" "E}m Nc" "E}m Nc" ja-kanji)
("E" "N{" "N{`m N{" "N{`m N{" ja-kanji)
("F" "[" "h[ucm N}" "h[ucm N}" ja-kanji)
("G" "C" "N}hm N}" "N}hm N}" ja-kanji)
("H" "N" "NgRm N" "NgRm N" ja-kanji)
("I" "c" "N_mm N" "N_mm N" ja-kanji)
("J" "\[" "Nm \[" "Nm \[" ja-kanji)
("K" "\[" "Nm \[" "Nm \[" ja-kanji)
("L" "V[" "XL Nm N" "XL Nm N" ja-kanji)
("M" "N" "NV[m N" "NV[m N" ja-kanji)
("N" "N" "L~m N" "L~m N" ja-kanji)
("O" "N" "JIm N" "JIm N" ja-kanji)
("P" "N" "Nm N" "Nm N" ja-kanji)
("Q" "O" "m O" "m O" ja-kanji)
("R" "O" "O^Cm O" "O^Cm O" ja-kanji)
("S" "O" "RIm O" "RIm O" ja-kanji)
("T" "P" "nbPm P" "nbPm P" ja-kanji)
("U" "P" "PTm P" "PTm P" ja-kanji)
("V" "P" "LUm L" "LUm L" ja-kanji)
("W" "PC" "JJm PC" "JJm PC" ja-kanji)
("X" "PC" "J^Nm PC" "J^Nm PC" ja-kanji)
("Y" "PC" "PCocm PC" "PCocm PC" ja-kanji)
("Z" "PC" "Ajm PC" "Ajm PC" ja-kanji)
("[" "PC" "nCPCm PC" "nCPCm PC" ja-kanji)
("\" "PC" "c` t^cm PC" "c` t^cm PC" ja-kanji)
("]" "PC" "PCZLm PC" "PCZLm PC" ja-kanji)
("^" "PC" "IIK^m J^" "IIK^m J^" ja-kanji)
("_" "PC" "`Mm PC" "`Mm PC" ja-kanji)
("`" "PC" "J^`m PC" "J^`m PC" ja-kanji)
("a" "PC" "R~`m PC" "R~`m PC" ja-kanji)
("b" "PC" "Om PC" "Om PC" ja-kanji)
("c" "PC" "PCVNm PC" "PCVNm PC" ja-kanji)
("d" "PC" "PCKm PC" "PCKm PC" ja-kanji)
("e" "PC" "CRCm PC" "CRCm PC" ja-kanji)
("f" "PC" "JJQm PC" "JJQm PC" ja-kanji)
("g" "PC" "^YTGm PC" "^YTGm PC" ja-kanji)
("h" "PC" "E}Em PC" "E}Em PC" ja-kanji)
("i" "PC" "t[PCm PC" "t[PCm PC" ja-kanji)
("j" "PC" "Jcm PC" "Jcm PC" ja-kanji)
("k" "PC" "PCRNm PC" "PCRNm PC" ja-kanji)
("l" "PC" "A[m PC" "A[m PC" ja-kanji)
("m" "PC" "PCRm PC" "PCRm PC" ja-kanji)
("n" "PC" "JPCYm PC" "JPCYm PC" ja-kanji)
("o" "PC" "PCUCm PC" "PCUCm PC" ja-kanji)
("p" "PC" "cOm PC" "cOm PC" ja-kanji)
("q" "PC" "ciOm PC" "ciOm PC" ja-kanji)
("r" "PC" "PCVm PC" "PCVm PC" ja-kanji)
("s" "PC" "NLm PC" "NLm PC" ja-kanji)
("t" "PC" "Com PC" "Com PC" ja-kanji)
("u" "PC" "z^m PC" "z^m PC" ja-kanji)
("v" "PC" "nJm PC" "nJm PC" ja-kanji)
("w" "PC" "[fm PC" "[fm PC" ja-kanji)
("x" "PC" "PCTcm PC" "PCTcm PC" ja-kanji)
("y" "PC" "JCm PC" "JCm PC" ja-kanji)
("z" "PC" "Nrm PC" "Nrm PC" ja-kanji)
("{" "PC" "jgm PC" "jgm PC" ja-kanji)
("|" "QC" "QCWcm QC" "QCWcm QC" ja-kanji)
("}" "QC" "JGm QC" "JGm QC" ja-kanji)
("~" "QC" "NWm QC" "NWm QC" ja-kanji)
("" "QL" "GQLm QL" "GQLm QL" ja-kanji)
("" "QL" "zRm QL" "zRm QL" ja-kanji)
("" "QL" "Ecm QL" "Ecm QL" ja-kanji)
("" "QL" "nQVCm QL" "nQVCm QL" ja-kanji)
("" "QL" "XL}m QL" "XL}m QL" ja-kanji)
("" "R[" "R[m P^" "R[m P^" ja-kanji)
("" "Pc" "S[Pcm Pc" "S[Pcm Pc" ja-kanji)
("" "Pc" "JPm Pc" "JPm Pc" ja-kanji)
("" "Pc" "Lm Pc" "Lm Pc" ja-kanji)
("" "Pc" "CTMCm Pc" "CTMCm Pc" ja-kanji)
("" "Pc" "Aim Pc" "Aim Pc" ja-kanji)
("" "Pc" "Xum Pc" "Xum Pc" ja-kanji)
("" "Pc" "`m Pc" "`m Pc" ja-kanji)
("" "Pc" "Pcxcm Pc" "Pcxcm Pc" ja-kanji)
("" "Qc" "cLm Qc" "cLm Qc" ja-kanji)
("" "P" "WPm P" "WPm P" ja-kanji)
("" "P" "PNm P" "PNm P" ja-kanji)
("" "P" "Em P" "Em P" ja-kanji)
("" "P" "XRJm P" "XRJm P" ja-kanji)
("" "P" "Jlm P" "Jlm P" ja-kanji)
("" "P" "JuPm P" "JuPm P" ja-kanji)
("" "P" "cMm P" "cMm P" ja-kanji)
("" "P" "PJm P" "PJm P" ja-kanji)
("" "P" "^CLPm P" "^CLPm P" ja-kanji)
("" "P" "J^Cm P" "J^Cm P" ja-kanji)
("" "P" "LEm P" "LEm P" ja-kanji)
("" "P" "^em P" "^em P" ja-kanji)
("" "P" "P|[m P" "P|[m P" ja-kanji)
("" "P" "JPm P" "JPm P" ja-kanji)
("" "P" "RuVm P" "RuVm P" ja-kanji)
("" "P" "}Nm P" "}Nm P" ja-kanji)
("" "P" "PeCm P" "PeCm P" ja-kanji)
("" "P" "Pm P" "Pm P" ja-kanji)
("" "P" "PCm P" "PCm P" ja-kanji)
("" "P" "Ckm P" "Ckm P" ja-kanji)
("" "P" "PW[m P" "PW[m P" ja-kanji)
("" "P" "gOm P" "gOm P" ja-kanji)
("" "P" "XYm P" "XYm P" ja-kanji)
("" "P" "Lkm P" "Lkm P" ja-kanji)
("" "P" "gh[tPm P" "gh[tPm P" ja-kanji)
("" "P" "J^m P" "J^m P" ja-kanji)
("" "P" "~m P" "~m P" ja-kanji)
("" "P" "PLm P" "PLm P" ja-kanji)
("" "P" "JVRCm P" "JVRCm P" ja-kanji)
("" "P" "mLm P" "mLm P" ja-kanji)
("" "P" "cJEm P" "cJEm P" ja-kanji)
("" "P" "JMm P" "JMm P" ja-kanji)
("" "P" "PVCm P" "PVCm P" ja-kanji)
("" "P" "PrL[m P" "PrL[m P" ja-kanji)
("" "P" "PCPm P" "PCPm P" ja-kanji)
("" "P" "ZbPm P" "ZbPm P" ja-kanji)
("" "Q" "gm Q" "gm Q" ja-kanji)
("" "Q" "nm Q" "nm Q" ja-kanji)
("" "Q" "LrVCm Q" "LrVCm Q" ja-kanji)
("" "Q" "}{Vm Q" "}{Vm Q" ja-kanji)
("" "Q" "cm Q" "cm Q" ja-kanji)
("" "Q" "wm Q" "wm Q" ja-kanji)
("" "Q" "~igm Q" "~igm Q" ja-kanji)
("" "Q" "[Qm Q" "[Qm Q" ja-kanji)
("" "Q" "Am Q" "Am Q" ja-kanji)
("" "Q" "QKbLm Q" "QKbLm Q" ja-kanji)
("" "Q" "tio^m Q" "tio^m Q" ja-kanji)
("" "Q" "CEm Q" "CEm Q" ja-kanji)
("" "Q" "RgUm Q" "RgUm Q" ja-kanji)
("" "Q" "JMm Q" "JMm Q" ja-kanji)
("" "R" "_RgVem R" "_RgVem R" ja-kanji)
("" "R" "RZCm R" "RZCm R" ja-kanji)
("" "R" "tCm R" "tCm R" ja-kanji)
("" "R" "um R" "um R" ja-kanji)
("" "R" "J^m R" "J^m R" ja-kanji)
("" "R" "V[gm R" "V[gm R" ja-kanji)
("" "R" "RhNm R" "RhNm R" ja-kanji)
("" "R" "Imm R" "Imm R" ja-kanji)
("" "R" "\[Rm R" "\[Rm R" ja-kanji)
("" "R" "JbRm R" "JbRm R" ja-kanji)
("" "R" "gm R" "gm R" ja-kanji)
("" "R" "Gm R" "Gm R" ja-kanji)
("" "R" "Jm R" "Jm R" ja-kanji)
("" "R" "~YE~m R" "~YE~m R" ja-kanji)
("" "R" "Lclm R" "Lclm R" ja-kanji)
("" "R" "mm R" "mm R" ja-kanji)
("" "R" "nJ}m R" "nJ}m R" ja-kanji)
("" "R" "}^m R" "}^m R" ja-kanji)
("" "R" "GrXm R" "GrXm R" ja-kanji)
("" "R" "Rm R" "Rm R" ja-kanji)
("" "R" "gm R" "gm R" ja-kanji)
("" "R" "zRm R" "zRm R" ja-kanji)
("" "R" "}^Km R" "}^Km R" ja-kanji)
("" "R" "gbRm R" "gbRm R" ja-kanji)
("" "R" "gEm R" "gEm R" ja-kanji)
("" "R" "JG~m R" "JG~m R" ja-kanji)
("" "R" "cd~m R" "cd~m R" ja-kanji)
("" "S" "JX[Wm S" "JX[Wm S" ja-kanji)
("" "S" "^KCm S" "^KCm S" ja-kanji)
("" "S" "S`[m S" "S`[m S" ja-kanji)
("" "S" "V[Sm S" "V[Sm S" ja-kanji)
("" "S" "StNm S" "StNm S" ja-kanji)
("" "S" "m S" "m S" ja-kanji)
("" "S" "SNm S" "SNm S" ja-kanji)
("" "S" "m`m S" "m`m S" ja-kanji)
("" "S" "Im S" "Im S" ja-kanji)
("" "S" "Tgm S" "Tgm S" ja-kanji)
("" "S" "AIMm S" "AIMm S" ja-kanji)
("" "S" "Sm S" "Sm S" ja-kanji)
("" "S" "TSm S" "TSm S" ja-kanji)
("" "S" "SCVm S" "SCVm S" ja-kanji)
("" "S" "J^m S" "J^m S" ja-kanji)
("" "S" "A}m S" "A}m S" ja-kanji)
("" "S" "}m S" "}m S" ja-kanji)
("" "S" "_CS~m S" "_CS~m S" ja-kanji)
("" "Rc" "RWLm Rc" "RWLm Rc" ja-kanji)
("" "" "TJim RC" "TJim RC" ja-kanji)
("" "R[" "}Wm R[" "}Wm R[" ja-kanji)
("" "R[" "~Cm R[" "~Cm R[" ja-kanji)
("" "R[" "R[VNm R[" "R[VNm R[" ja-kanji)
("" "R[" "\[[m R[" "\[[m R[" ja-kanji)
("" "R[" "M[R[m R[" "M[R[m R[" ja-kanji)
("" "R[" "qJm R[" "qJm R[" ja-kanji)
("" "R[" "IIPm R[" "IIPm R[" ja-kanji)
("" "R[" "R[[Vm R[" "R[[Vm R[" ja-kanji)
("" "R[" "LNm R[" "LNm R[" ja-kanji)
("" "R[" "R[oCm R[" "R[oCm R[" ja-kanji)
("" "R[" "AcCm R[" "AcCm R[" ja-kanji)
("" "R[" "N`m R[" "N`m R[" ja-kanji)
("" "R[" "Nm R[" "Nm R[" ja-kanji)
("@" "R[" "LTLm R[" "LTLm R[" ja-kanji)
("A" "R[" "mhm R[" "mhm R[" ja-kanji)
("B" "R[" "^R[m R[" "^R[m R[" ja-kanji)
("C" "R[" "AJm R[" "AJm R[" ja-kanji)
("D" "R[" "Rmm R[" "Rmm R[" ja-kanji)
("E" "R[" "R[Vm R[" "R[Vm R[" ja-kanji)
("F" "R[" "`[R[m R[" "`[R[m R[" ja-kanji)
("G" "R[" "R[_Cm R[" "R[_Cm R[" ja-kanji)
("H" "R[" "R[TNm R[" "R[TNm R[" ja-kanji)
("I" "R[" "^N~m R[" "^N~m R[" ja-kanji)
("J" "R[" "`}^m R[" "`}^m R[" ja-kanji)
("K" "R[" "TCCm R[" "TCCm R[" ja-kanji)
("L" "R[" "qCm R[" "qCm R[" ja-kanji)
("M" "R[" "JmGm R[" "JmGm R[" ja-kanji)
("N" "R[" "CGXm R[" "CGXm R[" ja-kanji)
("O" "R[" "R[{[m R[" "R[{[m R[" ja-kanji)
("P" "R[" "clm R[" "clm R[" ja-kanji)
("Q" "R[" "Aem R[" "Aem R[" ja-kanji)
("R" "R[" "nR[m R[" "nR[m R[" ja-kanji)
("S" "R[" "JJm R[" "JJm R[" ja-kanji)
("T" "R[" "qJGm R[" "qJGm R[" ja-kanji)
("U" "R[" "Zm R[" "Zm R[" ja-kanji)
("V" "R[" "^Jum R[" "^Jum R[" ja-kanji)
("W" "R[" "ALJm R[" "ALJm R[" ja-kanji)
("X" "R[" "Tm R[" "Tm R[" ja-kanji)
("Y" "R[" "NCm R[" "NCm R[" ja-kanji)
("Z" "R[" "KbR[m R[" "KbR[m R[" ja-kanji)
("[" "R[" "tTOm R[" "tTOm R[" ja-kanji)
("\" "R[" "J}Gm R[" "J}Gm R[" ja-kanji)
("]" "R[" "Gm R[" "Gm R[" ja-kanji)
("^" "R[" "R[YCm R[" "R[YCm R[" ja-kanji)
("_" "R[" "R[[m R[" "R[[m R[" ja-kanji)
("`" "R[" "~igm R[" "~igm R[" ja-kanji)
("a" "R[" "~]m R[" "~]m R[" ja-kanji)
("b" "R[" "R[Icm R[" "R[Icm R[" ja-kanji)
("c" "R[" "R[^CVm R[" "R[^CVm R[" ja-kanji)
("d" "R[" "J^Cm R[" "J^Cm R[" ja-kanji)
("e" "R[" "QR[m R[" "QR[m R[" ja-kanji)
("f" "R[" "kJm R[" "kJm R[" ja-kanji)
("g" "R[" "NiCm R[" "NiCm R[" ja-kanji)
("h" "R[" "JPqm R[" "JPqm R[" ja-kanji)
("i" "R[" "V{m R[" "V{m R[" ja-kanji)
("j" "R[" "cim R[" "cim R[" ja-kanji)
("k" "R[" "^KXm R[" "^KXm R[" ja-kanji)
("l" "R[" "JKGm R[" "JKGm R[" ja-kanji)
("m" "R[" "EiYNm R[" "EiYNm R[" ja-kanji)
("n" "R[" "qWm R[" "qWm R[" ja-kanji)
("o" "R[" "tNR[m R[" "tNR[m R[" ja-kanji)
("p" "R[" "R[Nm R[" "R[Nm R[" ja-kanji)
("q" "R[" "R[N[Lm R[" "R[N[Lm R[" ja-kanji)
("r" "R[" "Am R[" "Am R[" ja-kanji)
("s" "R[" "CNm R[" "CNm R[" ja-kanji)
("t" "R[" "nJm R[" "nJm R[" ja-kanji)
("u" "R[" "R[_m R[" "R[_m R[" ja-kanji)
("v" "R[" "~cOm R[" "~cOm R[" ja-kanji)
("w" "R[" "JEm R[" "JEm R[" ja-kanji)
("x" "R[" "R[KCm R[" "R[KCm R[" ja-kanji)
("y" "R[" "R[\m R[" "R[\m R[" ja-kanji)
("z" "R[" "R[Um R[" "R[Um R[" ja-kanji)
("{" "R[" "AKlm R[" "AKlm R[" ja-kanji)
("|" "R[" "ebR[m R[" "ebR[m R[" ja-kanji)
("}" "R[" "^CR[Lm R[" "^CR[Lm R[" ja-kanji)
("~" "R[" "Im R[" "Im R[" ja-kanji)
("" "R[" "R[Nm R[" "R[Nm R[" ja-kanji)
("" "R[" "JIm R[" "JIm R[" ja-kanji)
("" "R[" "^JCm R[" "^JCm R[" ja-kanji)
("" "R[" "R[m CPm R[" "R[m CPm R[" ja-kanji)
("" "S[" "J^Cm S[" "J^Cm S[" ja-kanji)
("" "S[" "IrJXm S[" "IrJXm S[" ja-kanji)
("" "S[" "S[Cm S[" "S[Cm S[" ja-kanji)
("" "S[" "AEm S[" "AEm S[" ja-kanji)
("" "S[" "J{m S[" "J{m S[" ja-kanji)
("" "S[" "S[m S[" "S[m S[" ja-kanji)
("" "S[" "zm S[" "zm S[" ja-kanji)
("" "S[" "S[Jom S[" "S[Jom S[" ja-kanji)
("" "S[" "ghNm S[" "ghNm S[" ja-kanji)
("" "LN" "LNm R[W" "LNm R[W" ja-kanji)
("" "RN" "RNtNm RN" "RNtNm RN" ja-kanji)
("" "RN" "LUm RN" "LUm RN" ja-kanji)
("" "RN" "cQm RN" "cQm RN" ja-kanji)
("" "RN" "Njm RN" "Njm RN" ja-kanji)
("" "RN" "RNcm RN" "RNcm RN" ja-kanji)
("" "RN" "URNm RN" "URNm RN" ja-kanji)
("" "RN" "NOCm RN" "NOCm RN" ja-kanji)
("" "RN" "Nm RN" "Nm RN" ja-kanji)
("" "SN" "WSNm SN" "WSNm SN" ja-kanji)
("" "N" "RXm N" "RXm N" ja-kanji)
("" "[" "AVRVm RV" "AVRVm RV" ja-kanji)
("" "\" "ZCm RVL" "ZCm RVL" ja-kanji)
("" "Rc" "^`}`m Rc" "^`}`m Rc" ja-kanji)
("" "Rc" "zm Rc" "zm Rc" ja-kanji)
("" "Rc" "zlm Rc" "zlm Rc" ja-kanji)
("" "nN" "R}Ckm R}" "R}Ckm R}" ja-kanji)
("" "R" "[VRm R" "[VRm R" ja-kanji)
("" "V" "R_Pm Rm" "R_Pm Rm" ja-kanji)
("" "R" "RmSm R" "RmSm R" ja-kanji)
("" "R" "C}m R" "C}m R" ja-kanji)
("" "R" "R}m R" "R}m R" ja-kanji)
("" "R" "PRm R" "PRm R" ja-kanji)
("" "R" "JCRm R" "JCRm R" ja-kanji)
("" "R" "PbRm R" "PbRm R" ja-kanji)
("" "R" "Em R" "Em R" ja-kanji)
("" "R" "R_m R" "R_m R" ja-kanji)
("" "R" "^\Km R" "^\Km R" ja-kanji)
("" "R" "R`[m R" "R`[m R" ja-kanji)
("" "R" "lbRm R" "lbRm R" ja-kanji)
("" "R" "R|[m R" "R|[m R" ja-kanji)
("" "R" "}Wm R" "}Wm R" ja-kanji)
("" "R" "RZLm R" "RZLm R" ja-kanji)
("" "R" "RCm R" "RCm R" ja-kanji)
("" "R" "EVgm R" "EVgm R" ja-kanji)
("" "R" "^}VCm R" "^}VCm R" ja-kanji)
("" "T" "XRVm T" "XRVm T" ja-kanji)
("" "T" "zTm T" "zTm T" ja-kanji)
("" "T" "R[Tem T" "R[Tem T" ja-kanji)
("" "T" "VTXm T" "VTXm T" ja-kanji)
("" "T" "TKmm T" "TKmm T" ja-kanji)
("" "T" "q_m T" "q_m T" ja-kanji)
("" "T" "TXm T" "TXm T" ja-kanji)
("" "T" "WTm T" "WTm T" ja-kanji)
("" "T" "WSNm T^m T" "WSNm T^m T" ja-kanji)
("" "T" "AUJm T" "AUJm T" ja-kanji)
("" "T" "Xim T" "Xim T" ja-kanji)
("" "T" "TMm T" "TMm T" ja-kanji)
("" "T" "NTm T" "NTm T" ja-kanji)
("" "T" "PTm T" "PTm T" ja-kanji)
("" "U" "}_iVm U" "}_iVm U" ja-kanji)
("" "U" "Xm U" "Xm U" ja-kanji)
("" "U" "UZcm U" "UZcm U" ja-kanji)
("" "TC" "tTCm TC" "tTCm TC" ja-kanji)
("" "TC" "IXm TC" "IXm TC" ja-kanji)
("" "TC" "t^^rm TC" "t^^rm TC" ja-kanji)
("" "TC" "bgm TC" "bgm TC" ja-kanji)
("" "TC" "m TC" "m TC" ja-kanji)
("" "TC" "[TCm TC" "[TCm TC" ja-kanji)
("" "TC" "c}m TC" "c}m TC" ja-kanji)
("" "TC" "TCV[m TC" "TCV[m TC" ja-kanji)
("" "TC" "Chm TC" "Chm TC" ja-kanji)
("" "TC" "TCm[m TC" "TCm[m TC" ja-kanji)
("" "TC" "gm TC" "gm TC" ja-kanji)
("" "TC" "TCoCm TC" "TCoCm TC" ja-kanji)
("" "TC" "gVm TC" "gVm TC" ja-kanji)
("" "TC" "Xm TC" "Xm TC" ja-kanji)
("" "TC" "UCm TC" "UCm TC" ja-kanji)
("" "TC" "TCnCm TC" "TCnCm TC" ja-kanji)
("" "TC" "h[ucm TC" "h[ucm TC" ja-kanji)
("" "TC" "N_Nm TC" "N_Nm TC" ja-kanji)
("" "TC" "gfm TC" "gfm TC" ja-kanji)
("" "TC" "}cm TC" "}cm TC" ja-kanji)
("" "TC" "VTCm TC" "VTCm TC" ja-kanji)
("" "TC" "z\Cm TC" "z\Cm TC" ja-kanji)
("" "TC" "TCm TC" "TCm TC" ja-kanji)
("" "TC" "ToNm TC" "ToNm TC" ja-kanji)
("" "TC" "mZm TC" "mZm TC" ja-kanji)
("" "TC" "}hMm TC" "}hMm TC" ja-kanji)
("" "UC" "NUCm UC" "NUCm UC" ja-kanji)
("" "UC" "Am UC" "Am UC" ja-kanji)
("" "UC" "UCNm UC" "UCNm UC" ja-kanji)
("" "UC" "c~m UC" "c~m UC" ja-kanji)
("" "UC" "UCTm UC" "UCTm UC" ja-kanji)
("" "S" "TGm S" "TGm S" ja-kanji)
("" "n" "TJm n" "TJm n" ja-kanji)
("" "n" "nVm n" "nVm n" ja-kanji)
("" "JC" "TJCVm TJC" "TJCVm TJC" ja-kanji)
("" "V" "J~m Lm TJL" "J~m Lm TJL" ja-kanji)
("" "R[" "TPm TJi" "TPm TJi" ja-kanji)
("" "V[" "TNm V[" "TNm V[" ja-kanji)
("" "L" "~TLm TL" "~TLm TL" ja-kanji)
("" "L" "TC^}m TC" "TC^}m TC" ja-kanji)
("" "L" "CVwm L" "CVwm L" ja-kanji)
("" "" "gm TM" "gm TM" ja-kanji)
("" "TN" "cNm TN" "cNm TN" ja-kanji)
("" "TN" "PYm TN" "PYm TN" ja-kanji)
("" "TN" "NEm TN" "NEm TN" ja-kanji)
("" "TN" "V{m TN" "V{m TN" ja-kanji)
("" "TN" "TNlm TN" "TNlm TN" ja-kanji)
("" "TN" "cC^`m TN" "cC^`m TN" ja-kanji)
("" "TN" "JRCm TN" "JRCm TN" ja-kanji)
("" "TN" "Z}Cm TN" "Z}Cm TN" ja-kanji)
("" "TN" "TNNm TN" "TNNm TN" ja-kanji)
("" "TN" "TNCm TN" "TNCm TN" ja-kanji)
("" "TN" "TNm TN" "TNm TN" ja-kanji)
("" "I[" "TNm I[" "TNm I[" ja-kanji)
("" "PC" "TJim TP" "TJim TP" ja-kanji)
("" "ZC" "VNucm TT" "VNucm TT" ja-kanji)
("" "V" "Xv[m TW" "Xv[m TW" ja-kanji)
("" "Tc" "xbTcm Tc" "xbTcm Tc" ja-kanji)
("" "Tc" "Xm Tc" "Xm Tc" ja-kanji)
("@" "Tc" "PCTcm Tc" "PCTcm Tc" ja-kanji)
("A" "Tc" "ACTcm Tc" "ACTcm Tc" ja-kanji)
("B" "Tc" "gm Tc" "gm Tc" ja-kanji)
("C" "Tc" "}Tcm Tc" "}Tcm Tc" ja-kanji)
("D" "Tc" "t_m Tc" "t_m Tc" ja-kanji)
("E" "Tc" "RXm Tc" "RXm Tc" ja-kanji)
("F" "Tc" "Tc}m Tc" "Tc}m Tc" ja-kanji)
("G" "Uc" "tNUcm Uc" "tNUcm Uc" ja-kanji)
("H" "R[" "TcLm T" "TcLm T" ja-kanji)
("I" "ZC" "TJim To" "TJim To" ja-kanji)
("J" "nc" "ToNm nc" "ToNm nc" ja-kanji)
("K" "V[" "Trm V[" "Trm V[" ja-kanji)
("L" "R[" "TJim T" "TJim T" ja-kanji)
("M" "T" "Tm xC" "Tm xC" ja-kanji)
("N" "TC" "TXm TC" "TXm TC" ja-kanji)
("O" "T" "JX[Wm T" "JX[Wm T" ja-kanji)
("P" "T" "JTm T" "JTm T" ja-kanji)
("Q" "T" "}Cm T" "}Cm T" ja-kanji)
("R" "T" "}m T" "}m T" ja-kanji)
("S" "T" "~Wm T" "~Wm T" ja-kanji)
("T" "T" "}Nm T" "}Nm T" ja-kanji)
("U" "T" "`m T" "`m T" ja-kanji)
("V" "T" "ToVm T" "ToVm T" ja-kanji)
("W" "T" "T[m T" "T[m T" ja-kanji)
("X" "T" "TSV[m T" "TSV[m T" ja-kanji)
("Y" "T" "Em T" "Em T" ja-kanji)
("Z" "T" "TX[m T" "TX[m T" ja-kanji)
("[" "T" "Acm T" "Acm T" ja-kanji)
("\" "T" "JCRm T" "JCRm T" ja-kanji)
("]" "T" "^^Gm T" "^^Gm T" ja-kanji)
("^" "T" "Tsm T" "Tsm T" ja-kanji)
("_" "T" "XbpCm T" "XbpCm T" ja-kanji)
("`" "T" "oTm T" "oTm T" ja-kanji)
("a" "U" "Lm U" "Lm U" ja-kanji)
("b" "U" "VoNm U" "VoNm U" ja-kanji)
("c" "U" "mRm U" "mRm U" ja-kanji)
("d" "V" "cJGm V" "cJGm V" ja-kanji)
("e" "V" "jxj Rm V" "jxj Rm V" ja-kanji)
("f" "V" "EJKEm V" "EJKEm V" ja-kanji)
("g" "V" "cJEm V" "cJEm V" ja-kanji)
("h" "V" "TXm V" "TXm V" ja-kanji)
("i" "V" "cJTm V" "cJTm V" ja-kanji)
("j" "V" "LVm V" "LVm V" ja-kanji)
("k" "V" "cOm V" "cOm V" ja-kanji)
("l" "" "JX[Wm V" "JX[Wm V" ja-kanji)
("m" "V" "TCm V" "TCm V" ja-kanji)
("n" "V" "nWm V" "nWm V" ja-kanji)
("o" "V" "Alm V" "Alm V" ja-kanji)
("p" "V" "XK^m V" "XK^m V" ja-kanji)
("q" "V" "Rhm R" "Rhm R" ja-kanji)
("r" "V" "VJolm V" "VJolm V" ja-kanji)
("s" "V" "V`[\m V" "V`[\m V" ja-kanji)
("t" "V" "VV[m V" "VV[m V" ja-kanji)
("u" "V" "RRUXm V" "RRUXm V" ja-kanji)
("v" "V" "IEm V" "IEm V" ja-kanji)
("w" "V" "rm V" "rm V" ja-kanji)
("x" "V" "TTGm V" "TTGm V" ja-kanji)
("y" "V" "cgm V" "cgm V" ja-kanji)
("z" "V" "Rm V" "Rm V" ja-kanji)
("{" "V" "zhRXm V" "zhRXm V" ja-kanji)
("|" "V" "lm V" "lm V" ja-kanji)
("}" "V" "G_m V" "G_m V" ja-kanji)
("~" "V" "g}m V" "g}m V" ja-kanji)
("" "V" "Vkm V" "Vkm V" ja-kanji)
("" "V" "EWm V" "EWm V" ja-kanji)
("" "V" "VVm V" "VVm V" ja-kanji)
("" "V" "tNVm V" "tNVm V" ja-kanji)
("" "V" "^Vm V" "^Vm V" ja-kanji)
("" "V" "Cgm V" "Cgm V" ja-kanji)
("" "V" "J~m V" "J~m V" ja-kanji)
("" "V" "TLm V" "TLm V" ja-kanji)
("" "V" "eAVm V" "eAVm V" ja-kanji)
("" "V" "Aum V" "Aum V" ja-kanji)
("" "V" "C^m V" "C^m V" ja-kanji)
("" "V" "~m V" "~m V" ja-kanji)
("" "V" "Rgom V" "Rgom V" ja-kanji)
("" "V" "JVm V" "JVm V" ja-kanji)
("" "V" "^Xm V" "^Xm V" ja-kanji)
("" "V" "UbVm V" "UbVm V" ja-kanji)
("" "V" "V LJm V" "V LJm V" ja-kanji)
("" "V" "Vzm V" "Vzm V" ja-kanji)
("" "V" "^}m V" "^}m V" ja-kanji)
("" "V" "Xm V" "Xm V" ja-kanji)
("" "V" "JEm V" "JEm V" ja-kanji)
("" "V" "nm V" "nm V" ja-kanji)
("" "W" "Rgm W" "Rgm W" ja-kanji)
("" "W" "jm W" "jm W" ja-kanji)
("" "W" "TCm W" "TCm W" ja-kanji)
("" "W" "CNWm W" "CNWm W" ja-kanji)
("" "W" "JWm W" "JWm W" ja-kanji)
("" "W" "em W" "em W" ja-kanji)
("" "W" "CcNVm W" "CcNVm W" ja-kanji)
("" "W" "cm W" "cm W" ja-kanji)
("" "W" "gLm W" "gLm W" ja-kanji)
("" "W" "cMm W" "cMm W" ja-kanji)
("" "W" "VQm W" "VQm W" ja-kanji)
("" "W" "ITm W" "ITm W" ja-kanji)
("" "W" "VJm W" "VJm W" ja-kanji)
("" "W" "eVm Cm W" "eVm Cm W" ja-kanji)
("" "W" "r[Lm W" "r[Lm W" ja-kanji)
("" "W" "WVNm W" "WVNm W" ja-kanji)
("" "W" "VXm W" "VXm W" ja-kanji)
("" "W" "VR[Vem W" "VR[Vem W" ja-kanji)
("" "W" "~~m W" "~~m W" ja-kanji)
("" "W" "~YJm W" "~YJm W" ja-kanji)
("" "W" "}Nm W" "}Nm W" ja-kanji)
("" "W" "m W" "m W" ja-kanji)
("" "ZL" "VIm ZL" "VIm ZL" ja-kanji)
("" "N" "VJm N" "VJm N" ja-kanji)
("" "VL" "z[VLm VL" "z[VLm VL" ja-kanji)
("" "VL" "`VLm VL" "`VLm VL" ja-kanji)
("" "f" "gm VM" "gm VM" ja-kanji)
("" "WN" "eWNm WN" "eWNm WN" ja-kanji)
("" "WN" "JPWNm WN" "JPWNm WN" ja-kanji)
("" "VV" "W[jNm VV" "W[jNm VV" ja-kanji)
("" "_" "VYNm _" "VYNm _" ja-kanji)
("" "ii" "JX[Wm V`" "JX[Wm V`" ja-kanji)
("" "Vc" "VJm Vc" "VJm Vc" ja-kanji)
("" "Vc" "gm Vc" "gm Vc" ja-kanji)
("" "Vc" "EViEm Vc" "EViEm Vc" ja-kanji)
("" "Vc" "Vbgm Vc" "Vbgm Vc" ja-kanji)
("" "Vc" "L[Vcm Vc" "L[Vcm Vc" ja-kanji)
("" "Vc" "RgSgNm Vc" "RgSgNm Vc" ja-kanji)
("" "Vc" "Vm Vc" "Vm Vc" ja-kanji)
("" "Vc" "EVm Vc" "EVm Vc" ja-kanji)
("" "Vc" "}Cm Vc" "}Cm Vc" ja-kanji)
("" "Vc" "zVcm Vc" "zVcm Vc" ja-kanji)
("" "Wc" "~mm Wc" "~mm Wc" ja-kanji)
("" "z[" "Vg~m z[" "Vg~m z[" ja-kanji)
("" "ZC" "VmUTm Vm" "VmUTm Vm" ja-kanji)
("" "V" "Vmum TC" "Vmum TC" ja-kanji)
("" "V" "VoJm Vo" "VoJm Vo" ja-kanji)
("" "V" "Vom V" "Vom V" ja-kanji)
("" "" "VoVom " "VoVom " ja-kanji)
("" "YC" "Vxm YC" "Vxm YC" ja-kanji)
("" "R[" "V}E}m V}" "V}E}m V}" ja-kanji)
("" "V" "R[Vm V" "R[Vm V" ja-kanji)
("" "V" "EcXm V" "EcXm V" ja-kanji)
("" "V" "Cm V" "Cm V" ja-kanji)
("" "V" "Xem V" "Xem V" ja-kanji)
("" "V" "Xm V" "Xm V" ja-kanji)
("" "V" "iim V" "iim V" ja-kanji)
("" "V" "jm V" "jm V" ja-kanji)
("" "V" "VJCm V" "VJCm V" ja-kanji)
("" "V" "EXMkm V" "EXMkm V" ja-kanji)
("" "V" "mm V" "mm V" ja-kanji)
("" "V" "A}m V" "A}m V" ja-kanji)
("" "V" "N}m V" "N}m V" ja-kanji)
("" "V" "TGMm V" "TGMm V" ja-kanji)
("" "W" "wrm W" "wrm W" ja-kanji)
("" "W" "RV}m W" "RV}m W" ja-kanji)
("" "VN" "Jm VN" "Jm VN" ja-kanji)
("" "VN" "qVNm VN" "qVNm VN" ja-kanji)
("" "VN" "VNn`m VN" "VNn`m VN" ja-kanji)
("" "VN" "Lm qVNm VN" "Lm qVNm VN" ja-kanji)
("" "VN" "VNlcm VN" "VNlcm VN" ja-kanji)
("" "VN" "_VNm VN" "_VNm VN" ja-kanji)
("" "VN" "Nm VN" "Nm VN" ja-kanji)
("" "VN" "JCVNm VN" "JCVNm VN" ja-kanji)
("" "VN" "XYm VN" "XYm VN" ja-kanji)
("" "WN" "JCm WN" "JCm WN" ja-kanji)
("" "WN" "TrVCm WN" "TrVCm WN" ja-kanji)
("" "WN" "Cm WN" "Cm WN" ja-kanji)
("" "WN" "WbLm WN" "WbLm WN" ja-kanji)
("" "V" "kVm V" "kVm V" ja-kanji)
("" "V" "gm V" "gm V" ja-kanji)
("" "V" "}m V" "}m V" ja-kanji)
("" "V" "em V" "em V" ja-kanji)
("" "V" "AJm V" "AJm V" ja-kanji)
("" "V" "Rgm V" "Rgm V" ja-kanji)
("" "V" "Jm V" "Jm V" ja-kanji)
("" "V" "VUm V" "VUm V" ja-kanji)
("" "V" "^lm V" "^lm V" ja-kanji)
("" "V" "V[m V" "V[m V" ja-kanji)
("" "V" "ILm V" "ILm V" ja-kanji)
("" "V" "TPm V" "TPm V" ja-kanji)
("" "V" "Nrm V" "Nrm V" ja-kanji)
("" "W" "WL[m W" "WL[m W" ja-kanji)
("" "W" "EPm W" "EPm W" ja-kanji)
("" "W" "mEm W" "mEm W" ja-kanji)
("" "W" "RguLm W" "RguLm W" ja-kanji)
("" "W" "TYPm W" "TYPm W" ja-kanji)
("" "W" "Lm W" "Lm W" ja-kanji)
("" "W" "qm W" "qm W" ja-kanji)
("" "W" "W[m W" "W[m W" ja-kanji)
("" "V[" "gm V[" "gm V[" ja-kanji)
("" "V[" "ITm V[" "ITm V[" ja-kanji)
("" "V[" "}m V[" "}m V[" ja-kanji)
("@" "V[" "V[L[m V[" "V[L[m V[" ja-kanji)
("A" "V[" "cNm V[" "cNm V[" ja-kanji)
("B" "V[" "L[V[m V[" "L[V[m V[" ja-kanji)
("C" "V[" "V[m V[" "V[m V[" ja-kanji)
("D" "V[" "ECm V[" "ECm V[" ja-kanji)
("E" "V[" "q[m V[" "q[m V[" ja-kanji)
("F" "V[" "iJXm X" "iJXm X" ja-kanji)
("G" "V[" "qCfm V[" "qCfm V[" ja-kanji)
("H" "V[" "ALm V[" "ALm V[" ja-kanji)
("I" "V[" "Im V[" "Im V[" ja-kanji)
("J" "V[" "VV[m V[" "VV[m V[" ja-kanji)
("K" "V[" "iEm V[" "iEm V[" ja-kanji)
("L" "V[" "NTCm V[" "NTCm V[" ja-kanji)
("M" "V[" "tlm V[" "tlm V[" ja-kanji)
("N" "V[" "V[V[m V[" "V[V[m V[" ja-kanji)
("O" "V[" "V[MCm V[" "V[MCm V[" ja-kanji)
("P" "V[" "I\Em V[" "I\Em V[" ja-kanji)
("Q" "V[" "IV[m V[" "IV[m V[" ja-kanji)
("R" "V[" "Pm V[" "Pm V[" ja-kanji)
("S" "V[" "V[{Nm V[" "V[{Nm V[" ja-kanji)
("T" "V[" "V[JVm V[" "V[JVm V[" ja-kanji)
("U" "V[" "V[`[m V[" "V[`[m V[" ja-kanji)
("V" "V[" "NCm V[" "NCm V[" ja-kanji)
("W" "V[" "Acm V[" "Acm V[" ja-kanji)
("X" "V[" "~jNCm V[" "~jNCm V[" ja-kanji)
("Y" "W[" "jxm W[" "jxm W[" ja-kanji)
("Z" "W[" "Xm W[" "Xm W[" ja-kanji)
("[" "W[" "W[um W[" "W[um W[" ja-kanji)
("\" "W[" "JX[Wm W[" "JX[Wm W[" ja-kanji)
("]" "W[" "V^KEm W[" "V^KEm W[" ja-kanji)
("^" "W[" "W[om W[" "W[om W[" ja-kanji)
("_" "W[" "Jm W[" "Jm W[" ja-kanji)
("`" "W[" "Vm W[" "Vm W[" ja-kanji)
("a" "W[" "VuCm W[" "VuCm W[" ja-kanji)
("b" "W[" "Pmm W[" "Pmm W[" ja-kanji)
("c" "W[" "^em W[" "^em W[" ja-kanji)
("d" "W[" "ICm W[" "ICm W[" ja-kanji)
("e" "W[" "PW[m W[" "PW[m W[" ja-kanji)
("f" "VN" "IWm VN" "IWm VN" ja-kanji)
("g" "VN" "cgjm VN" "cgjm VN" ja-kanji)
("h" "VN" "hm VN" "hm VN" ja-kanji)
("i" "VN" "VNWm VN" "VNWm VN" ja-kanji)
("j" "VN" "CEm VN" "CEm VN" ja-kanji)
("k" "VN" "`am VN" "`am VN" ja-kanji)
("l" "VN" "ccVm VN" "ccVm VN" ja-kanji)
("m" "WN" "WN`[m WN" "WN`[m WN" ja-kanji)
("n" "WN" "Em WN" "Em WN" ja-kanji)
("o" "Vc" "fm Vc" "fm Vc" ja-kanji)
("p" "Wc" "VWcm Wc" "VWcm Wc" ja-kanji)
("q" "Wc" "mxm Wc" "mxm Wc" ja-kanji)
("r" "V" "gVm V" "gVm V" ja-kanji)
("s" "V" "PVCm V" "PVCm V" ja-kanji)
("t" "V" "nm V" "nm V" ja-kanji)
("u" "V" "}^^Nm V" "}^^Nm V" ja-kanji)
("v" "V" "VR[m V" "VR[m V" ja-kanji)
("w" "V" "NQm V" "NQm V" ja-kanji)
("x" "V" "Vom V" "Vom V" ja-kanji)
("y" "W" "qWm W" "qWm W" ja-kanji)
("z" "W" "WJLm W" "WJLm W" ja-kanji)
("{" "W" "QWm W" "QWm W" ja-kanji)
("|" "W" "Lm ^em W" "Lm ^em W" ja-kanji)
("}" "W" "WVNm W" "WVNm W" ja-kanji)
("~" "W" "AcCm W" "AcCm W" ja-kanji)
("" "W" "Wrm W" "Wrm W" ja-kanji)
("" "W" "EIEm W" "EIEm W" ja-kanji)
("" "W" "^em W" "^em W" ja-kanji)
("" "W" "WXCm W" "WXCm W" ja-kanji)
("" "W" "Om W" "Om W" ja-kanji)
("" "W" "WVm W" "WVm W" ja-kanji)
("" "W" "LUPm W" "LUPm W" ja-kanji)
("" "W" "Wom W" "Wom W" ja-kanji)
("" "V" "Vm V" "Vm V" ja-kanji)
("" "V" "ncm V" "ncm V" ja-kanji)
("" "V" "gRm V" "gRm V" ja-kanji)
("" "V" "AcCm V" "AcCm V" ja-kanji)
("" "V" "AP{mm V" "AP{mm V" ja-kanji)
("" "V" "iMTm V" "iMTm V" ja-kanji)
("" "V" "V~m V" "V~m V" ja-kanji)
("" "V" "Im V" "Im V" ja-kanji)
("" "V" "PCTcVm V" "PCTcVm V" ja-kanji)
("" "V" "JNm V" "JNm V" ja-kanji)
("" "V" "JVm V" "JVm V" ja-kanji)
("" "V" "Cm V" "Cm V" ja-kanji)
("" "V" "CbVm V" "CbVm V" ja-kanji)
("" "W" "^XPm W" "^XPm W" ja-kanji)
("" "W" "WWVm W" "WWVm W" ja-kanji)
("" "W" "Iim W" "Iim W" ja-kanji)
("" "W" "nVKLm W" "nVKLm W" ja-kanji)
("" "W" "Im W" "Im W" ja-kanji)
("" "W" "Xm W" "Xm W" ja-kanji)
("" "W" "XLm W" "XLm W" ja-kanji)
("" "W" "m]Nm W" "m]Nm W" ja-kanji)
("" "V[" "LYm V[" "LYm V[" ja-kanji)
("" "V[" "cOiEm V[" "cOiEm V[" ja-kanji)
("" "V[" "Jcm V[" "Jcm V[" ja-kanji)
("" "V[" "^N~m V[" "^N~m V[" ja-kanji)
("" "V[" "}Xm V[" "}Xm V[" ja-kanji)
("" "V[" "Xm V[" "Xm V[" ja-kanji)
("" "V[" "~nm V[" "~nm V[" ja-kanji)
("" "V[" "ALiEm V[" "ALiEm V[" ja-kanji)
("" "V[" "KbV[_m V[" "KbV[_m V[" ja-kanji)
("" "V[" "im V[" "im V[" ja-kanji)
("" "V[" "XXm V[" "XXm V[" ja-kanji)
("" "V[" "JPm V[" "JPm V[" ja-kanji)
("" "V[" "V[tm V[" "V[tm V[" ja-kanji)
("" "V[" "Cm V[" "Cm V[" ja-kanji)
("" "V[" "V[Om V[" "V[Om V[" ja-kanji)
("" "V[" "`CTCm V[" "`CTCm V[" ja-kanji)
("" "V[" "XNiCm V[" "XNiCm V[" ja-kanji)
("" "V[" "iIm V[" "iIm V[" ja-kanji)
("" "V[" "V[m V[" "V[m V[" ja-kanji)
("" "V[" "Jm V[" "Jm V[" ja-kanji)
("" "V[" "Rm V[" "Rm V[" ja-kanji)
("" "V[" "ALJm V[" "ALJm V[" ja-kanji)
("" "V[" "EP^}Em V[" "EP^}Em V[" ja-kanji)
("" "V[" "V[Nm V[" "V[Nm V[" ja-kanji)
("" "V[" "}lNm V[" "}lNm V[" ja-kanji)
("" "V[" "^iSRm V[" "^iSRm V[" ja-kanji)
("" "V[" "XoCm V[" "XoCm V[" ja-kanji)
("" "V[" "m{m V[" "m{m V[" ja-kanji)
("" "V[" "q t^cm V[" "q t^cm V[" ja-kanji)
("" "V[" "V[m V[" "V[m V[" ja-kanji)
("" "V[" "XCV[m V[" "XCV[m V[" ja-kanji)
("" "V[" "}cm V[" "}cm V[" ja-kanji)
("" "V[" "RYGm V[" "RYGm V[" ja-kanji)
("" "V[" "NXm Lm V[" "NXm Lm V[" ja-kanji)
("" "V[" "LRm V[" "LRm V[" ja-kanji)
("" "V[" "k}m V[" "k}m V[" ja-kanji)
("" "V[" "LGm V[" "LGm V[" ja-kanji)
("" "V[" "^m V[" "^m V[" ja-kanji)
("" "V[" "V[im V[" "V[im V[" ja-kanji)
("" "V[" "Nm V[" "Nm V[" ja-kanji)
("" "V[" "RQm V[" "RQm V[" ja-kanji)
("" "V[" "em V[" "em V[" ja-kanji)
("" "V[" "}Cm V[" "}Cm V[" ja-kanji)
("" "V[" "nuNm V[" "nuNm V[" ja-kanji)
("" "V[" "KXm V[" "KXm V[" ja-kanji)
("" "V[" "TSV[m V[" "TSV[m V[" ja-kanji)
("" "V[" "nbV[m V[" "nbV[m V[" ja-kanji)
("" "V[" "giGm V[" "giGm V[" ja-kanji)
("" "V[" "uV[m V[" "uV[m V[" ja-kanji)
("" "V[" "Em V[" "Em V[" ja-kanji)
("" "V[" "\IEm V[" "\IEm V[" ja-kanji)
("" "V[" "V[JCVm V[" "V[JCVm V[" ja-kanji)
("" "V[" "V[][Km V[" "V[][Km V[" ja-kanji)
("" "V[" "niV[um V[" "niV[um V[" ja-kanji)
("" "V[" "}Rm V[" "}Rm V[" ja-kanji)
("" "V[" "oV[m V[" "oV[m V[" ja-kanji)
("" "V[" "V[gcm V[" "V[gcm V[" ja-kanji)
("" "V[" "m V[" "m V[" ja-kanji)
("" "V[" "\V[m V[" "\V[m V[" ja-kanji)
("" "V[" "AJXm V[" "AJXm V[" ja-kanji)
("" "V[" "~Rgmm V[" "~Rgmm V[" ja-kanji)
("" "V[" "NVCm V[" "NVCm V[" ja-kanji)
("" "V[" "][m V[" "][m V[" ja-kanji)
("" "V[" "zm V[" "zm V[" ja-kanji)
("" "V[" "V[m V[" "V[m V[" ja-kanji)
("" "V[" "hm V[" "hm V[" ja-kanji)
("" "V[" "V[LT}m V[" "V[LT}m V[" ja-kanji)
("" "V[" "Jlm V[" "Jlm V[" ja-kanji)
("" "V[" "TVTm V[" "TVTm V[" ja-kanji)
("" "V[" "Tm V[" "Tm V[" ja-kanji)
("" "W[" "EGm W[" "EGm W[" ja-kanji)
("" "W[" "^Pm W[" "^Pm W[" ja-kanji)
("" "W[" "W[wQm W[" "W[wQm W[" ja-kanji)
("" "W[" "mm W[" "mm W[" ja-kanji)
("" "W[" "W[_m W[" "W[_m W[" ja-kanji)
("" "W[" "A}m W[" "A}m W[" ja-kanji)
("" "W[" "Vm W[" "Vm W[" ja-kanji)
("" "W[" "om W[" "om W[" ja-kanji)
("" "W[" "c`m W[" "c`m W[" ja-kanji)
("" "W[" "IW[Tm W[" "IW[Tm W[" ja-kanji)
("" "W[" "clm W[" "clm W[" ja-kanji)
("" "W[" "iTPm W[" "iTPm W[" ja-kanji)
("" "W[" "~_m W[" "~_m W[" ja-kanji)
("" "W[" "W[Nm W[" "W[Nm W[" ja-kanji)
("" "W[" "cGm W[" "cGm W[" ja-kanji)
("" "W[" "Lm W[" "Lm W[" ja-kanji)
("" "W[" "W[L[m W[" "W[L[m W[" ja-kanji)
("" "W[" "^^~m W[" "^^~m W[" ja-kanji)
("" "W[" "~mm W[" "~mm W[" ja-kanji)
("" "W[" "Xm W[" "Xm W[" ja-kanji)
("" "W[" "Ym W[" "Ym W[" ja-kanji)
("" "W[" "W[][m W[" "W[][m W[" ja-kanji)
("" "W[" "eW[m W[" "eW[m W[" ja-kanji)
("" "VN" "^mm VN" "^mm VN" ja-kanji)
("" "VN" "njm VN" "njm VN" ja-kanji)
("" "VN" "JUm VN" "JUm VN" ja-kanji)
("@" "VN" "tNm VN" "tNm VN" ja-kanji)
("A" "VN" "EGm VN" "EGm VN" ja-kanji)
("B" "VN" "tGm VN" "tGm VN" ja-kanji)
("C" "VN" "[\Nm VN" "[\Nm VN" ja-kanji)
("D" "VN" "Im VN" "Im VN" ja-kanji)
("E" "VN" "VNM[m VN" "VNM[m VN" ja-kanji)
("F" "C" "Cm VN" "Cm VN" ja-kanji)
("G" "VN" "Tm VN" "Tm VN" ja-kanji)
("H" "VN" "^xm VN" "^xm VN" ja-kanji)
("I" "VN" "Vom VN" "Vom VN" ja-kanji)
("J" "WN" "nYJVm WN" "nYJVm WN" ja-kanji)
("K" "R[" "Vm R[" "Vm R[" ja-kanji)
("L" "V" "mrm V" "mrm V" ja-kanji)
("M" "V" "VWm V" "VWm V" ja-kanji)
("N" "V" "IJXm V" "IJXm V" ja-kanji)
("O" "V" "N`rm V" "N`rm V" ja-kanji)
("P" "V" "nm V" "nm V" ja-kanji)
("Q" "V" "lm V" "lm V" ja-kanji)
("R" "V" "VTm V" "VTm V" ja-kanji)
("S" "V" "RRm V" "RRm V" ja-kanji)
("T" "V" "ccVm V" "ccVm V" ja-kanji)
("U" "V" "tm V" "tm V" ja-kanji)
("V" "V" "A^VCm V" "A^VCm V" ja-kanji)
("W" "V" "Vm Njm V" "Vm Njm V" ja-kanji)
("X" "V" "m V" "m V" ja-kanji)
("Y" "V" "nVo~m V" "nVo~m V" ja-kanji)
("Z" "V" "q^Xm V" "q^Xm V" ja-kanji)
("[" "V" "tJCm V" "tJCm V" ja-kanji)
("\" "V" "[Xm V" "[Xm V" ja-kanji)
("]" "V" "t[Vm V" "t[Vm V" ja-kanji)
("^" "V" "VWcm V" "VWcm V" ja-kanji)
("_" "V" "J~m V" "J~m V" ja-kanji)
("`" "V" "Jm }Gm V" "Jm }Gm V" ja-kanji)
("a" "V" "VVm V" "VVm V" ja-kanji)
("b" "V" "_CWm V" "_CWm V" ja-kanji)
("c" "V" "N_mm V" "N_mm V" ja-kanji)
("d" "V" "^LMm V" "^LMm V" ja-kanji)
("e" "V" "Im V" "Im V" ja-kanji)
("f" "V" "~m V" "~m V" ja-kanji)
("g" "V" "~m V" "~m V" ja-kanji)
("h" "V" "JCm V" "JCm V" ja-kanji)
("i" "V" "XXm V" "XXm V" ja-kanji)
("j" "V" "nm V" "nm V" ja-kanji)
("k" "V" "tGm V" "tGm V" ja-kanji)
("l" "W" "qgm W" "qgm W" ja-kanji)
("m" "W" "WMm W" "WMm W" ja-kanji)
("n" "W" "nm W" "nm W" ja-kanji)
("o" "W" "`m W" "`m W" ja-kanji)
("p" "W" "~um W" "~um W" ja-kanji)
("q" "W" "Wm W" "Wm W" ja-kanji)
("r" "W" "nin_m W" "nin_m W" ja-kanji)
("s" "W" "cNXm W" "cNXm W" ja-kanji)
("t" "W" "W][m W" "W][m W" ja-kanji)
("u" "W" "^Ylm W" "^Ylm W" ja-kanji)
("v" "W" "W\Nm W" "W\Nm W" ja-kanji)
("w" "W" "W`m W" "W`m W" ja-kanji)
("x" "W" "W^Cm W" "W^Cm W" ja-kanji)
("y" "X" "^Xm X" "^Xm X" ja-kanji)
("z" "X" "Xm X" "Xm X" ja-kanji)
("{" "X" "XxJNm X" "XxJNm X" ja-kanji)
("|" "TN" "Xm TN" "Xm TN" ja-kanji)
("}" "Y" "nJm Y" "nJm Y" ja-kanji)
("~" "`[" "`[{[m `[" "`[{[m `[" ja-kanji)
("" "Y" "YVm Y" "YVm Y" ja-kanji)
("" "XC" "tNm XC" "tNm XC" ja-kanji)
("" "XC" "^m XC" "^m XC" ja-kanji)
("" "XC" "QXCm XC" "QXCm XC" ja-kanji)
("" "XC" "IXm XC" "IXm XC" ja-kanji)
("" "XC" "~Ym XC" "~Ym XC" ja-kanji)
("" "XC" "XCWm XC" "XCWm XC" ja-kanji)
("" "XC" "lm XC" "lm XC" ja-kanji)
("" "XC" "CLim XC" "CLim XC" ja-kanji)
("" "XC" "~hm XC" "~hm XC" ja-kanji)
("" "XC" "IgGm XC" "IgGm XC" ja-kanji)
("" "XC" "gQm XC" "gQm XC" ja-kanji)
("" "XC" "Em XC" "Em XC" ja-kanji)
("" "XC" "Lm XC" "Lm XC" ja-kanji)
("" "XC" "cm XC" "cm XC" ja-kanji)
("" "YC" "V^KEm YC" "V^KEm YC" ja-kanji)
("" "YC" "~Ym YC" "~Ym YC" ja-kanji)
("" "YC" "RcYCm YC" "RcYCm YC" ja-kanji)
("" "X[" "g[gum X[" "g[gum X[" ja-kanji)
("" "X[" "JTm X[" "JTm X[" ja-kanji)
("" "X[" "JYm X[" "JYm X[" ja-kanji)
("" "X[" "X[WNm X[" "X[WNm X[" ja-kanji)
("" "X[" "X[ZCm X[" "X[ZCm X[" ja-kanji)
("" "X[" "qim X[" "qim X[" ja-kanji)
("" "L" "XGm L" "XGm L" ja-kanji)
("" "T" "XMm T" "XMm T" ja-kanji)
("" "V[" "XMm RNW" "XMm RNW" ja-kanji)
("" "J" "XKm J" "XKm J" ja-kanji)
("" "n" "XRum n" "XRum n" ja-kanji)
("" "WN" "gm XY" "gm XY" ja-kanji)
("" "L" "X\m L" "X\m L" ja-kanji)
("" "`[" "Xm `[" "Xm `[" ja-kanji)
("" "V[" "Xm V[" "Xm V[" ja-kanji)
("" "X" "iKTm X" "iKTm X" ja-kanji)
("" "Z" "m Z" "m Z" ja-kanji)
("" "Z" "Zgm Z" "Zgm Z" ja-kanji)
("" "Z" "Elm Z" "Elm Z" ja-kanji)
("" "[" "Rm [" "Rm [" ja-kanji)
("" "ZC" "XSCm ZC" "XSCm ZC" ja-kanji)
("" "ZC" "ZCMm ZC" "ZCMm ZC" ja-kanji)
("" "ZC" "CLICm ZC" "CLICm ZC" ja-kanji)
("" "ZC" "EWm ZC" "EWm ZC" ja-kanji)
("" "ZC" "ZCtNVm ZC" "ZCtNVm ZC" ja-kanji)
("" "ZC" "ZCVcm ZC" "ZCVcm ZC" ja-kanji)
("" "ZC" "im ZC" "im ZC" ja-kanji)
("" "ZC" "ZCWm ZC" "ZCWm ZC" ja-kanji)
("" "ZC" "ggmGm ZC" "ggmGm ZC" ja-kanji)
("" "ZC" "zVm ZC" "zVm ZC" ja-kanji)
("" "ZC" "nm ZC" "nm ZC" ja-kanji)
("" "ZC" "Xm ZC" "Xm ZC" ja-kanji)
("" "ZC" "gXVm ZC" "gXVm ZC" ja-kanji)
("" "ZC" "^_VCm ZC" "^_VCm ZC" ja-kanji)
("" "ZC" "LCm ZC" "LCm ZC" ja-kanji)
("" "ZC" "MZCm ZC" "MZCm ZC" ja-kanji)
("" "ZC" "CLm ZC" "CLm ZC" ja-kanji)
("" "ZC" "TJm ZC" "TJm ZC" ja-kanji)
("" "ZC" "ZCVm ZC" "ZCVm ZC" ja-kanji)
("" "ZC" "qWm ZC" "qWm ZC" ja-kanji)
("" "ZC" "RGm ZC" "RGm ZC" ja-kanji)
("" "ZC" "ZC][m ZC" "ZC][m ZC" ja-kanji)
("" "ZC" "jVm ZC" "jVm ZC" ja-kanji)
("" "ZC" "}Rgm ZC" "}Rgm ZC" ja-kanji)
("" "ZC" "`JEm ZC" "`JEm ZC" ja-kanji)
("" "ZC" "REm ZC" "REm ZC" ja-kanji)
("" "ZC" "Nm ZC" "Nm ZC" ja-kanji)
("" "ZC" "Tm ZC" "Tm ZC" ja-kanji)
("" "ZC" "AIm ZC" "AIm ZC" ja-kanji)
("" "ZC" "VYJm ZC" "VYJm ZC" ja-kanji)
("" "ZC" "CbZCm ZC" "CbZCm ZC" ja-kanji)
("" "[C" "[CVm [C" "[CVm [C" ja-kanji)
("" "[C" "Cm [C" "Cm [C" ja-kanji)
("" "ZL" "ZLKm ZL" "ZLKm ZL" ja-kanji)
("" "ZL" "UZLm ZL" "UZLm ZL" ja-kanji)
("" "ZL" "IVCm ZL" "IVCm ZL" ja-kanji)
("" "ZL" "VZLm ZL" "VZLm ZL" ja-kanji)
("" "ZL" "V]Pm ZL" "V]Pm ZL" ja-kanji)
("" "ZL" "JVm ZL" "JVm ZL" ja-kanji)
("" "ZL" "uZLm ZL" "uZLm ZL" ja-kanji)
("" "ZL" "CVm ZL" "CVm ZL" ja-kanji)
("" "ZL" "cm ZL" "cm ZL" ja-kanji)
("" "ZL" "RZLm ZL" "RZLm ZL" ja-kanji)
("" "ZL" "ZCZLm ZL" "ZCZLm ZL" ja-kanji)
("" "ZL" "ZLYCm ZL" "ZLYCm ZL" ja-kanji)
("" "ZL" "Zm ZL" "Zm ZL" ja-kanji)
("" "ZL" "AJm ZL" "AJm ZL" ja-kanji)
("" "ZL" "AVAgm ZL" "AVAgm ZL" ja-kanji)
("" "ZL" "Agm L[W" "Agm L[W" ja-kanji)
("" "ZL" "ZLJNm ZL" "ZLJNm ZL" ja-kanji)
("" "Zc" "Lm Zc" "Lm Zc" ja-kanji)
("" "Zc" "ZbVm Zc" "ZbVm Zc" ja-kanji)
("" "Zc" "cOm Zc" "cOm Zc" ja-kanji)
("" "Zc" "ZbV[m Zc" "ZbV[m Zc" ja-kanji)
("" "Zc" "Im Zc" "Im Zc" ja-kanji)
("" "Zc" "[Pm Zc" "[Pm Zc" ja-kanji)
("" "Zc" "Zbg[m Zc" "Zbg[m Zc" ja-kanji)
("" "Zc" "tVm Zc" "tVm Zc" ja-kanji)
("" "Zc" "gNm Zc" "gNm Zc" ja-kanji)
("" "Zc" "Lm Zc" "Lm Zc" ja-kanji)
("" "[c" "^Gm [c" "^Gm [c" ja-kanji)
("" "[c" "V^m [c" "V^m [c" ja-kanji)
("" "[" "R`[m Z~" "R`[m Z~" ja-kanji)
("" "Z" "Zjm Z" "Zjm Z" ja-kanji)
("" "Z" "TLm Z" "TLm Z" ja-kanji)
("" "Z" "^Cm Z" "^Cm Z" ja-kanji)
("" "Z" "EiEm Z" "EiEm Z" ja-kanji)
("" "Z" "Zfm Z" "Zfm Z" ja-kanji)
("" "Z" "bpm Z" "bpm Z" ja-kanji)
("" "Z" "gKm Z" "gKm Z" ja-kanji)
("" "Z" "Jm Z" "Jm Z" ja-kanji)
("" "Z" "^^JEm Z" "^^JEm Z" ja-kanji)
("" "Z" "I[Mm Z" "I[Mm Z" ja-kanji)
("" "Z" "ZWm Z" "ZWm Z" ja-kanji)
("" "Z" "rm Z" "rm Z" ja-kanji)
("" "Z" "Z_m Z" "Z_m Z" ja-kanji)
("" "Z" "CY~m Z" "CY~m Z" ja-kanji)
("" "Z" "ATCm Z" "ATCm Z" ja-kanji)
("" "Z" "AEm Z" "AEm Z" ja-kanji)
("" "Z" "\m Z" "\m Z" ja-kanji)
("" "Z" "q\m Z" "q\m Z" ja-kanji)
("" "Z" "Cm Z" "Cm Z" ja-kanji)
("" "Z" "AIm Z" "AIm Z" ja-kanji)
("" "Z" "Zom Z" "Zom Z" ja-kanji)
("" "Z" "EKcm Z" "EKcm Z" ja-kanji)
("" "Z" "^Pj }Gm Z" "^Pj }Gm Z" ja-kanji)
("" "Z" "Cgm Z" "Cgm Z" ja-kanji)
("@" "Z" "Zm[m Z" "Zm[m Z" ja-kanji)
("A" "Z" "Z{[m Z" "Z{[m Z" ja-kanji)
("B" "Z" "CZm Z" "CZm Z" ja-kanji)
("C" "Z" "^KGm Z" "^KGm Z" ja-kanji)
("D" "Z" "tlm Z" "tlm Z" ja-kanji)
("E" "Z" "XXm Z" "XXm Z" ja-kanji)
("F" "Z" "ZTNm Z" "ZTNm Z" ja-kanji)
("G" "Z" "CVCm Z" "CVCm Z" ja-kanji)
("H" "Z" "WbZm Z" "WbZm Z" ja-kanji)
("I" "Z" "Gum Z" "Gum Z" ja-kanji)
("J" "Z" "Zgm Z" "Zgm Z" ja-kanji)
("K" "Z" "[jm Z" "[jm Z" ja-kanji)
("L" "Z" "Zecm Z" "Zecm Z" ja-kanji)
("M" "Z" "qNm Z" "qNm Z" ja-kanji)
("N" "Z" "AUJm Z" "AUJm Z" ja-kanji)
("O" "[" "}Gm [" "}Gm [" ja-kanji)
("P" "[" "Cm [" "Cm [" ja-kanji)
("Q" "[" "[Nm [" "[Nm [" ja-kanji)
("R" "[" "V[m [" "V[m [" ja-kanji)
("S" "[" "}b^Nm [" "}b^Nm [" ja-kanji)
("T" "[" "[V[m [" "[V[m [" ja-kanji)
("U" "[" "cNEm [" "cNEm [" ja-kanji)
("V" "[" "I[m [" "I[m [" ja-kanji)
("W" "Z`[g" "iKTm Z`" "iKTm Z`" ja-kanji)
("X" "\" "~\m \" "~\m \" ja-kanji)
("Y" "\" "\][m \" "\][m \" ja-kanji)
("Z" "\" "P\m \" "P\m \" ja-kanji)
("[" "\" "INm \" "INm \" ja-kanji)
("\" "\" "L\Wm \" "L\Wm \" ja-kanji)
("]" "\" "\[\tm \" "\[\tm \" ja-kanji)
("^" "\" "ZC\m \" "ZC\m \" ja-kanji)
("_" "\" "lEm \" "lEm \" ja-kanji)
("`" "\" "\XCm \" "\XCm \" ja-kanji)
("a" "\" "EgCm \" "EgCm \" ja-kanji)
("b" "\" "CVYGm \" "CVYGm \" ja-kanji)
("c" "\" "\Zm \" "\Zm \" ja-kanji)
("d" "\" "\[Cm \" "\[Cm \" ja-kanji)
("e" "\" "ACm \" "ACm \" ja-kanji)
("f" "\" "gm \" "gm \" ja-kanji)
("g" "\" "Nm \" "Nm \" ja-kanji)
("h" "\" "~KGm \" "~KGm \" ja-kanji)
("i" "\" "Eb^Gm \" "Eb^Gm \" ja-kanji)
("j" "\" "nom \" "nom \" ja-kanji)
("k" "\" "TJm{m \" "TJm{m \" ja-kanji)
("l" "\" "lY~m \" "lY~m \" ja-kanji)
("m" "\[" "\[m \[" "\[m \[" ja-kanji)
("n" "\[" "\[cm \[" "\[cm \[" ja-kanji)
("o" "\[" "t^Sm \[" "t^Sm \[" ja-kanji)
("p" "\[" "NTm \[" "NTm \[" ja-kanji)
("q" "\[" "Nm \[" "Nm \[" ja-kanji)
("r" "\[" "tNm " "tNm " ja-kanji)
("s" "\[" "\[lm \[" "\[lm \[" ja-kanji)
("t" "\[" "Jifm \[" "Jifm \[" ja-kanji)
("u" "\[" "TJm \[" "TJm \[" ja-kanji)
("v" "\[" "\[m Njm \[" "\[m Njm \[" ja-kanji)
("w" "\[" "`\[m \[" "`\[m \[" ja-kanji)
("x" "\[" "Om \[" "Om \[" ja-kanji)
("y" "\[" "\[[m \[" "\[[m \[" ja-kanji)
("z" "\[" "Q\[m \[" "Q\[m \[" ja-kanji)
("{" "\[" "TKXm \[" "TKXm \[" ja-kanji)
("|" "\[" "nNm \[" "nNm \[" ja-kanji)
("}" "\[" "TXm \[" "TXm \[" ja-kanji)
("~" "\[" "\[nm \[" "\[nm \[" ja-kanji)
("" "\[" "Acm \[" "Acm \[" ja-kanji)
("" "\[" "nCm \[" "nCm \[" ja-kanji)
("" "\[" "I][Vm \[" "I][Vm \[" ja-kanji)
("" "\[" "Xm \[" "Xm \[" ja-kanji)
("" "\[" "m \[" "m \[" ja-kanji)
("" "\[" "XC\[m \[" "XC\[m \[" ja-kanji)
("" "\[" "ROm \[" "ROm \[" ja-kanji)
("" "\[" "JNm \[" "JNm \[" ja-kanji)
("" "\[" "A\Em \[" "A\Em \[" ja-kanji)
("" "\[" "Zm \[" "Zm \[" ja-kanji)
("" "\[" "ACm \[" "ACm \[" ja-kanji)
("" "\[" "}hm \[" "}hm \[" ja-kanji)
("" "\[" "JXm \[" "JXm \[" ja-kanji)
("" "\[" "\[S[m \[" "\[S[m \[" ja-kanji)
("" "\[" "Cgwm \[" "Cgwm \[" ja-kanji)
("" "\[" "\[Cm \[" "\[Cm \[" ja-kanji)
("" "\[" "NTm \[" "NTm \[" ja-kanji)
("" "\[" "xb\[m \[" "xb\[m \[" ja-kanji)
("" "\[" "z[m \[" "z[m \[" ja-kanji)
("" "\[" "AICm \[" "AICm \[" ja-kanji)
("" "\[" "m \[" "m \[" ja-kanji)
("" "\[" "\IEm \[" "\IEm \[" ja-kanji)
("" "\[" "nVm \[" "nVm \[" ja-kanji)
("" "\[" "INm \[" "INm \[" ja-kanji)
("" "\[" "\[O[m \[" "\[O[m \[" ja-kanji)
("" "\[" "Jlwm " "Jlwm " ja-kanji)
("" "\[" "Vm \[" "Vm \[" ja-kanji)
("" "\[" "TOm \[" "TOm \[" ja-kanji)
("" "][" "h[][m ][" "h[][m ][" ja-kanji)
("" "][" "}Xm ][" "}Xm ][" ja-kanji)
("" "][" "jNm ][" "jNm ][" ja-kanji)
("" "][" "V][m ][" "V][m ][" ja-kanji)
("" "][" "Nm ][" "Nm ][" ja-kanji)
("" "][" "INm ][" "INm ][" ja-kanji)
("" "][" "cNm ][" "cNm ][" ja-kanji)
("" "\N" "EiKXm \N" "EiKXm \N" ja-kanji)
("" "\N" "Km \N" "Km \N" ja-kanji)
("" "\N" "mbgm \N" "mbgm \N" ja-kanji)
("" "\N" "\NUm \N" "\NUm \N" ja-kanji)
("" "\N" "CLm \N" "CLm \N" ja-kanji)
("" "\N" "cJ}Gm \N" "cJ}Gm \N" ja-kanji)
("" "\N" "^om \N" "^om \N" ja-kanji)
("" "\N" "nJm \N" "nJm \N" ja-kanji)
("" "\N" "AVm \N" "AVm \N" ja-kanji)
("" "\N" "nCm \N" "nCm \N" ja-kanji)
("" "]N" "]NSm ]N" "]NSm ]N" ja-kanji)
("" "]N" "]NXm ]N" "]NXm ]N" ja-kanji)
("" "]N" "JC]Nm ]N" "JC]Nm ]N" ja-kanji)
("" "]N" "J]Nm ]N" "J]Nm ]N" ja-kanji)
("" "]N" "cdNm ]N" "cdNm ]N" ja-kanji)
("" "\c" "\cM[m \c" "\cM[m \c" ja-kanji)
("" "V[" "Lmm \f" "Lmm \f" ja-kanji)
("" "L" "\mm \" "\mm \" ja-kanji)
("" "Z" "\Em Z" "\Em Z" ja-kanji)
("" "\" "\UCm \" "\UCm \" ja-kanji)
("" "\" "}Sm \" "}Sm \" ja-kanji)
("" "\" "g[gCm \" "g[gCm \" ja-kanji)
("" "\" "\RiEm \" "\RiEm \" ja-kanji)
("" "\" "m \" "m \" ja-kanji)
("" "\" "\VNm \" "\VNm \" ja-kanji)
("" "^" "^jm ^" "^jm ^" ja-kanji)
("" "^" "IICm ^" "IICm ^" ja-kanji)
("" "^" "tgCm ^C" "tgCm ^C" ja-kanji)
("" "^" "g[^m ^" "g[^m ^" ja-kanji)
("" "^" "AUNm ^" "AUNm ^" ja-kanji)
("" "_" "_GLm _" "_GLm _" ja-kanji)
("" "_" "_Nm _" "_Nm _" ja-kanji)
("" "_" "_L[m _" "_L[m _" ja-kanji)
("" "_" "^C_m _" "^C_m _" ja-kanji)
("" "_" "Ecm _" "Ecm _" ja-kanji)
("" "_" "Lwm JWm _" "Lwm JWm _" ja-kanji)
("" "_" "tlm JWm _" "tlm JWm _" ja-kanji)
("" "_" "_GPCm _" "_GPCm _" ja-kanji)
("" "_" "A~_ucm _" "A~_ucm _" ja-kanji)
("" "_" "_KVm _" "_KVm _" ja-kanji)
("" "_" "q_m _" "q_m _" ja-kanji)
("" "^C" "J_m ^C" "J_m ^C" ja-kanji)
("" "^C" "EY^JCm ^C" "EY^JCm ^C" ja-kanji)
("" "^C" "cCm ^C" "cCm ^C" ja-kanji)
("" "^C" "^Gm ^C" "^Gm ^C" ja-kanji)
("" "^C" "^CUm ^C" "^CUm ^C" ja-kanji)
("" "^C" "Irm ^C" "Irm ^C" ja-kanji)
("" "^C" "}cm ^C" "}cm ^C" ja-kanji)
("" "^C" "i}Pm ^C" "i}Pm ^C" ja-kanji)
("" "^C" "^Chm ^C" "^Chm ^C" ja-kanji)
("" "^C" "C^_Nm ^C" "C^_Nm ^C" ja-kanji)
("" "^C" "JG_}m ^C" "JG_}m ^C" ja-kanji)
("" "^C" "A^Cm ^C" "A^Cm ^C" ja-kanji)
("" "^C" "ghRIm ^C" "ghRIm ^C" ja-kanji)
("" "^C" "nm ^C" "nm ^C" ja-kanji)
("" "^C" "m ^C" "m ^C" ja-kanji)
("" "^C" "RPm ^C" "RPm ^C" ja-kanji)
("" "^C" "tNm ^C" "tNm ^C" ja-kanji)
("" "^C" "JXm ^C" "JXm ^C" ja-kanji)
("" "^C" "V]Nm ^C" "V]Nm ^C" ja-kanji)
("" "^C" "^Czm ^C" "^Czm ^C" ja-kanji)
("" "^C" "wC^Cm ^C" "wC^Cm ^C" ja-kanji)
("" "^C" "}Y~m ^C" "}Y~m ^C" ja-kanji)
("" "`[" "TJim ^C" "TJim ^C" ja-kanji)
("" "_C" "_Cm _C" "_Cm _C" ja-kanji)
("" "_C" "_ChRm _C" "_ChRm _C" ja-kanji)
("" "_C" "IILCm _C" "IILCm _C" ja-kanji)
("" "_C" "_CC`m _C" "_CC`m _C" ja-kanji)
("" "_C" "_CS~m _C" "_CS~m _C" ja-kanji)
("" "_C" "_CCm _C" "_CCm _C" ja-kanji)
("" "I[" "gm ^J" "gm ^J" ja-kanji)
("" "^L" "^Lc{m ^L" "^Lc{m ^L" ja-kanji)
("" "^L" "L[JWm ^L" "L[JWm ^L" ja-kanji)
("" "^N" "^bL[m ^N" "^bL[m ^N" ja-kanji)
("" "^N" "^N{Nm ^N" "^N{Nm ^N" ja-kanji)
("" "^N" "W^Nm ^N" "W^Nm ^N" ja-kanji)
("" "^N" "^Nncm ^N" "^Nncm ^N" ja-kanji)
("" "^N" "TC^Nm ^N" "TC^Nm ^N" ja-kanji)
("" "^N" "JC^Nm ^N" "JC^Nm ^N" ja-kanji)
("" "^N" "Tm ^N" "Tm ^N" ja-kanji)
("" "^N" "Z^NLm ^N" "Z^NLm ^N" ja-kanji)
("" "^N" "^N}m ^N" "^N}m ^N" ja-kanji)
("" "^N" "^NWVm ^N" "^NWVm ^N" ja-kanji)
("" "^N" "h[^Nm ^N" "h[^Nm ^N" ja-kanji)
("" "_N" "jSm _N" "jSm _N" ja-kanji)
("" "_N" "V[_Nm _N" "V[_Nm _N" ja-kanji)
("" "W[" "}c^Pm ^P" "}c^Pm ^P" ja-kanji)
("" "L" "^RAQm ^R" "^RAQm ^R" ja-kanji)
("" "V[" "E~m ^R" "E~m ^R" ja-kanji)
("" "V" "^_C}m ^_" "^_C}m ^_" ja-kanji)
("@" "R[" "^^Nm R[" "^^Nm R[" ja-kanji)
("A" "^" "Zc]Nm ^_V" "Zc]Nm ^_V" ja-kanji)
("B" "^`" "\N^cm ^c" "\N^cm ^c" ja-kanji)
("C" "V" "W[jVm ^c" "W[jVm ^c" ja-kanji)
("D" "_c" "EoEm _c" "EoEm _c" ja-kanji)
("E" "_c" "kOm _c" "kOm _c" ja-kanji)
("F" "\" "z[KNm ^c~" "z[KNm ^c~" ja-kanji)
("G" "W" "^eSgm ^e" "^eSgm ^e" ja-kanji)
("H" "e" "^hcNm ^h" "^hcNm ^h" ja-kanji)
("I" "z[" "z_im ^i" "z_im ^i" ja-kanji)
("J" "RN" "^jm RN" "^jm RN" ja-kanji)
("K" "" "h[ucm ^kL" "h[ucm ^kL" ja-kanji)
("L" "Zc" "TJim ^" "TJim ^" ja-kanji)
("M" "\" "TJ_m ^" "TJ_m ^" ja-kanji)
("N" "_" "_\m _" "_\m _" ja-kanji)
("O" "^" "^lm ^" "^lm ^" ja-kanji)
("P" "^" "^Ccm ^" "^Ccm ^" ja-kanji)
("Q" "^" "iQNm ^" "iQNm ^" ja-kanji)
("R" "^" "^JCm ^" "^JCm ^" ja-kanji)
("S" "^" "jiEm ^" "jiEm ^" ja-kanji)
("T" "^" "TOm ^" "TOm ^" ja-kanji)
("U" "^" "K^m ^" "K^m ^" ja-kanji)
("V" "^" "^jV[m ^" "^jV[m ^" ja-kanji)
("W" "^" "ACm ^" "ACm ^" ja-kanji)
("X" "^" "^Um ^" "^Um ^" ja-kanji)
("Y" "^" "X~m ^" "X~m ^" ja-kanji)
("Z" "^" "~WJCm ^" "~WJCm ^" ja-kanji)
("[" "^" "nVm ^" "nVm ^" ja-kanji)
("\" "^" "q[^m ^" "q[^m ^" ja-kanji)
("]" "^" "ILem ^" "ILem ^" ja-kanji)
("^" "^" "tPm ^" "tPm ^" ja-kanji)
("_" "^" "^W[m ^" "^W[m ^" ja-kanji)
("`" "^" "^pNm ^" "^pNm ^" ja-kanji)
("a" "^" "^W[m ^" "^W[m ^" ja-kanji)
("b" "^" "L^Gm ^" "L^Gm ^" ja-kanji)
("c" "_" "_^Cm _" "_^Cm _" ja-kanji)
("d" "_" "_W[m _" "_W[m _" ja-kanji)
("e" "_" "qNm _" "qNm _" ja-kanji)
("f" "_" "^cm _" "^cm _" ja-kanji)
("g" "_" "A^^JCm _" "A^^JCm _" ja-kanji)
("h" "_" "Z_m _" "Z_m _" ja-kanji)
("i" "_" "JC_m _" "JC_m _" ja-kanji)
("j" "_" "IgRm _" "IgRm _" ja-kanji)
("k" "_" "\[_m _" "\[_m _" ja-kanji)
("l" "`" "A^Cm `" "A^Cm `" ja-kanji)
("m" "`" "Vm `" "Vm `" ja-kanji)
("n" "`" "Wm `" "Wm `" ja-kanji)
("o" "`" "m `" "m `" ja-kanji)
("p" "`" "nWm `" "nWm `" ja-kanji)
("q" "`" "`Gm `" "`Gm `" ja-kanji)
("r" "`" "CPm `" "CPm `" ja-kanji)
("s" "`" "`Jm `" "`Jm `" ja-kanji)
("t" "`" "[`m `" "[`m `" ja-kanji)
("u" "`" "INm `" "INm `" ja-kanji)
("v" "`" "C^Xm `" "C^Xm `" ja-kanji)
("w" "`" "Vm Nm N" "Vm Nm N" ja-kanji)
("x" "`" "INm `" "INm `" ja-kanji)
("y" "`" "S`\[m `" "S`\[m `" ja-kanji)
("z" "`N" "LYNm `N" "LYNm `N" ja-kanji)
("{" "`N" "`NV[m `N" "`NV[m `N" ja-kanji)
("|" "`N" "^Pm `N" "^Pm `N" ja-kanji)
("}" "`N" "`NSm `N" "`NSm `N" ja-kanji)
("~" "`N" "^NGm `N" "^NGm `N" ja-kanji)
("" "`N" "`NWm `N" "`NWm `N" ja-kanji)
("" "`c" "`cWm `c" "`cWm `c" ja-kanji)
("" "`c" "`b\Nm `c" "`b\Nm `c" ja-kanji)
("" "`" "m~mm `" "m~mm `" ja-kanji)
("" "`N" "`NVm `N" "`NVm `N" ja-kanji)
("" "`N" "cNm `N" "cNm `N" ja-kanji)
("" "`[" "iJm `[" "iJm `[" ja-kanji)
("" "`[" "iJ}m `[" "iJ}m `[" ja-kanji)
("" "`[" "E`[m `[" "E`[m `[" ja-kanji)
("" "`[" "`[Wcm `[" "`[Wcm `[" ja-kanji)
("" "`[" "`[Vcm `[" "`[Vcm `[" ja-kanji)
("" "`[" "qm `[" "qm `[" ja-kanji)
("" "`[" "nVm `[" "nVm `[" ja-kanji)
("" "`[" "\\Om `[" "\\Om `[" ja-kanji)
("" "`[" "Vm `[" "Vm `[" ja-kanji)
("" "`[" "Zb`[m `[" "Zb`[m `[" ja-kanji)
("" "`[" "Sxm `[" "Sxm `[" ja-kanji)
("" "`[" "V[`[m `[" "V[`[m `[" ja-kanji)
("" "`[" "Cmm `[" "Cmm `[" ja-kanji)
("" "`[" "`[VW[m `[" "`[VW[m `[" ja-kanji)
("" "`" "Lm CbVm `" "Lm CbVm `" ja-kanji)
("" "`" "^CPm `" "^CPm `" ja-kanji)
("" "`" "CmVVm `" "CmVVm `" ja-kanji)
("" "`" "ATm CbVm `" "ATm CbVm `" ja-kanji)
("" "`" "C`WVCm `" "C`WVCm `" ja-kanji)
("" "`" "^NGm `" "^NGm `" ja-kanji)
("" "`[" "`[hm `[" "`[hm `[" ja-kanji)
("" "`[" "LUXm `[" "LUXm `[" ja-kanji)
("" "`[" "`[Nm `[" "`[Nm `[" ja-kanji)
("" "`[" "Vxm `[" "Vxm `[" ja-kanji)
("" "`[" "`[ACm `[" "`[ACm `[" ja-kanji)
("" "`[" "C`W[m `[" "C`W[m `[" ja-kanji)
("" "`[" "e`[m `[" "e`[m `[" ja-kanji)
("" "`[" "g`[m `[" "g`[m `[" ja-kanji)
("" "`[" "gEm `[" "gEm `[" ja-kanji)
("" "`[" "nm `[" "nm `[" ja-kanji)
("" "`[" "zm `[" "zm `[" ja-kanji)
("" "`[" "gN`[m `[" "gN`[m `[" ja-kanji)
("" "`[" "RVm `[" "RVm `[" ja-kanji)
("" "`[" "Chm `[" "Chm `[" ja-kanji)
("" "`[" "mLm `[" "mLm `[" ja-kanji)
("" "`[" "ATm `[" "ATm `[" ja-kanji)
("" "`[" "VIm `[" "VIm `[" ja-kanji)
("" "`[" "t_m `[" "t_m `[" ja-kanji)
("" "`[" "}`m `[" "}`m `[" ja-kanji)
("" "`[" "iKm `[" "iKm `[" ja-kanji)
("" "`[" "LNm `[" "LNm `[" ja-kanji)
("" "`[" "tNm `[" "tNm `[" ja-kanji)
("" "`[" "_C`[m `[" "_C`[m `[" ja-kanji)
("" "`[" "R`[m `[" "R`[m `[" ja-kanji)
("" "`[" "Vxm `[" "Vxm `[" ja-kanji)
("" "`[" "}Vmm `[" "}Vmm `[" ja-kanji)
("" "`[" "RGm `[" "RGm `[" ja-kanji)
("" "`[" "gum `[" "gum `[" ja-kanji)
("" "`[" "I`[Vm `[" "I`[Vm `[" ja-kanji)
("" "`[" "iKCm `[" "iKCm `[" ja-kanji)
("" "`[" "C^_Lm `[" "C^_Lm `[" ja-kanji)
("" "`[" "gm `[" "gm `[" ja-kanji)
("" "`N" "~Rgm m `N" "~Rgm m `N" ja-kanji)
("" "`N" "nJhm `N" "nJhm `N" ja-kanji)
("" "`N" "iIm `N" "iIm `N" ja-kanji)
("" "`" "m `" "m `" ja-kanji)
("" "`" "VYm `" "VYm `" ja-kanji)
("" "`" "YVCm `" "YVCm `" ja-kanji)
("" "`" "`Mm `" "`Mm `" ja-kanji)
("" "`" "VYm `" "VYm `" ja-kanji)
("" "`" "`cm `" "`cm `" ja-kanji)
("" "V" "IIcm c" "IIcm c" ja-kanji)
("" "cC" "cCNm cC" "cCNm cC" ja-kanji)
("" "cC" "Lm VC" "Lm VC" ja-kanji)
("" "cC" "Ld`m cC" "Ld`m cC" ja-kanji)
("" "cC" "IEm cC" "IEm cC" ja-kanji)
("" "cC" "eCecm cC" "eCecm cC" ja-kanji)
("" "c[" "C^Cm c[" "C^Cm c[" ja-kanji)
("" "c[" "gIm c[" "gIm c[" ja-kanji)
("" "cJ" "nJm cJ" "nJm cJ" ja-kanji)
("" "{" "Lm cK" "Lm cK" ja-kanji)
("" "JN" "cJm JN" "cJm JN" ja-kanji)
("" "L" "cLPLm cL" "cLPLm cL" ja-kanji)
("" "f" "cN_m f" "cN_m f" ja-kanji)
("" "V" "cPmm cP" "cPmm cP" ja-kanji)
("" "c" "UNm UN" "UNm UN" ja-kanji)
("" "cW" "cWMm cW" "cWMm cW" ja-kanji)
("" "`[" "VNucm c^" "VNucm c^" ja-kanji)
("" "eC" "cdm eC" "cdm eC" ja-kanji)
("" "KN" "J^im co" "J^im co" ja-kanji)
("" "`" "Lm coL" "Lm coL" ja-kanji)
("" "JC" "cuXm JC" "cuXm JC" ja-kanji)
("" "wC" "c{m wC" "c{m wC" ja-kanji)
("" "R" "c{m R" "c{m R" ja-kanji)
("" "W" "c}m W" "c}m W" ja-kanji)
("" "`[" "Imm cM" "Imm cM" ja-kanji)
("" "\[" "cm \[" "cm \[" ja-kanji)
("" "`[" "coVm c" "coVm c" ja-kanji)
("" "`[" "cm `[" "cm `[" ja-kanji)
("" "JN" "gm c" "gm c" ja-kanji)
("" "eC" "eCVm eC" "eCVm eC" ja-kanji)
("" "eC" "qNCm eC" "qNCm eC" ja-kanji)
("" "eC" "eCVm eC" "eCVm eC" ja-kanji)
("" "eC" "^eCm eC" "^eCm eC" ja-kanji)
("" "eC" "\m eC" "\m eC" ja-kanji)
("" "eC" "eCVNm eC" "eCVNm eC" ja-kanji)
("" "eC" "VeCm eC" "VeCm eC" ja-kanji)
("" "eC" "cc~m eC" "cc~m eC" ja-kanji)
("" "eC" "T_m eC" "T_m eC" ja-kanji)
("" "eC" "eCI[m eC" "eCI[m eC" ja-kanji)
("" "eC" "\Rm eC" "\Rm eC" ja-kanji)
("" "eC" "jm eC" "jm eC" ja-kanji)
("" "eC" "z[eCm eC" "z[eCm eC" ja-kanji)
("" "eC" "Ig[gm eC" "Ig[gm eC" ja-kanji)
("" "eC" "eC[m eC" "eC[m eC" ja-kanji)
("" "eC" "eCR[m eC" "eCR[m eC" ja-kanji)
("" "eC" "~ eCXm eC" "~ eCXm eC" ja-kanji)
("" "eC" "eCL[m eC" "eCL[m eC" ja-kanji)
("" "eC" "nVSm eC" "nVSm eC" ja-kanji)
("" "eC" "~Mm eC" "~Mm eC" ja-kanji)
("" "eC" "CJm eC" "CJm eC" ja-kanji)
("" "eC" "VXwm eC" "VXwm eC" ja-kanji)
("" "eC" "zhm eC" "zhm eC" ja-kanji)
("" "eC" "V}m eC" "V}m eC" ja-kanji)
("" "eC" "tlm eC" "tlm eC" ja-kanji)
("" "eC" "eCZCm eC" "eCZCm eC" ja-kanji)
("" "eC" "ALm eC" "ALm eC" ja-kanji)
("" "eC" "qdm eC" "qdm eC" ja-kanji)
("" "eC" "eCVm eC" "eCVm eC" ja-kanji)
("@" "eC" "eC^Nm eC" "eC^Nm eC" ja-kanji)
("A" "eC" "Njm im eC" "Njm im eC" ja-kanji)
("B" "eC" "NMm eC" "NMm eC" ja-kanji)
("C" "eC" "JiGm eC" "JiGm eC" ja-kanji)
("D" "fC" "hm fC" "hm fC" ja-kanji)
("E" "eL" "cm eL" "cm eL" ja-kanji)
("F" "eL" "kLfm eL" "kLfm eL" ja-kanji)
("G" "eL" "eL ~J^m eL" "eL ~J^m eL" ja-kanji)
("H" "eL" "VYNm eL" "VYNm eL" ja-kanji)
("I" "eL" "}gm eL" "}gm eL" ja-kanji)
("J" "eL" "tGm eL" "tGm eL" ja-kanji)
("K" "eL" "eLI[m eL" "eLI[m eL" ja-kanji)
("L" "eL" "Jum eL" "Jum eL" ja-kanji)
("M" "fL" "I{m fL" "I{m fL" ja-kanji)
("N" "ec" "ecKNm ec" "ecKNm ec" ja-kanji)
("O" "ec" "ecm ec" "ecm ec" ja-kanji)
("P" "ec" "ebLm ec" "ebLm ec" ja-kanji)
("Q" "ec" "_`m ec" "_`m ec" ja-kanji)
("R" "ec" "R[ecm ec" "R[ecm ec" ja-kanji)
("S" "ec" "R[ucm ec" "R[ucm ec" ja-kanji)
("T" "e" "mm e" "mm e" ja-kanji)
("U" "e" "EYm e" "EYm e" ja-kanji)
("V" "e" "A}m e" "A}m e" ja-kanji)
("W" "e" "e{[m e" "e{[m e" ja-kanji)
("X" "e" "~Zm e" "~Zm e" ja-kanji)
("Y" "e" "\Gm e" "\Gm e" ja-kanji)
("Z" "e" "}gCm e" "}gCm e" ja-kanji)
("[" "e" "A}Cm e" "A}Cm e" ja-kanji)
("\" "e" "nm e" "nm e" ja-kanji)
("]" "e" "RKm e" "RKm e" ja-kanji)
("^" "e" "e}cm e" "e}cm e" ja-kanji)
("_" "e" "eWm e" "eWm e" ja-kanji)
("`" "f" "c^Gm f" "c^Gm f" ja-kanji)
("a" "hm" "gmm f" "gmm f" ja-kanji)
("b" "f" "hm f" "hm f" ja-kanji)
("c" "f" "^{m f" "^{m f" ja-kanji)
("d" "f" "fm f" "fm f" ja-kanji)
("e" "g" "h[ucm ETM" "h[ucm ETM" ja-kanji)
("f" "g" "nNm g" "nNm g" ja-kanji)
("g" "g" "JLlm g" "JLlm g" ja-kanji)
("h" "g" "km g" "km g" ja-kanji)
("i" "g" "Vbgm g" "Vbgm g" ja-kanji)
("j" "g" "g\m g" "g\m g" ja-kanji)
("k" "g" "gzm g" "gzm g" ja-kanji)
("l" "g" "zNgm g" "zNgm g" ja-kanji)
("m" "g" "m g" "m g" ja-kanji)
("n" "g" "^m g" "^m g" ja-kanji)
("o" "g" "m{m g" "m{m g" ja-kanji)
("p" "g" "liVJYm g" "liVJYm g" ja-kanji)
("q" "g" "JPm g" "JPm g" ja-kanji)
("r" "g" "g`[m g" "g`[m g" ja-kanji)
("s" "g" "~Rm g" "~Rm g" ja-kanji)
("t" "g" "bLm g" "bLm g" ja-kanji)
("u" "g" "gCVm g" "gCVm g" ja-kanji)
("v" "C" "~KNm g" "~KNm g" ja-kanji)
("w" "h" "cgm h" "cgm h" ja-kanji)
("x" "h" "^r^rm h" "^r^rm h" ja-kanji)
("y" "h" "c`m h" "c`m h" ja-kanji)
("z" "h" "bRm h" "bRm h" ja-kanji)
("{" "h" "CJm h" "CJm h" ja-kanji)
("|" "g[" "^Im g[" "^Im g[" ja-kanji)
("}" "g[" "g[m g[" "g[m g[" ja-kanji)
("~" "g[" "tm g[" "tm g[" ja-kanji)
("" "g[" "RIm g[" "RIm g[" ja-kanji)
("" "g[" "J^im g[" "J^im g[" ja-kanji)
("" "g[" "Jm g[" "Jm g[" ja-kanji)
("" "g[" "erg[m g[" "erg[m g[" ja-kanji)
("" "g[" "cc~m g[" "cc~m g[" ja-kanji)
("" "g[" "KCg[Mm g[" "KCg[Mm g[" ja-kanji)
("" "g[" "XMm g[" "XMm g[" ja-kanji)
("" "g[" "V}m g[" "V}m g[" ja-kanji)
("" "g[" "}wm V}" "}wm V}" ja-kanji)
("" "g[" "C^m g[" "C^m g[" ja-kanji)
("" "g[" "iQm g[" "iQm g[" ja-kanji)
("" "g[" "mm g[" "mm g[" ja-kanji)
("" "g[" "qKVm g[" "qKVm g[" ja-kanji)
("" "g[" "N_mm " "N_mm " ja-kanji)
("" "g[" "g[Rcm g[" "g[Rcm g[" ja-kanji)
("" "g[" "lm g[" "lm g[" ja-kanji)
("" "g[" "kXm g[" "kXm g[" ja-kanji)
("" "g[" "g[^Xm g[" "g[^Xm g[" ja-kanji)
("" "g[" "m g[" "m g[" ja-kanji)
("" "g[" "hg[m g[" "hg[m g[" ja-kanji)
("" "g[" "AJm g[" "AJm g[" ja-kanji)
("" "g[" "gVrm L[W" "gVrm L[W" ja-kanji)
("" "g[" "A^m g[" "A^m g[" ja-kanji)
("" "g[" "Vg[m g[" "Vg[m g[" ja-kanji)
("" "g[" "Ng[m g[" "Ng[m g[" ja-kanji)
("" "ih" "qgVCm g[" "qgVCm g[" ja-kanji)
("" "g[" "R^Gm g[" "R^Gm g[" ja-kanji)
("" "g[" "ccm g[" "ccm g[" ja-kanji)
("" "g[" "Tg[m g[" "Tg[m g[" ja-kanji)
("" "g[" "g[\cm g[" "g[\cm g[" ja-kanji)
("" "g[" "C^m g[" "C^m g[" ja-kanji)
("" "g[" "Rbg[m g[" "Rbg[m g[" ja-kanji)
("" "g[" "z[g[m g[" "z[g[m g[" ja-kanji)
("" "g[" "tWm g[" "tWm g[" ja-kanji)
("" "g[" "Ecm g[" "Ecm g[" ja-kanji)
("" "g[" "g[Vm g[" "g[Vm g[" ja-kanji)
("" "g[" "}m g[" "}m g[" ja-kanji)
("" "g[" "tm g[" "tm g[" ja-kanji)
("" "g[" "jQm g[" "jQm g[" ja-kanji)
("" "g[" "XPm g[" "XPm g[" ja-kanji)
("" "g[" "Au~m g[" "Au~m g[" ja-kanji)
("" "g[" "g[Lm g[" "g[Lm g[" ja-kanji)
("" "g[" "A^}m g[" "A^}m g[" ja-kanji)
("" "g[" "tbg[m g[" "tbg[m g[" ja-kanji)
("" "g[" "^^JEm g[" "^^JEm g[" ja-kanji)
("" "h[" "n^Nm h[" "n^Nm h[" ja-kanji)
("" "h[" "ESNm h[" "ESNm h[" ja-kanji)
("" "h[" "IiWm h[" "IiWm h[" ja-kanji)
("" "h[" "VNh[m h[" "VNh[m h[" ja-kanji)
("" "h[" "~`rNm h[" "~`rNm h[" ja-kanji)
("" "h[" "ARKm h[" "ARKm h[" ja-kanji)
("" "h[" "h[`Nm h[" "h[`Nm h[" ja-kanji)
("" "h[" "zm h[" "zm h[" ja-kanji)
("" "h[" "qg~m h[" "qg~m h[" ja-kanji)
("" "h[" "xm h[" "xm h[" ja-kanji)
("" "h[" "h[^Cm h[" "h[^Cm h[" ja-kanji)
("" "h[" "uh[m h[" "uh[m h[" ja-kanji)
("" "h[" "~`m h[" "~`m h[" ja-kanji)
("" "h[" "L]Nm h[" "L]Nm h[" ja-kanji)
("" "g[Q" "}m g[Q" "}m g[Q" ja-kanji)
("" "z[" "gm im gL" "gm im gL" ja-kanji)
("" "gN" "gNCm gN" "gNCm gN" ja-kanji)
("" "gN" "Gm gN" "Gm gN" ja-kanji)
("" "gN" "gNKm gN" "gNKm gN" ja-kanji)
("" "gN" "h[gNm gN" "h[gNm gN" ja-kanji)
("" "gN" "gNxcm gN" "gNxcm gN" ja-kanji)
("" "gN" "JgNm gN" "JgNm gN" ja-kanji)
("" "gN" "nQm gN" "nQm gN" ja-kanji)
("" "gN" "LgNm gN" "LgNm gN" ja-kanji)
("" "hN" "hNNm hN" "hNNm hN" ja-kanji)
("" "hN" "qgm hN" "qgm hN" ja-kanji)
("" "hN" "m hN" "m hN" ja-kanji)
("" "}" "g`MPm g`" "g`MPm g`" ja-kanji)
("" "V[" "g`m Lm V[" "g`m Lm V[" ja-kanji)
("" "gc" "I[gcm gc" "I[gcm gc" ja-kanji)
("" "gc" "cNm gc" "cNm gc" ja-kanji)
("" "_" "Lm gh}c" "Lm gh}c" ja-kanji)
("" "JC" "ghPm P" "ghPm P" ja-kanji)
("" "G" "gm gr" "gm gr" ja-kanji)
("" "Z" "g}m g}" "g}m g}" ja-kanji)
("" "C" "W[jVm g" "W[jVm g" ja-kanji)
("" "[" "W[jVm g" "W[jVm g" ja-kanji)
("" "ZC" "iKgm g" "iKgm g" ja-kanji)
("" "g" "ITm g" "ITm g" ja-kanji)
("" "g" "gfm g" "gfm g" ja-kanji)
("" "g" "AcCm g" "AcCm g" ja-kanji)
("" "g" "gR[m g" "gR[m g" ja-kanji)
("" "g" "Rgm g" "Rgm g" ja-kanji)
("" "g" "u^m g" "u^m g" ja-kanji)
("" "g" "g\[m g" "g\[m g" ja-kanji)
("" "g" "gL[m g" "gL[m g" ja-kanji)
("" "h" "mm h" "mm h" ja-kanji)
("" "h" "Nm h" "Nm h" ja-kanji)
("" "h" "juCm h" "juCm h" ja-kanji)
("" "i" "iNm i" "iNm i" ja-kanji)
("" "i" "_im i" "_im i" ja-kanji)
("" "iC" "E`m iC" "E`m iC" ja-kanji)
("" "T" "iKm T" "iKm T" ja-kanji)
("" "V" "[iMm iM" "[iMm iM" ja-kanji)
("" "`" "iMi^m iM" "iMi^m iM" ja-kanji)
("" "C" "i]i]m i]" "i]i]m i]" ja-kanji)
("" "i_" "i_Vm i_" "i_Vm i_" ja-kanji)
("" "ic" "icCm ic" "icCm ic" ja-kanji)
("" "J" "ixJ}m ix" "ixJ}m ix" ja-kanji)
("" "V[" "im Lm i" "im Lm i" ja-kanji)
("" "W" "iW~m i" "iW~m i" ja-kanji)
("" "W[" "im W[" "im W[" ja-kanji)
("" "ec" "A[~`m ie" "A[~`m ie" ja-kanji)
("" "i" "~i~m i" "~i~m i" ja-kanji)
("" "i" "NXm Lm i" "NXm Lm i" ja-kanji)
("" "i" "Jm i" "Jm i" ja-kanji)
("" "i" "YJVCm i" "YJVCm i" ja-kanji)
("" "W" "I}Gm iW" "I}Gm iW" ja-kanji)
("" "j[" "JX[Wm j" "JX[Wm j" ja-kanji)
("" "j" "A}m j" "A}m j" ja-kanji)
("" "j[" "VCj cJE j" "VCj cJE j" ja-kanji)
("" "j" "WCm W" "WCm W" ja-kanji)
("" "N" "JIm jIC" "JIm jIC" ja-kanji)
("" "V" "jMEm V" "jMEm V" ja-kanji)
("" "jN" "LjNm jN" "LjNm jN" ja-kanji)
("" "R[" "jWm R[" "jWm R[" ja-kanji)
("" "jW[" "qgWm jW[" "qgWm jW[" ja-kanji)
("" "j`" "j`[rm j`" "j`[rm j`" ja-kanji)
("" "j[" "``m j[" "``m j[" ja-kanji)
("" "j[" "nCm j[" "nCm j[" ja-kanji)
("@" "j" "jCm j" "jCm j" ja-kanji)
("A" "j[" "j[\m j[" "j[\m j[" ja-kanji)
("B" "L[" "VNcm j" "VNcm j" ja-kanji)
("C" "j" "}JZm j" "}JZm j" ja-kanji)
("D" "j" "jVm j" "jVm j" ja-kanji)
("E" "j" "Vmum j" "Vmum j" ja-kanji)
("F" "j" "~gm j" "~gm j" ja-kanji)
("G" "W" "km W" "km W" ja-kanji)
("H" "l" "lMm lC" "lMm lC" ja-kanji)
("I" "l" "lCm NW" "lCm NW" ja-kanji)
("J" "lC" "eClCm lC" "eClCm lC" ja-kanji)
("K" "\[" "VNcm lM" "VNcm lM" ja-kanji)
("L" "r[" "lRm r[" "lRm r[" ja-kanji)
("M" "lc" "AcCm lc" "AcCm lc" ja-kanji)
("N" "l" "gVm l" "gVm l" ja-kanji)
("O" "l" "lucm l" "lucm l" ja-kanji)
("P" "l" "lUm l" "lUm l" ja-kanji)
("Q" "l" "m l" "m l" ja-kanji)
("R" "l" "Gm l" "Gm l" ja-kanji)
("S" "l" "lom l" "lom l" ja-kanji)
("T" "iC" "iCVm m" "iCVm m" ja-kanji)
("U" "m" "\Kmm m" "\Kmm m" ja-kanji)
("V" "m" "WCm L" "WCm L" ja-kanji)
("W" "" "mnm mm RW" "mnm mm RW" ja-kanji)
("X" "m[" "tNm m[" "tNm m[" ja-kanji)
("Y" "m[" "i~m m[" "i~m m[" ja-kanji)
("Z" "m[" "RCm m[" "RCm m[" ja-kanji)
("[" "m[" "ITm m[" "ITm m[" ja-kanji)
("\" "m[" "m[Nm m[" "m[Nm m[" ja-kanji)
("]" "m[" "Ym[m m[" "Ym[m m[" ja-kanji)
("^" "m[" "E~m m[" "E~m m[" ja-kanji)
("_" "m[" "m[~m m[" "m[~m m[" ja-kanji)
("`" "V" "m]Nm V" "m]Nm V" ja-kanji)
("a" "V" "Vm m~" "Vm m~" ja-kanji)
("b" "n" "gGm n" "gGm n" ja-kanji)
("c" "n" "nANm n" "nANm n" ja-kanji)
("d" "n" "^l }Nm n" "^l }Nm n" ja-kanji)
("e" "n" "nVm n" "nVm n" ja-kanji)
("f" "n" "TCm n" "TCm n" ja-kanji)
("g" "n" "i~m n" "i~m n" ja-kanji)
("h" "n" "nocm n" "nocm n" ja-kanji)
("i" "n" "KbLm rm " "KbLm rm " ja-kanji)
("j" "n" "um n" "um n" ja-kanji)
("k" "o" "IoATm o" "IoATm o" ja-kanji)
("l" "o" "oZCm o" "oZCm o" ja-kanji)
("m" "o" "oV[m o" "oV[m o" ja-kanji)
("n" "o" "E}m o" "E}m o" ja-kanji)
("o" "nC" "nCNm nC" "nCNm nC" ja-kanji)
("p" "nC" "X^m nC" "X^m nC" ja-kanji)
("q" "nC" "IKm nC" "IKm nC" ja-kanji)
("r" "nC" "nCXCm nC" "nCXCm nC" ja-kanji)
("s" "nC" "um nC" "um nC" ja-kanji)
("t" "nC" "TJYLm nC" "TJYLm nC" ja-kanji)
("u" "nC" "Tm TJYL" "Tm TJYL" ja-kanji)
("v" "nC" "t_m nC" "t_m nC" ja-kanji)
("w" "nC" "Zm nC" "Zm nC" ja-kanji)
("x" "nC" "iC][m nC" "iC][m nC" ja-kanji)
("y" "nC" "ZpCm nC" "ZpCm nC" ja-kanji)
("z" "nC" "Nom nC" "Nom nC" ja-kanji)
("{" "oC" "ToCm oC" "ToCm oC" ja-kanji)
("|" "oC" "c`JEm oC" "c`JEm oC" ja-kanji)
("}" "oC" "oCJCm oC" "oCJCm oC" ja-kanji)
("~" "oC" "Em oC" "Em oC" ja-kanji)
("" "oC" "A}CuVm E" "A}CuVm E" ja-kanji)
("" "oC" "XXm oC" "XXm oC" ja-kanji)
("" "oC" "[oCm oC" "[oCm oC" ja-kanji)
("" "oC" "JEm oC" "JEm oC" ja-kanji)
("" "oC" "Em oC" "Em oC" ja-kanji)
("" "oC" "oCV[m oC" "oCV[m oC" ja-kanji)
("" "oC" "oCVm oC" "oCVm oC" ja-kanji)
("" "V" "nEm V" "nEm V" ja-kanji)
("" "[" "Vm nG" "Vm nG" ja-kanji)
("" "r" "nJm V[" "nJm V[" ja-kanji)
("" "V" "nMm nM" "nMm nM" ja-kanji)
("" "V[" "nim nM" "nim nM" ja-kanji)
("" "nN" "IWm nN" "IWm nN" ja-kanji)
("" "nN" "nNZCm nN" "nNZCm nN" ja-kanji)
("" "nN" "nJZm nN" "nJZm nN" ja-kanji)
("" "nN" "nNVm nN" "nNVm nN" ja-kanji)
("" "nN" "JVm nN" "JVm nN" ja-kanji)
("" "nN" "g}m nN" "g}m nN" ja-kanji)
("" "nN" "Vm nN" "Vm nN" ja-kanji)
("" "nN" "LpNm nN" "LpNm nN" ja-kanji)
("" "nN" "TPJXm nN" "TPJXm nN" ja-kanji)
("" "nN" "ZpNm nN" "ZpNm nN" ja-kanji)
("" "nN" "EXCm nN" "EXCm nN" ja-kanji)
("" "nN" "Z}m nN" "Z}m nN" ja-kanji)
("" "oN" "TXm oN" "TXm oN" ja-kanji)
("" "oN" "ToNm oN" "ToNm oN" ja-kanji)
("" "oN" "oNQLm oN" "oNQLm oN" ja-kanji)
("" "oN" "Vom oN" "Vom oN" ja-kanji)
("" "oN" "oN_Cm oN" "oN_Cm oN" ja-kanji)
("" "oN" "oNm oN" "oNm oN" ja-kanji)
("" "oN" "Mm oN" "Mm oN" ja-kanji)
("" "J" "nR_em nR" "nR_em nR" ja-kanji)
("" "\[" "nRm \[" "nRm \[" ja-kanji)
("" "RN" "^jACm nU}" "^jACm nU}" ja-kanji)
("" "`" "^x nV" "^x nV" ja-kanji)
("" "`[" "nWm `[" "nWm `[" ja-kanji)
("" "Jc" "V^nYm nY" "V^nYm nY" ja-kanji)
("" "" "Lm n[" "Lm n[" ja-kanji)
("" "}" "n^m n^" "n^m n^" ja-kanji)
("" "L" "qtm n_" "qtm n_" ja-kanji)
("" "n^" "n^Pm f" "n^Pm f" ja-kanji)
("" "n^" "nNfm n^P" "nNfm n^P" ja-kanji)
("" "n`" "JX[Wm n`" "JX[Wm n`" ja-kanji)
("" "n`" "Cncm nc" "Cncm nc" ja-kanji)
("" "nc" "nccm nc" "nccm nc" ja-kanji)
("" "nc" "nb^cm nc" "nb^cm nc" ja-kanji)
("" "nc" "nbR[m nc" "nbR[m nc" ja-kanji)
("" "nc" "J~m nc" "J~m nc" ja-kanji)
("" "oc" "Lm oc" "Lm oc" ja-kanji)
("" "oc" "c~g ocm oc" "c~g ocm oc" ja-kanji)
("" "oc" "kNm oc" "kNm oc" ja-kanji)
("" "oc" "CJ_m oc" "CJ_m oc" ja-kanji)
("" "oc" "UCocm oc" "UCocm oc" ja-kanji)
("" "L[" "gm ng" "gm ng" ja-kanji)
("" "V" "niVJm niV" "niVJm niV" ja-kanji)
("" "R[" "nim JN" "nim JN" ja-kanji)
("" "R[" "JCm n}O" "JCm n}O" ja-kanji)
("" "W" "gm nuT" "gm nuT" ja-kanji)
("" "n" "giEm n" "giEm n" ja-kanji)
("" "n" "n_m n" "n_m n" ja-kanji)
("" "n" "iJom n" "iJom n" ja-kanji)
("" "n" "\m n" "\m n" ja-kanji)
("" "n" "\Nm n" "\Nm n" ja-kanji)
("" "n" "bgm z" "bgm z" ja-kanji)
("" "n" "Epm n" "Epm n" ja-kanji)
("" "n" "}_m n" "}_m n" ja-kanji)
("" "n" "C^m n" "C^m n" ja-kanji)
("" "n" "nm n" "nm n" ja-kanji)
("" "n" "qCm n" "qCm n" ja-kanji)
("" "n" "Vbpm n" "Vbpm n" ja-kanji)
("" "n" "IJXm n" "IJXm n" ja-kanji)
("" "n" "n`[m n" "n`[m n" ja-kanji)
("" "n" "A[m n" "A[m n" ja-kanji)
("" "n" "VQm n" "VQm n" ja-kanji)
("" "n" "Cbpm n" "Cbpm n" ja-kanji)
("" "n" "nVm n" "nVm n" ja-kanji)
("" "n" "noCm n" "noCm n" ja-kanji)
("" "n" "nCm n" "nCm n" ja-kanji)
("" "n" "Pm n" "Pm n" ja-kanji)
("" "n" "YEm n" "YEm n" ja-kanji)
("" "n" "nvm n" "nvm n" ja-kanji)
("" "n" "Vm n" "Vm n" ja-kanji)
("" "o" "oJCm o" "oJCm o" ja-kanji)
("" "o" "ATom o" "ATom o" ja-kanji)
("" "o" "oO~m o" "oO~m o" ja-kanji)
("" "o" "Som o" "Som o" ja-kanji)
("" "o" "oWNm o" "oWNm o" ja-kanji)
("" "o" "oVm o" "oVm o" ja-kanji)
("" "o" "om o" "om o" ja-kanji)
("" "q" "q]Nm q" "q]Nm q" ja-kanji)
("" "q" "CVCm q" "CVCm q" ja-kanji)
("" "q" "qeCm q" "qeCm q" ja-kanji)
("" "q" "ILTLm q" "ILTLm q" ja-kanji)
("" "q" "qTVm q" "qTVm q" ja-kanji)
("" "q" "Jm q" "Jm q" ja-kanji)
("" "q" "JiVCm q" "JiVCm q" ja-kanji)
("" "q" "grm q" "grm q" ja-kanji)
("" "q" "qnm q" "qnm q" ja-kanji)
("" "q" "q[Gm q" "q[Gm q" ja-kanji)
("" "q" "AK Am q" "AK Am q" ja-kanji)
("" "q" "Nxm q" "Nxm q" ja-kanji)
("" "q" "qj[Lm q" "qj[Lm q" ja-kanji)
("" "q" "cJm q" "cJm q" ja-kanji)
("" "q" "Jm q" "Jm q" ja-kanji)
("" "q" "CVu~m q" "CVu~m q" ja-kanji)
("" "q" "qm q" "qm q" ja-kanji)
("" "q" "qCm q" "qCm q" ja-kanji)
("" "q" "qm q" "qm q" ja-kanji)
("" "q" "RGm q" "RGm q" ja-kanji)
("" "q" "R[m q" "R[m q" ja-kanji)
("" "q" "q{[m q" "q{[m q" ja-kanji)
("" "q" "cCXm q" "cCXm q" ja-kanji)
("" "q" "TPm q" "TPm q" ja-kanji)
("" "q" "qW[VLm q" "qW[VLm q" ja-kanji)
("" "q" "gum q" "gum q" ja-kanji)
("" "q" "gCm q" "gCm q" ja-kanji)
("" "n" "qm n" "qm n" ja-kanji)
("" "r" "\iGm r" "\iGm r" ja-kanji)
("" "r" "Im r" "Im r" ja-kanji)
("" "r" "rZCucm r" "rZCucm r" ja-kanji)
("" "r" "rm Lm r" "rm Lm r" ja-kanji)
("" "r" "rVem r" "rVem r" ja-kanji)
("" "r" "rRm r" "rRm r" ja-kanji)
("" "r" "}m r" "}m r" ja-kanji)
("" "r" "EcNVCm r" "EcNVCm r" ja-kanji)
("@" "r" "nim r" "nim r" ja-kanji)
("A" "V[" "Lm qCM" "Lm qCM" ja-kanji)
("B" "nC" "VNucm qG" "VNucm qG" ja-kanji)
("C" "qL" "CbsLm qL" "CbsLm qL" ja-kanji)
("D" "qL" "AVm qL" "AVm qL" ja-kanji)
("E" "V" "N`qQm qQ" "N`qQm qQ" ja-kanji)
("F" "qR" "qRlm qR" "qRlm qR" ja-kanji)
("G" "Vc" "qUm Vc" "qUm Vc" ja-kanji)
("H" "[" "qVK^m qV" "qVK^m qV" ja-kanji)
("I" "`[" "Efm qW" "Efm qW" ja-kanji)
("J" "qc" "^XPm qc" "^XPm qc" ja-kanji)
("K" "qc" "JiYm qc" "JiYm qc" ja-kanji)
("L" "qc" "qbZCm qc" "qbZCm qc" ja-kanji)
("M" "qc" "tfm qc" "tfm qc" ja-kanji)
("N" "qc" "qbpNm qc" "qbpNm qc" ja-kanji)
("O" "JC" "Lm qm L" "Lm qm L" ja-kanji)
("P" "L" "qm L" "qm L" ja-kanji)
("Q" "G" "TCGm q" "TCGm q" ja-kanji)
("R" "`[" "Irqm q" "Irqm q" ja-kanji)
("S" "qN" "JX[Wm qN" "JX[Wm qN" ja-kanji)
("T" "r[" "A}m r[" "A}m r[" ja-kanji)
("U" "q[" "^m q[" "^m q[" ja-kanji)
("V" "q[" "}_m q[" "}_m q[" ja-kanji)
("W" "q[" "q[VLm q[" "q[VLm q[" ja-kanji)
("X" "q[" "RIm q[" "RIm q[" ja-kanji)
("Y" "q[" "^_Em q[" "^_Em q[" ja-kanji)
("Z" "q[" "q[^m q[" "q[^m q[" ja-kanji)
("[" "q[" "g[q[m q[" "g[q[m q[" ja-kanji)
("\" "q[" "AXm q[" "AXm q[" ja-kanji)
("]" "q[" "q[om q[" "q[om q[" ja-kanji)
("^" "q[" "h[ucm q[" "h[ucm q[" ja-kanji)
("_" "r[" "I^}m r[" "I^}m r[" ja-kanji)
("`" "r[" "GKNm r[" "GKNm r[" ja-kanji)
("a" "r[" "}Cm r[" "}Cm r[" ja-kanji)
("b" "r[" "WJm r[" "WJm r[" ja-kanji)
("c" "r[" "iGm r[" "iGm r[" ja-kanji)
("d" "r[" "CJm r[" "CJm r[" ja-kanji)
("e" "r[" "m g r[" "m g r[" ja-kanji)
("f" "T" "mrm q" "mrm q" ja-kanji)
("g" "`c" "Vm q" "Vm q" ja-kanji)
("h" "L" "TJim q" "TJim q" ja-kanji)
("i" "q" "Vim q" "Vim q" ja-kanji)
("j" "q" "Lj XMm AL" "Lj XMm AL" ja-kanji)
("k" "q" "t~j um q" "t~j um q" ja-kanji)
("l" "q" "n}m q" "n}m q" ja-kanji)
("m" "q" "qVm q" "qVm q" ja-kanji)
("n" "q" "}YVCm q" "}YVCm q" ja-kanji)
("o" "q" "qLNm q" "qLNm q" ja-kanji)
("p" "q" "qpm q" "qpm q" ja-kanji)
("q" "r" "rV[m r" "rV[m r" ja-kanji)
("r" "r" "KXrm r" "KXrm r" ja-kanji)
("s" "t" "tJm[m t" "tJm[m t" ja-kanji)
("t" "t" "cPm t" "cPm t" ja-kanji)
("u" "t" "ngom t" "ngom t" ja-kanji)
("v" "t" "Ibgm t" "Ibgm t" ja-kanji)
("w" "t" "tWm t" "tWm t" ja-kanji)
("x" "t" "gm t" "gm t" ja-kanji)
("y" "t" "gm ]NW" "gm ]NW" ja-kanji)
("z" "t" "kmm t" "kmm t" ja-kanji)
("{" "t" "ZCtm t" "ZCtm t" ja-kanji)
("|" "t" "RCm t" "RCm t" ja-kanji)
("}" "t" "tWm t" "tWm t" ja-kanji)
("~" "t" "VNm t" "VNm t" ja-kanji)
("" "t" "Imm t" "Imm t" ja-kanji)
("" "t" "tc[m t" "tc[m t" ja-kanji)
("" "t" "ENm t" "ENm t" ja-kanji)
("" "t" "``m t" "``m t" ja-kanji)
("" "t" "tS[m t" "tS[m t" ja-kanji)
("" "t" "NTm t" "NTm t" ja-kanji)
("" "t" "qtm t" "qtm t" ja-kanji)
("" "t" "t[m t" "t[m t" ja-kanji)
("" "t" "KNtm t" "KNtm t" ja-kanji)
("" "t" "}Pm t" "}Pm t" ja-kanji)
("" "t" "evm t" "evm t" ja-kanji)
("" "t" "INm t" "INm t" ja-kanji)
("" "t" "MtPm t" "MtPm t" ja-kanji)
("" "t" "RUgwm t" "RUgwm t" ja-kanji)
("" "u" "Aihm u" "Aihm u" ja-kanji)
("" "u" "u[m u" "u[m u" ja-kanji)
("" "u" "uVm u" "uVm u" ja-kanji)
("" "u" "}Em u" "}Em u" ja-kanji)
("" "u" "uh[m u" "uh[m u" ja-kanji)
("" "u" "Tm u\m u" "Tm u\m u" ja-kanji)
("" "u" "wm u" "wm u" ja-kanji)
("" "t[" "t[g[m t[" "t[g[m t[" ja-kanji)
("" "t[" "Lm JGf" "Lm JGf" ja-kanji)
("" "t[" "J[m t[" "J[m t[" ja-kanji)
("" "V[" "l tNm tN" "l tNm tN" ja-kanji)
("" "" "VNucm tL" "VNucm tL" ja-kanji)
("" "tN" "tZm tN" "tZm tN" ja-kanji)
("" "tN" "tNM[m tN" "tNM[m tN" ja-kanji)
("" "tN" "tbJcm tN" "tbJcm tN" ja-kanji)
("" "tN" "nom tN" "nom tN" ja-kanji)
("" "tN" "[tNm tN" "[tNm tN" ja-kanji)
("" "tN" "R[tNm tN" "R[tNm tN" ja-kanji)
("" "tN" "nm tN" "nm tN" ja-kanji)
("" "tN" "tNUcm tN" "tNUcm tN" ja-kanji)
("" "tN" "IIEm tN" "IIEm tN" ja-kanji)
("" "G" "t`m G" "t`m G" ja-kanji)
("" "tc" "tb\m tc" "tb\m tc" ja-kanji)
("" "tc" "nEm tc" "nEm tc" ja-kanji)
("" "tc" "Nm tc" "Nm tc" ja-kanji)
("" "uc" "zgPm uc" "zgPm uc" ja-kanji)
("" "uc" "mm uc" "mm uc" ja-kanji)
("" "t" "TJim ti" "TJim ti" ja-kanji)
("" "u" "Pm u" "Pm u" ja-kanji)
("" "t" "Zbvm t" "Zbvm t" ja-kanji)
("" "t" "tNm t" "tNm t" ja-kanji)
("" "t" "Rtm t" "Rtm t" ja-kanji)
("" "t" "CLhIm t" "CLhIm t" ja-kanji)
("" "t" "t\[m t" "t\[m t" ja-kanji)
("" "t" "tVm t" "tVm t" ja-kanji)
("" "t" "tEm t" "tEm t" ja-kanji)
("" "t" "Rim t" "Rim t" ja-kanji)
("" "t" "N\m t" "N\m t" ja-kanji)
("" "t" "}Mm t" "}Mm t" ja-kanji)
("" "t" "tCLm t" "tCLm t" ja-kanji)
("" "u" "t~m u" "t~m u" ja-kanji)
("" "u" "LNm u" "LNm u" ja-kanji)
("" "wC" "qmGm wC" "qmGm wC" ja-kanji)
("" "wC" "AZm wC" "AZm wC" ja-kanji)
("" "wC" "wC^Cm wC" "wC^Cm wC" ja-kanji)
("" "wC" "JLlm wC" "JLlm wC" ja-kanji)
("" "wC" "VwCm wC" "VwCm wC" ja-kanji)
("" "wC" "^Cm wC" "^Cm wC" ja-kanji)
("" "wC" "wCKCm wC" "wCKCm wC" ja-kanji)
("" "wC" "gbem G" "gbem G" ja-kanji)
("" "wC" "i~m wC" "i~m wC" ja-kanji)
("" "wC" "um wC" "um wC" ja-kanji)
("" "wC" "gWm wC" "gWm wC" ja-kanji)
("" "wC" "wCJm wC" "wCJm wC" ja-kanji)
("" "xC" "Rm xC" "Rm xC" ja-kanji)
("" "y[W" "yCWm Pc" "yCWm Pc" ja-kanji)
("" "wL" "wL`m wL" "wL`m wL" ja-kanji)
("" "wL" "Jxm wL" "Jxm wL" ja-kanji)
("" "wL" "NZm wL" "NZm wL" ja-kanji)
("" "wL" "AICm wL" "AICm wL" ja-kanji)
("" "xc" "Jm xc" "Jm xc" ja-kanji)
("" "xc" "C`xcm xc" "C`xcm xc" ja-kanji)
("" "xc" "PCxcm xc" "PCxcm xc" ja-kanji)
("" "wC" "JUVm wC" "JUVm wC" ja-kanji)
("" "w" "J^m w" "J^m w" ja-kanji)
("" "w" "Jm w" "Jm w" ja-kanji)
("" "w" "J^m w" "J^m w" ja-kanji)
("" "w" "wV[m w" "wV[m w" ja-kanji)
("" "w" "wV[m w" "wV[m w" ja-kanji)
("" "w" "V[wm w" "V[wm w" ja-kanji)
("" "w" "JGXm w" "JGXm w" ja-kanji)
("" "w" "wm w" "wm w" ja-kanji)
("" "x" "xm x" "xm x" ja-kanji)
("" "x" "xL[m x" "xL[m x" ja-kanji)
("" "x" "uxm x" "uxm x" ja-kanji)
("" "x" "xPCm x" "xPCm x" ja-kanji)
("" "x" "`m x" "`m x" ja-kanji)
("" "z" "^cm z" "^cm z" ja-kanji)
("" "z" "z\[m z" "z\[m z" ja-kanji)
("" "z" "zm ZCW" "zm ZCW" ja-kanji)
("" "z" "n^Pm z" "n^Pm z" ja-kanji)
("" "z" "gGm z" "gGm z" ja-kanji)
("" "z" "ANm z" "ANm z" ja-kanji)
("" "z" "nWm z" "nWm z" ja-kanji)
("" "z" "IMiEm z" "IMiEm z" ja-kanji)
("" "z" "zTm z" "zTm z" ja-kanji)
("" "XC" "Cizm z" "Cizm z" ja-kanji)
("" "{" "cmm {" "cmm {" ja-kanji)
("" "{" "nJm {" "nJm {" ja-kanji)
("" "{" "V^Em {" "V^Em {" ja-kanji)
("" "{" "c`mGm {" "c`mGm {" ja-kanji)
("" "{" "Nm {" "Nm {" ja-kanji)
("" "{" "nnm {" "nnm {" ja-kanji)
("" "{" "{Lm {" "{Lm {" ja-kanji)
("" "{" "{Tcm {" "{Tcm {" ja-kanji)
("" "z[" "iEm z[" "iEm z[" ja-kanji)
("" "z[" "z[L[m z[" "z[L[m z[" ja-kanji)
("" "z[" "ccm z[" "ccm z[" ja-kanji)
("" "z[" "IJm z[" "IJm z[" ja-kanji)
("" "z[" "z[h[m z[" "z[h[m z[" ja-kanji)
("" "z[" "^e}cm z[" "^e}cm z[" ja-kanji)
("" "z[" "^Jm z[" "^Jm z[" ja-kanji)
("" "z[" "~lm z[" "~lm z[" ja-kanji)
("" "z[" "~lm ZCW" "~lm ZCW" ja-kanji)
("" "z[" "NYm z[" "NYm z[" ja-kanji)
("" "z[" "z[`[m z[" "z[`[m z[" ja-kanji)
("" "z[" "C_Nm z[" "C_Nm z[" ja-kanji)
("" "z[" "TTQm z[" "TTQm z[" ja-kanji)
("" "z[" "nicm z[" "nicm z[" ja-kanji)
("" "z[" "J^m z[" "J^m z[" ja-kanji)
("" "z[" "g_`m z[" "g_`m z[" ja-kanji)
("@" "z[" "z[cm z[" "z[cm z[" ja-kanji)
("A" "z[" "Am z[" "Am z[" ja-kanji)
("B" "z[" "Jb|[m z[" "Jb|[m z[" ja-kanji)
("C" "z[" "^Cz[m z[" "^Cz[m z[" ja-kanji)
("D" "z[" "kEm z[" "kEm z[" ja-kanji)
("E" "z[" "h[z[m z[" "h[z[m z[" ja-kanji)
("F" "z[" "JoVCm z[" "JoVCm z[" ja-kanji)
("G" "z[" "Gm z[" "Gm z[" ja-kanji)
("H" "z[" "z[Cm z[" "z[Cm z[" ja-kanji)
("I" "z[" "R`[m n`" "R`[m n`" ja-kanji)
("J" "z[" "zm z[" "zm z[" ja-kanji)
("K" "z[" "IgYm z[" "IgYm z[" ja-kanji)
("L" "z[" "^Jm z[" "^Jm z[" ja-kanji)
("M" "z[" "Njm z[" "Njm z[" ja-kanji)
("N" "z[" "zRTLm z[" "zRTLm z[" ja-kanji)
("O" "z[" "ALm z[" "ALm z[" ja-kanji)
("P" "z[" "z[I[m z[" "z[I[m z[" ja-kanji)
("Q" "z[" "IIgm z[" "IIgm z[" ja-kanji)
("R" "{[" "g{VCm {[" "g{VCm {[" ja-kanji)
("S" "{[" "iCm {[" "iCm {[" ja-kanji)
("T" "{[" "J^m {[" "J^m {[" ja-kanji)
("U" "{[" "JC{[m {[" "JC{[m {[" ja-kanji)
("V" "{[" "{b`m {[" "{b`m {[" ja-kanji)
("W" "{[" "T}^Qm {[" "T}^Qm {[" ja-kanji)
("X" "{[" "{[Vm {[" "{[Vm {[" ja-kanji)
("Y" "{[" "Xm {[" "Xm {[" ja-kanji)
("Z" "{[" "C\KVCm {[" "C\KVCm {[" ja-kanji)
("[" "{[" "tTm {[" "tTm {[" ja-kanji)
("\" "{[" "Aom {[" "Aom {[" ja-kanji)
("]" "{[" "m]m {[" "m]m {[" ja-kanji)
("^" "{[" "ijKVm {[" "ijKVm {[" ja-kanji)
("_" "{[" "R{[m {[" "R{[m {[" ja-kanji)
("`" "{[" "IJXm {[" "IJXm {[" ja-kanji)
("a" "{[" "cOm {[" "cOm {[" ja-kanji)
("b" "{[" "V{[m {[" "V{[m {[" ja-kanji)
("c" "{[" "tNm {[" "tNm {[" ja-kanji)
("d" "{[" "nJm {[" "nJm {[" ja-kanji)
("e" "{[" "[{[m {[" "[{[m {[" ja-kanji)
("f" "{[" "{[GLm {[" "{[GLm {[" ja-kanji)
("g" "{[" "zRm {[" "zRm {[" ja-kanji)
("h" "{[" "tZOm {[" "tZOm {[" ja-kanji)
("i" "oC" "CkK zG" "CkK zG" ja-kanji)
("j" "L[" "zby^m zI" "zby^m zI" ja-kanji)
("k" "zN" "L^m {N" "L^m {N" ja-kanji)
("l" "{N" "^Vm {N" "^Vm {N" ja-kanji)
("m" "{N" "EiCm {N" "EiCm {N" ja-kanji)
("n" "{N" "X~m {N" "X~m {N" ja-kanji)
("o" "{N" "{Ncm {N" "{Ncm {N" ja-kanji)
("p" "{N" "{Ngcm {N" "{Ngcm {N" ja-kanji)
("q" "{N" "}Lm {N" "}Lm {N" ja-kanji)
("r" "{N" "c}VCm {N" "c}VCm {N" ja-kanji)
("s" "{N" "iSJi{N" "iSJi{N" ja-kanji)
("t" "R[" "[tNm {^" "[tNm {^" ja-kanji)
("u" "{c" "{bpcm {c" "{bpcm {c" ja-kanji)
("v" "{c" "{cNm {c" "{cNm {c" ja-kanji)
("w" "^C" "zghm ^C" "zghm ^C" ja-kanji)
("x" "Nc" "zm z" "zm z" ja-kanji)
("y" "R[" "zoVm z" "zoVm z" ja-kanji)
("z" "z" "z\[m z" "z\[m z" ja-kanji)
("{" "z" "gm z" "gm z" ja-kanji)
("|" "z" "qKGm z" "qKGm z" ja-kanji)
("}" "{" "wC{m {" "wC{m {" ja-kanji)
("~" "{" "I{m {" "I{m {" ja-kanji)
("" "}" "}Tcm }" "}Tcm }" ja-kanji)
("" "}" "~KNm }" "~KNm }" ja-kanji)
("" "}" "}z[m }" "}z[m }" ja-kanji)
("" "}" "ATm }" "ATm }" ja-kanji)
("" "}C" "Em }C" "Em }C" ja-kanji)
("" "}C" "C[gm }C" "C[gm }C" ja-kanji)
("" "}C" "NCm }C" "NCm }C" ja-kanji)
("" "}C" "C`}Cm }C" "C`}Cm }C" ja-kanji)
("" "}C" "}Cj`m }C" "}Cj`m }C" ja-kanji)
("" "}C" "iKTm }C" "iKTm }C" ja-kanji)
("" "e" "Lm }Lm L[W" "Lm }Lm L[W" ja-kanji)
("" "}N" "J[em }N" "J[em }N" ja-kanji)
("" "}N" "R}Nm }N" "R}Nm }N" ja-kanji)
("" "`" "}Nm `" "}Nm `" ja-kanji)
("" "[" "TJim }O" "TJim }O" ja-kanji)
("" "ZC" "}Tm }T" "}Tm }T" ja-kanji)
("" "\" "TJim }X" "TJim }X" ja-kanji)
("" "V[" "Lm }Xm ]NW" "Lm }Xm ]NW" ja-kanji)
("" "GL" "}^m GL" "}^m GL" ja-kanji)
("" "e" "t^}^m }^" "t^}^m }^" ja-kanji)
("" "[" "}^m [" "}^m [" ja-kanji)
("" "}c" "}bTcm }c" "}bTcm }c" ja-kanji)
("" "}c" "XGm }c" "XGm }c" ja-kanji)
("" "}c" "q}cm }c" "q}cm }c" ja-kanji)
("" "Lc" "C^m }f" "C^m }f" ja-kanji)
("" "W" "ZcrSm }}" "ZcrSm }}" ja-kanji)
("" "P" "PVm }" "PVm }" ja-kanji)
("" "}" "qg}m }" "qg}m }" ja-kanji)
("" "}" "JX[Wm }" "JX[Wm }" ja-kanji)
("" "}" "W}m }" "W}m }" ja-kanji)
("" "}" "~cm }" "~cm }" ja-kanji)
("" "}" "}Km }" "}Km }" ja-kanji)
("" "}" "cNTm }" "cNTm }" ja-kanji)
("" "~" "AWm ~" "AWm ~" ja-kanji)
("" "~" "~Cm ~" "~Cm ~" ja-kanji)
("" "~" "~Nm ~" "~Nm ~" ja-kanji)
("" "V" "W[jVm ~" "W[jVm ~" ja-kanji)
("" "L" "TJi g ~" "TJi g ~" ja-kanji)
("" "R[" "~TLm R[" "~TLm R[" ja-kanji)
("" "~c" "q~cm ~c" "q~cm ~c" ja-kanji)
("" "~c" "~co`m ~c" "~co`m ~c" ja-kanji)
("" "\[" "~igm \[" "~igm \[" ja-kanji)
("" "T" "~mVm ~m" "~mVm ~m" ja-kanji)
("" "W" "~mm l" "~mm l" ja-kanji)
("" "~N" "~NnNm ~N" "~NnNm ~N" ja-kanji)
("" "~[" "~[Am ~[" "~[Am ~[" ja-kanji)
("" "~[g" "iKTm ~" "iKTm ~" ja-kanji)
("" "~" "^~m ~" "^~m ~" ja-kanji)
("" "~" "lm ~" "lm ~" ja-kanji)
("" "" "cgm " "cgm " ja-kanji)
("" "" "m " "m " ja-kanji)
("" "" "iCm " "iCm " ja-kanji)
("" "" "II^Vm " "II^Vm " ja-kanji)
("" "" "zRm " "zRm " ja-kanji)
("" "" "Lm " "Lm " ja-kanji)
("" "" "I[m " "I[m " ja-kanji)
("" "[" "Lm N" "Lm N" ja-kanji)
("" "R" "niRm R" "niRm R" ja-kanji)
("" "X" "Xm W[" "Xm W[" ja-kanji)
("" "C" "~[m C" "~[m C" ja-kanji)
("" "C" "i}Gm i" "i}Gm i" ja-kanji)
("" "C" "Cm`m C" "Cm`m C" ja-kanji)
("" "C" "AJCm C" "AJCm C" ja-kanji)
("" "C" "Cm C" "Cm C" ja-kanji)
("" "C" "}Em C" "}Em C" ja-kanji)
("" "C" "C`m C" "C`m C" ja-kanji)
("" "C" "iNm C" "iNm C" ja-kanji)
("" "C" "ICCm C" "ICCm C" ja-kanji)
("" "q" "EVwm X" "EVwm X" ja-kanji)
("" "c" "zrm c" "zrm c" ja-kanji)
("" "" "}kKm " "}kKm " ja-kanji)
("" "" "^m Lm " "^m Lm " ja-kanji)
("" "" "^m " "^m " ja-kanji)
("" "" "[m " "[m " ja-kanji)
("" "" "Iem " "Iem " ja-kanji)
("" "" "Ehm " "Ehm " ja-kanji)
("" "" "TNm " "TNm " ja-kanji)
("" "" "z[m " "z[m " ja-kanji)
("" "" "VQm " "VQm " ja-kanji)
("" "[" "{[Qm {[" "{[Qm {[" ja-kanji)
("" "[" "[Vm [" "[Vm [" ja-kanji)
("" "[" "Pm [" "Pm [" ja-kanji)
("" "[" "^P_PVCm [" "^P_PVCm [" ja-kanji)
("" "[" "[Wm [" "[Wm [" ja-kanji)
("" "[" "A~m [" "A~m [" ja-kanji)
("" "[" "V[[m [" "V[[m [" ja-kanji)
("" "[" "[Rm [" "[Rm [" ja-kanji)
("" "`" "[Pm `" "[Pm `" ja-kanji)
("" "N" "Lm N" "Lm N" ja-kanji)
("" "N" "_}m N" "_}m N" ja-kanji)
("" "N" "m N" "m N" ja-kanji)
("" "N" "WCm N" "WCm N" ja-kanji)
("" "`" "`m `" "`m `" ja-kanji)
("" "wC" "^xm `" "^xm `" ja-kanji)
("" "[" "bgm [" "bgm [" ja-kanji)
("" "C" "hm C" "hm C" ja-kanji)
("" "W" "~Km ~" "~Km ~" ja-kanji)
("" "ZC" "Em ZC" "Em ZC" ja-kanji)
("" "" "gEm " "gEm " ja-kanji)
("" "" "_Gm " "_Gm " ja-kanji)
("" "" "Vm " "Vm " ja-kanji)
("" "" "Jhm " "Jhm " ja-kanji)
("" "" "ITm " "ITm " ja-kanji)
("" "" "im " "im " ja-kanji)
("" "" "Lm " "Lm " ja-kanji)
("" "" "m " "m " ja-kanji)
("" "" "WWCm " "WWCm " ja-kanji)
("" "" "}^CRNm " "}^CRNm " ja-kanji)
("" "" "mnm m" "mnm m" ja-kanji)
("" "" "WE}m " "WE}m " ja-kanji)
("" "V" "~m " "~m " ja-kanji)
("" "N" "NoCm N" "NoCm N" ja-kanji)
("" "N" "NVm GL" "NVm GL" ja-kanji)
("" "N" "N\Nm N" "N\Nm N" ja-kanji)
("" "N" "NXm N" "NXm N" ja-kanji)
("" "N" "zNm N" "zNm N" ja-kanji)
("" "N" "Ihm N" "Ihm N" ja-kanji)
("" "ZC" "XNjm ZC" "XNjm ZC" ja-kanji)
("" "[" "iMm [" "iMm [" ja-kanji)
("" "\[" "uCVm u" "uCVm u" ja-kanji)
("" "" "RNWm " "RNWm " ja-kanji)
("" "" "^mVCm " "^mVCm " ja-kanji)
("" "" "VJm " "VJm " ja-kanji)
("" "" "Aum " "Aum " ja-kanji)
("" "" "CXm " "CXm " ja-kanji)
("@" "" "TgXm " "TgXm " ja-kanji)
("A" "" "Vcm " "Vcm " ja-kanji)
("B" "C" "CCcm C" "CCcm C" ja-kanji)
("C" "[" "WCm [" "WCm [" ja-kanji)
("D" "[" "TVCm [" "TVCm [" ja-kanji)
("E" "[" "CTm [" "CTm [" ja-kanji)
("F" "[" "gm [" "gm [" ja-kanji)
("G" "[" "Xm [" "Xm [" ja-kanji)
("H" "[" "[Cm [" "[Cm [" ja-kanji)
("I" "[" "nJm [" "nJm [" ja-kanji)
("J" "[" "ECm [" "ECm [" ja-kanji)
("K" "[" "IWMm [" "IWMm [" ja-kanji)
("L" "[" "Am [" "Am [" ja-kanji)
("M" "[" "Lm Y" "Lm Y" ja-kanji)
("N" "[" "uVK [m N" "uVK [m N" ja-kanji)
("O" "[" "Nm [" "Nm [" ja-kanji)
("P" "[" "[m [" "[m [" ja-kanji)
("Q" "[" "nJm [" "nJm [" ja-kanji)
("R" "[" "[m [" "[m [" ja-kanji)
("S" "[" "^XPm [" "^XPm [" ja-kanji)
("T" "[" "[tNm [" "[tNm [" ja-kanji)
("U" "[" "T\Em [" "T\Em [" ja-kanji)
("V" "[" "A\um [" "A\um [" ja-kanji)
("W" "[" "m [" "m [" ja-kanji)
("X" "[" "[rm [" "[rm [" ja-kanji)
("Y" "[" "IXm [" "IXm [" ja-kanji)
("Z" "[" "gPm [" "gPm [" ja-kanji)
("[" "ZL" "[K^m [" "[K^m [" ja-kanji)
("\" "" "\[m " "\[m " ja-kanji)
("]" "" "A}m " "A}m " ja-kanji)
("^" "" "A^Gm " "A^Gm " ja-kanji)
("_" "" "z}m " "z}m " ja-kanji)
("`" "" "JcOm " "JcOm " ja-kanji)
("a" "" "AYPm " "AYPm " ja-kanji)
("b" "[" "gEm [" "gEm [" ja-kanji)
("c" "[" "ITiCm [" "ITiCm [" ja-kanji)
("d" "[" "AVCm [" "AVCm [" ja-kanji)
("e" "[" "iC[m [" "iC[m [" ja-kanji)
("f" "[" "{[m [" "{[m [" ja-kanji)
("g" "[" "AQm [" "AQm [" ja-kanji)
("h" "[" "m [" "m [" ja-kanji)
("i" "[" "[cm [" "[cm [" ja-kanji)
("j" "[" "[rm [" "[rm [" ja-kanji)
("k" "[" "[Wm [" "[Wm [" ja-kanji)
("l" "T}" "T}m [" "T}m [" ja-kanji)
("m" "[" "ZC[m [" "ZC[m [" ja-kanji)
("n" "[" "gPm [" "gPm [" ja-kanji)
("o" "[" "[Km [" "[Km [" ja-kanji)
("p" "[" "`Cm [" "`Cm [" ja-kanji)
("q" "[" "J}m [" "J}m [" ja-kanji)
("r" "[" "qcWm [" "qcWm [" ja-kanji)
("s" "[" "JKNm [" "JKNm [" ja-kanji)
("t" "[" "nm [" "nm [" ja-kanji)
("u" "[" "t[m [" "t[m [" ja-kanji)
("v" "[" "Cm [" "Cm [" ja-kanji)
("w" "[" "E^Cm [" "E^Cm [" ja-kanji)
("x" "[" "Ihm [" "Ihm [" ja-kanji)
("y" "[" "nJm [" "nJm [" ja-kanji)
("z" "[" "^C[m [" "^C[m [" ja-kanji)
("{" "[" "ViEm [" "ViEm [" ja-kanji)
("|" "N" "RRcLm N" "RRcLm N" ja-kanji)
("}" "N" "ITGm N" "ITGm N" ja-kanji)
("~" "N" "zbXm N" "zbXm N" ja-kanji)
("" "N" "RGm N" "RGm N" ja-kanji)
("" "N" "Arm N" "Arm N" ja-kanji)
("" "N" "NWcm N" "NWcm N" ja-kanji)
("" "N" "coTm N" "coTm N" ja-kanji)
("" "f" "hKm h" "hKm h" ja-kanji)
("" "" "V[m " "V[m " ja-kanji)
("" "" "lWm " "lWm " ja-kanji)
("" "" "n_Jm " "n_Jm " ja-kanji)
("" "C" "L^m C" "L^m C" ja-kanji)
("" "C" "AJUm C" "AJUm C" ja-kanji)
("" "C" "^mm C" "^mm C" ja-kanji)
("" "C" "J~im C" "J~im C" ja-kanji)
("" "N" "N[m N" "N[m N" ja-kanji)
("" "N" "Jm N" "Jm N" ja-kanji)
("" "N" "I`m N" "I`m N" ja-kanji)
("" "N" "Nm[m N" "Nm[m N" ja-kanji)
("" "" "~_m " "~_m " ja-kanji)
("" "" "^}Sm " "^}Sm " ja-kanji)
("" "" "AVm " "AVm " ja-kanji)
("" "" "TGMm " "TGMm " ja-kanji)
("" "" "nm " "nm " ja-kanji)
("" "" "ACm " "ACm " ja-kanji)
("" "" "nim " "nim " ja-kanji)
("" "" "~m " "~m " ja-kanji)
("" "" "Vm " "Vm " ja-kanji)
("" "" "Njm " "Njm " ja-kanji)
("" "" "LVm " "LVm " ja-kanji)
("" "" "N_mm X" "N_mm X" ja-kanji)
("" "" "N_mm iV" "N_mm iV" ja-kanji)
("" "" "ZCm " "ZCm " ja-kanji)
("" "" "W[m " "W[m " ja-kanji)
("" "" "Qm " "Qm " ja-kanji)
("" "" "Em " "Em " ja-kanji)
("" "" "VXwm E" "VXwm E" ja-kanji)
("" "" "Tgm " "Tgm " ja-kanji)
("" "" "nim " "nim " ja-kanji)
("" "N" "^CNm N" "^CNm N" ja-kanji)
("" "c" "z[cm c" "z[cm c" ja-kanji)
("" "c" "qLCm c" "qLCm c" ja-kanji)
("" "c" "^cm c" "^cm c" ja-kanji)
("" "c" "Om c" "Om c" ja-kanji)
("" "N" "N_cm N" "N_cm N" ja-kanji)
("" "N" "V[Nm N" "V[Nm N" ja-kanji)
("" "[" "ZCm [" "ZCm [" ja-kanji)
("" "[" "iKm [" "iKm [" ja-kanji)
("" "[" "^CPm [" "^CPm [" ja-kanji)
("" "[" "[L[m [" "[L[m [" ja-kanji)
("" "[" "gm [" "gm [" ja-kanji)
("" "[" "[Am [" "[Am [" ja-kanji)
("" "[" "cum [" "cum [" ja-kanji)
("" "[" "[Lm [" "[Lm [" ja-kanji)
("" "[" "[O[m [" "[O[m [" ja-kanji)
("" "[" "^cm ZCW" "^cm ZCW" ja-kanji)
("" "" "nm " "nm " ja-kanji)
("" "" "nJm " "nJm " ja-kanji)
("" "" "^rm " "^rm " ja-kanji)
("" "" "zm " "zm " ja-kanji)
("" "[" "[V[m [" "[V[m [" ja-kanji)
("" "[" "[[m [" "[[m [" ja-kanji)
("" "[" "h[[m [" "h[[m [" ja-kanji)
("" "[" "t^cm [" "t^cm [" ja-kanji)
("" "[" "[Km [" "[Km [" ja-kanji)
("" "[" "hNV[m [" "hNV[m [" ja-kanji)
("" "[" "[m [" "[m [" ja-kanji)
("" "[" "nm [" "nm [" ja-kanji)
("" "[" "XYVCm [" "XYVCm [" ja-kanji)
("" "[" "Jm [" "Jm [" ja-kanji)
("" "[" "`[m [" "`[m [" ja-kanji)
("" "[" "C[m [" "C[m [" ja-kanji)
("" "[" "[Zm [" "[Zm [" ja-kanji)
("" "[" "Jem [" "Jem [" ja-kanji)
("" "[" "Cm [" "Cm [" ja-kanji)
("" "[" "}Rgjm [" "}Rgjm [" ja-kanji)
("" "[" "Njm im [" "Njm im [" ja-kanji)
("" "[" "nJm [" "nJm [" ja-kanji)
("" "[" "~TTMm [" "~TTMm [" ja-kanji)
("" "[" "[hm [" "[hm [" ja-kanji)
("" "N" "`Jm N" "`Jm N" ja-kanji)
("" "N" "~hm N" "~hm N" ja-kanji)
("" "" "m " "m " ja-kanji)
("" "" "^Cm " "^Cm " ja-kanji)
("" "" "nVm " "nVm " ja-kanji)
("" "" "TrVCm " "TrVCm " ja-kanji)
("" "" "R[ucm " "R[ucm " ja-kanji)
("" "" "R[m " "R[m " ja-kanji)
("" "" "m]m " "m]m " ja-kanji)
("" "" "m " "m " ja-kanji)
("" "" "gim " "gim " ja-kanji)
("" "" "ERm " "ERm " ja-kanji)
("" "" "Lm " "Lm " ja-kanji)
("" "" "W[m " "W[m " ja-kanji)
("" "C" "g[Cm C" "g[Cm C" ja-kanji)
("" "C" "i~_m C" "i~_m C" ja-kanji)
("" "C" "CVm C" "CVm C" ja-kanji)
("" "C" "uCm C" "uCm C" ja-kanji)
("" "C" "CCm C" "CCm C" ja-kanji)
("" "C" "CWm C" "CWm C" ja-kanji)
("" "C" "^gGm C" "^gGm C" ja-kanji)
("" "C" "qGm C" "qGm C" ja-kanji)
("" "C" "nQm C" "nQm C" ja-kanji)
("" "C" "~lm C" "~lm C" ja-kanji)
("" "C" "Cm C" "Cm C" ja-kanji)
("" "C" "C[m C" "C[m C" ja-kanji)
("" "C" "CMm C" "CMm C" ja-kanji)
("" "C" "~~iOTm C" "~~iOTm C" ja-kanji)
("" "C" "XYm C" "XYm C" ja-kanji)
("" "C" "hCm C" "hCm C" ja-kanji)
("" "[" "[m C" "[m C" ja-kanji)
("" "C" "^}m C" "^}m C" ja-kanji)
("" "C" "EVCm C" "EVCm C" ja-kanji)
("" "C" "lCm C" "lCm C" ja-kanji)
("" "L" "R~m L" "R~m L" ja-kanji)
("" "L" "LVm L" "LVm L" ja-kanji)
("" "c" "ZCcm c" "ZCcm c" ja-kanji)
("" "c" "Igm c" "Igm c" ja-kanji)
("" "c" "nQVCm c" "nQVCm c" ja-kanji)
("" "c" "TPm c" "TPm c" ja-kanji)
("" "" "oCm " "oCm " ja-kanji)
("" "" "RCm " "RCm " ja-kanji)
("" "" "Am " "Am " ja-kanji)
("" "" "TUi~m " "TUi~m " ja-kanji)
("" "" "Km " "Km " ja-kanji)
("" "" "X_m " "X_m " ja-kanji)
("" "" "lm " "lm " ja-kanji)
("" "" "~~wm " "~~wm " ja-kanji)
("@" "" "nXm " "nXm " ja-kanji)
("A" "" "cim " "cim " ja-kanji)
("B" "" "Jlwm " "Jlwm " ja-kanji)
("C" "" "\g[m " "\g[m " ja-kanji)
("D" "" "Wm " "Wm " ja-kanji)
("E" "" "Om " "Om " ja-kanji)
("F" "" "_m " "_m " ja-kanji)
("G" "" "Cm " "Cm " ja-kanji)
("H" "" "h[m " "h[m " ja-kanji)
("I" "" "cm " "cm " ja-kanji)
("J" "[" "[h[m [" "[h[m [" ja-kanji)
("K" "[" "VoVom [" "VoVom [" ja-kanji)
("L" "[" "[Jm [" "[Jm [" ja-kanji)
("M" "[" "eA\um [" "eA\um [" ja-kanji)
("N" "[" "[hNm [" "[hNm [" ja-kanji)
("O" "[" "[m [" "[m [" ja-kanji)
("P" "[" "NcOm [" "NcOm [" ja-kanji)
("Q" "[" "i~m [" "i~m [" ja-kanji)
("R" "[" "m [" "m [" ja-kanji)
("S" "[" "[m [" "[m [" ja-kanji)
("T" "[" "IIJ~m [" "IIJ~m [" ja-kanji)
("U" "[" "JSm [" "JSm [" ja-kanji)
("V" "[" "ICm [" "ICm [" ja-kanji)
("W" "[" "[Am [" "[Am [" ja-kanji)
("X" "[" "[\Nm [" "[\Nm [" ja-kanji)
("Y" "[" "[m [" "[m [" ja-kanji)
("Z" "N" "JX[Wm N" "JX[Wm N" ja-kanji)
("[" "N" "TNm N" "TNm N" ja-kanji)
("\" "N" "N_Jm N" "N_Jm N" ja-kanji)
("]" "N" "bRcm N" "bRcm N" ja-kanji)
("^" "N" "NIm N" "NIm N" ja-kanji)
("_" "" "Sm " "Sm " ja-kanji)
("`" "" "}gm " "}gm " ja-kanji)
("a" "" "Om " "Om " ja-kanji)
("b" "" "niVm " "niVm " ja-kanji)
("c" "C" "Km C" "Km C" ja-kanji)
("d" "C" "Cm C" "Cm C" ja-kanji)
("e" "L[" "Lm L[" "Lm L[" ja-kanji)
("f" "N" "}hEm N" "}hEm N" ja-kanji)
("g" "N" "t`m N" "t`m N" ja-kanji)
("h" "V[" "gm V" "gm V" ja-kanji)
("i" "R[" "^m L[W" "^m L[W" ja-kanji)
("j" "R[" "R[m ^" "R[m ^" ja-kanji)
("k" "KN" "h[ucm j" "h[ucm j" ja-kanji)
("l" "^" "rm ^" "rm ^" ja-kanji)
("m" "R[" "m R[" "m R[" ja-kanji)
("n" "Pc" "rm Pc" "rm Pc" ja-kanji)
("o" "" "Ro`m " "Ro`m " ja-kanji)
("p" "" "CGm " "CGm " ja-kanji)
("q" "" "`m " "`m " ja-kanji)
("r" "" "Efm " "Efm " ja-kanji)
("" "C`" "C`m L[W" "C`m L[W" ja-kanji)
("" "JC" "A^Gm JC" "A^Gm JC" ja-kanji)
("" "q" "IILCm q" "IILCm q" ja-kanji)
("" "J" "qTVm J" "qTVm J" ja-kanji)
("" "J" "AQ}Lm J" "AQ}Lm J" ja-kanji)
("" "`" "em VV" "em VV" ja-kanji)
("" "h" "hum h" "hum h" ja-kanji)
("" "wc" "mm wc" "mm wc" ja-kanji)
("" "KC" "Jm KC" "Jm KC" ja-kanji)
("" "JC" "\Nm JC" "\Nm JC" ja-kanji)
("" "W[" "mm L[W" "mm L[W" ja-kanji)
("" "" "~_m L[W" "~_m L[W" ja-kanji)
("" "Pc" "nl{[m Pc" "nl{[m Pc" ja-kanji)
("" "" "m L[W" "m L[W" ja-kanji)
("" "W" "Rgm L[W" "Rgm L[W" ja-kanji)
("" "W" "mrm W" "mrm W" ja-kanji)
("" "j[" "X[Wm jm L[W" "X[Wm jm L[W" ja-kanji)
("" "E" "Nm E" "Nm E" ja-kanji)
("" "A" "Am L[W" "Am L[W" ja-kanji)
("" "LN" "Lm LN" "Lm LN" ja-kanji)
("" "g[" "ixu^m g[" "ixu^m g[" ja-kanji)
("" "R[" "^Jum R[" "^Jum R[" ja-kanji)
("" "L[" "~Rm ]NW" "~Rm ]NW" ja-kanji)
("" "nN" "Cm `C" "Cm `C" ja-kanji)
("" "^" "}Rgm ^" "}Rgm ^" ja-kanji)
("" "W[" "V^KEm W[" "V^KEm W[" ja-kanji)
("" "W[" "m W[" "m W[" ja-kanji)
("" "\N" "zmJm \N" "zmJm \N" ja-kanji)
("" "t" "^Im t" "^Im t" ja-kanji)
("" "h[" "n^Nm NW" "n^Nm NW" ja-kanji)
("" "W[" "cGm W[" "cGm W[" ja-kanji)
("" "W" "nJm W" "nJm W" ja-kanji)
("" "W" "Wm ]NW" "Wm ]NW" ja-kanji)
("" "Z" "ZpNm Z" "ZpNm Z" ja-kanji)
("" "JC" "Cm JC" "Cm JC" ja-kanji)
("" "R[" "^Jum R[" "^Jum R[" ja-kanji)
("" "Cc" "Ccjm Cc" "Ccjm Cc" ja-kanji)
("" "R" "RPm R" "RPm R" ja-kanji)
("" "tc" "zgPm L[W" "zgPm L[W" ja-kanji)
("" "N" "ZVm N" "ZVm N" ja-kanji)
("" "^" "rVCm _" "rVCm _" ja-kanji)
("" "eC" "^^Ym eC" "^^Ym eC" ja-kanji)
("" "Lc" "jxj V" "jxj V" ja-kanji)
("" "V" "ISm V" "ISm V" ja-kanji)
("" "V" "jxj V" "jxj V" ja-kanji)
("" "^" "rVCm ^" "rVCm ^" ja-kanji)
("" "`[" "JCm `[" "JCm `[" ja-kanji)
("" "nC" "Irm nC" "Irm nC" ja-kanji)
("" "nN" "jxj qN" "jxj qN" ja-kanji)
("" "[" "XXm [" "XXm [" ja-kanji)
("" "[" "Ccm [" "Ccm [" ja-kanji)
("" "C" "Nm L[W" "Nm L[W" ja-kanji)
("" "" "}Cm " "}Cm " ja-kanji)
("" "W" "jxj cNX" "jxj cNX" ja-kanji)
("" "P" "jxj ~" "jxj ~" ja-kanji)
("" "V" "}cm V" "}cm V" ja-kanji)
("" "\" "}iC^m \" "}iC^m \" ja-kanji)
("" "t" "gRm t" "gRm t" ja-kanji)
("" "x" "jxj " "jxj " ja-kanji)
("" "[" "qgK^m [" "qgK^m [" ja-kanji)
("" "" "jxj Tg" "jxj Tg" ja-kanji)
("" "" "jxj " "jxj " ja-kanji)
("" "eC" "IJQm eC" "IJQm eC" ja-kanji)
("" "V" "jxj N}" "jxj N}" ja-kanji)
("" "L" "jxj L" "jxj L" ja-kanji)
("" "L" "ISm L" "ISm L" ja-kanji)
("" "Nc" "jxj Nc" "jxj Nc" ja-kanji)
("" "QC" "^QCm QC" "^QCm QC" ja-kanji)
("" "R[" "jxj \" "jxj \" ja-kanji)
("" "\c" "ZKm \c" "ZKm \c" ja-kanji)
("" "\c" "ZKm ]NW" "ZKm ]NW" ja-kanji)
("" "VN" "Cm VN" "Cm VN" ja-kanji)
("" "V[" "jxj V[" "jxj V[" ja-kanji)
("" "Z" "jxj AI" "jxj AI" ja-kanji)
("" "^N" "jxj ^N" "jxj ^N" ja-kanji)
("" "q" "Vxm q" "Vxm q" ja-kanji)
("" "t" "EcNm t" "EcNm t" ja-kanji)
("" "" "jxj " "jxj " ja-kanji)
("" "[" "jxj [" "jxj [" ja-kanji)
("" "G" "GQcm G" "GQcm G" ja-kanji)
("" "J" "Jm L[W" "Jm L[W" ja-kanji)
("" "JC" "AEm L[W" "AEm L[W" ja-kanji)
("" "JC" "JCZCVm JC" "JCZCVm JC" ja-kanji)
("" "K" "jZmm K" "jZmm K" ja-kanji)
("" "PC" "Qm PC" "Qm PC" ja-kanji)
("" "T" "~iXm T" "~iXm T" ja-kanji)
("" "V" "Tem V" "Tem V" ja-kanji)
("" "\[" "ZVCm \[" "ZVCm \[" ja-kanji)
("" "g[" "kXm g[" "kXm g[" ja-kanji)
("" "JC" "JCCm JC" "JCCm JC" ja-kanji)
("" "R[" "iEm R[" "iEm R[" ja-kanji)
("" "t" "JVYNm t" "JVYNm t" ja-kanji)
("" "E" "ZVm E" "ZVm E" ja-kanji)
("" "S[" "ISm S[" "ISm S[" ja-kanji)
("@" "Z" "ZMm Z" "ZMm Z" ja-kanji)
("A" "Z" "Zjm Z" "Zjm Z" ja-kanji)
("B" "f" "c^Gm L[W" "c^Gm L[W" ja-kanji)
("C" "[" "ZVm [" "ZVm [" ja-kanji)
("D" "L" "jxj L" "jxj L" ja-kanji)
("E" "M" "Ccm L[W" "Ccm L[W" ja-kanji)
("F" "M[" "lKEm M[" "lKEm M[" ja-kanji)
("G" "Z" "Zm L[W" "Zm L[W" ja-kanji)
("H" "Z" "ZGcm Z" "ZGcm Z" ja-kanji)
("I" "h[" "jxj x" "jxj x" ja-kanji)
("J" "J" "A^Cm L[W" "A^Cm L[W" ja-kanji)
("K" "L[" "^Im L[" "^Im L[" ja-kanji)
("L" "P" "Pm L[W" "Pm L[W" ja-kanji)
("M" "V" "XOm V" "XOm V" ja-kanji)
("N" "m[" "jxj m E" "jxj m E" ja-kanji)
("O" "" "jxj J" "jxj J" ja-kanji)
("P" "ZC" "iJ}m ZC" "iJ}m ZC" ja-kanji)
("Q" "`[" "`[m `[" "`[m `[" ja-kanji)
("R" "{[" "jxj " "jxj " ja-kanji)
("S" "C" "JCCm C" "JCCm C" ja-kanji)
("T" "_" "jxj i" "jxj i" ja-kanji)
("U" "C" "jxj C" "jxj C" ja-kanji)
("V" "Q" "jxj Q" "jxj Q" ja-kanji)
("W" "g[" "XOm g[" "XOm g[" ja-kanji)
("X" "W" "qgAVm W" "qgAVm W" ja-kanji)
("Y" "Sc" "ScScm Sc" "ScScm Sc" ja-kanji)
("Z" "W" "Rhm L[W" "Rhm L[W" ja-kanji)
("[" "_" "gJGm _" "gJGm _" ja-kanji)
("\" "g" "ETMm CW" "ETMm CW" ja-kanji)
("]" "L[" "I\m L[" "I\m L[" ja-kanji)
("^" "L[" "L\[m L[" "L\[m L[" ja-kanji)
("_" "[" "[m L[W" "[m L[W" ja-kanji)
("`" "" "CGm " "CGm " ja-kanji)
("a" "PC" "WWm CbV" "WWm CbV" ja-kanji)
("b" "L" "lKEm L" "lKEm L" ja-kanji)
("c" "PC" "GK}Gm PC" "GK}Gm PC" ja-kanji)
("d" "JC" "}m ZCW" "}m ZCW" ja-kanji)
("e" "Tc" "Tcm L[W" "Tcm L[W" ja-kanji)
("f" "[" "[L[m [" "[L[m [" ja-kanji)
("g" "PC" "ALJm PC" "ALJm PC" ja-kanji)
("h" "`[" "Jugm `[" "Jugm `[" ja-kanji)
("i" "R[" "J}Gm R[" "J}Gm R[" ja-kanji)
("j" "x" "Jm x" "Jm x" ja-kanji)
("k" "xL" "Jm xL" "Jm xL" ja-kanji)
("l" "G" "kMkm G" "kMkm G" ja-kanji)
("m" "R[" "QR[m R[" "QR[m R[" ja-kanji)
("n" "`[" "cJm `[" "cJm `[" ja-kanji)
("o" "V" "EcXm L[W" "EcXm L[W" ja-kanji)
("p" "xL" "Lm xL" "Lm xL" ja-kanji)
("q" "q[" "jXCjq[" "jXCjq[" ja-kanji)
("r" "Pc" "jXCj Pc" "jXCj Pc" ja-kanji)
("s" "S" "TGm CW" "TGm CW" ja-kanji)
("t" "`[" "jXCj iJ" "jXCj iJ" ja-kanji)
("u" "q[" "jXCj ~Y" "jXCj ~Y" ja-kanji)
("v" "L[" "jXCj L[" "jXCj L[" ja-kanji)
("w" "c" "jXCj c" "jXCj c" ja-kanji)
("x" "R" "jXCj J^C" "jXCj J^C" ja-kanji)
("y" "[" "jXCj L[" "jXCj L[" ja-kanji)
("z" "" "jXCj " "jXCj " ja-kanji)
("{" "L" "cNGm L" "cNGm L" ja-kanji)
("|" "V" "Vm L[W" "Vm L[W" ja-kanji)
("}" "t[" "J[m RKV" "J[m RKV" ja-kanji)
("~" "q[" "^m q[" "^m q[" ja-kanji)
("" "I[" "IIgm I[" "IIgm I[" ja-kanji)
("" "J" "EPoRm J" "EPoRm J" ja-kanji)
("" "J" "nRm L[W" "nRm L[W" ja-kanji)
("" "j" "J^im ]NW" "J^im ]NW" ja-kanji)
("" "Z" "PYm Z" "PYm Z" ja-kanji)
("" "Pc" "GOm Pc" "GOm Pc" ja-kanji)
("" "t" "tPCm t" "tPCm t" ja-kanji)
("" "S[" "S[m ]NW" "S[m ]NW" ja-kanji)
("" "T" "PYm T" "PYm T" ja-kanji)
("" "Jc" "PYm Jc" "PYm Jc" ja-kanji)
("" "R" "GOm R" "GOm R" ja-kanji)
("" "Zc" "Zcim Zc" "Zcim Zc" ja-kanji)
("" "\[" "nWm \[" "nWm \[" ja-kanji)
("" "PC" "bg[j PC" "bg[j PC" ja-kanji)
("" "RN" "QRNW[m RN" "QRNW[m RN" ja-kanji)
("" "c" "nccm c" "nccm c" ja-kanji)
("" "L" "bg[j L" "bg[j L" ja-kanji)
("" "eL" "bg[j GL" "bg[j GL" ja-kanji)
("" "Z" "Z_m Z" "Z_m Z" ja-kanji)
("" "KC" "KCg[m KC" "KCg[m KC" ja-kanji)
("" "W[" "A}m L[W" "A}m L[W" ja-kanji)
("" "g[" "bg[j g[" "bg[j g[" ja-kanji)
("" "\[" "bg[j X" "bg[j X" ja-kanji)
("" "q[" "kXm q[" "kXm q[" ja-kanji)
("" "P" "Pm L[W" "Pm L[W" ja-kanji)
("" "P" "Pm ]NW" "Pm ]NW" ja-kanji)
("" "P" "J^im cM" "J^im cM" ja-kanji)
("" "P" "Com cM" "Com cM" ja-kanji)
("" "wL" "TNm wL" "TNm wL" ja-kanji)
("" "UC" "NXm L[W" "NXm L[W" ja-kanji)
("" "x" "bg[m x" "bg[m x" ja-kanji)
("" "x" "J^im x" "J^im x" ja-kanji)
("" "N" "Nj `J" "Nj `J" ja-kanji)
("" "V[" "cgm V[" "cgm V[" ja-kanji)
("" "L`" "Vj `J" "Vj `J" ja-kanji)
("" "P" "Pm L[W" "Pm L[W" ja-kanji)
("" "PC" "cCm PC" "cCm PC" ja-kanji)
("" "PC" "L[j `J" "L[j `J" ja-kanji)
("" "LN" "cgm LN" "cgm LN" ja-kanji)
("" "[" "[m L[W" "[m L[W" ja-kanji)
("" "ZL" "`Jm ZL" "`Jm ZL" ja-kanji)
("" "\[" "Xj `J" "Xj `J" ja-kanji)
("" "`N" "^_Xm `N" "^_Xm `N" ja-kanji)
("" "N" "AZm N" "AZm N" ja-kanji)
("" "N" "CTIm L[W" "CTIm L[W" ja-kanji)
("" "C" "nQ}Xm L[W" "nQ}Xm L[W" ja-kanji)
("" "J" "XXm L[W" "XXm L[W" ja-kanji)
("" "z[" "cc~K}G" "cc~K}G" ja-kanji)
("" "\[" "\[\[m \[" "\[\[m \[" ja-kanji)
("" "L[" "L[hm L[" "L[hm L[" ja-kanji)
("" "e" "^j cc~K}G" "^j cc~K}G" ja-kanji)
("" "z" "nEm z" "nEm z" ja-kanji)
("" "tN" "ztNm tN" "ztNm tN" ja-kanji)
("" "z[" "qTSm z[" "qTSm z[" ja-kanji)
("" "q" "ACN`m q" "ACN`m q" ja-kanji)
("" "z[" "nRK}G" "nRK}G" ja-kanji)
("" "R[" "phm nR" "phm nR" ja-kanji)
("" "C" "Ac}m C" "Ac}m C" ja-kanji)
("" "L" "g{VCm L" "g{VCm L" ja-kanji)
("" "" "JK~oRm " "JK~oRm " ja-kanji)
("" "PC" "JNVK}G" "JNVK}G" ja-kanji)
("" "N" "Nm L[W" "Nm L[W" ja-kanji)
("" "\c" "\cm ]NW" "\cm ]NW" ja-kanji)
("" "\[" "NTm \[" "NTm \[" ja-kanji)
("" "ZC" "m L[W" "m L[W" ja-kanji)
("" "L" "NTm L" "NTm L" ja-kanji)
("" "o" "}Wm o" "}Wm o" ja-kanji)
("" "W" "jXCm W" "jXCm W" ja-kanji)
("" "x" "ZbJ`m x" "ZbJ`m x" ja-kanji)
("" "Zc" "tVdNm Zc" "tVdNm Zc" ja-kanji)
("" "V" "TJYLm V" "TJYLm V" ja-kanji)
("" "{[" "ETMm CW" "ETMm CW" ja-kanji)
("" "LN" "V]Nm L[W" "V]Nm L[W" ja-kanji)
("" "J" "}Lm L[W" "}Lm L[W" ja-kanji)
("" "J" "K_m J" "K_m J" ja-kanji)
("" "{[" "{[_Cm {[" "{[_Cm {[" ja-kanji)
("" "V" "Jm V" "Jm V" ja-kanji)
("" "J" "Jm ]NW" "Jm ]NW" ja-kanji)
("" "Pc" "\mm Pc" "\mm Pc" ja-kanji)
("" "V" "Vxm V" "Vxm V" ja-kanji)
("" "V[" "V[m ]NW" "V[m ]NW" ja-kanji)
("" "V" "m V" "m V" ja-kanji)
("" "T" "}Cm L[W" "}Cm L[W" ja-kanji)
("" "T" "EoEm T" "EoEm T" ja-kanji)
("" "\[" "\[m L[W" "\[m L[W" ja-kanji)
("" "\[" "ILim \[" "ILim \[" ja-kanji)
("" "}" "}_m }" "}_m }" ja-kanji)
("" "V[" "Qm V[" "Qm V[" ja-kanji)
("" "eC" "eClCm eC" "eClCm eC" ja-kanji)
("" "g[" "~_m g[" "~_m g[" ja-kanji)
("" "n" "bpm p" "bpm p" ja-kanji)
("" "j[" "N`wj nC" "N`wj nC" ja-kanji)
("" "N" "iQNm N" "iQNm N" ja-kanji)
("" "C" "N`wj EV" "N`wj EV" ja-kanji)
("" "K" "N`wj Lo" "N`wj Lo" ja-kanji)
("" "M" "|hm M" "|hm M" ja-kanji)
("" "R[" "mhm R[" "mhm R[" ja-kanji)
("" "R[" "zGm R[" "zGm R[" ja-kanji)
("" "Z" "XXm Z" "XXm Z" ja-kanji)
("" "gc" "gcxm gc" "gcxm gc" ja-kanji)
("" "t" "N`wj P" "N`wj P" ja-kanji)
("" "" "P`m " "P`m " ja-kanji)
("" "tB[g" "^Cm tB[g" "^Cm tB[g" ja-kanji)
("" "GC" "E^Em ]NW" "E^Em ]NW" ja-kanji)
("" "J" "VJm J" "VJm J" ja-kanji)
("" "L[" "gKm L[" "gKm L[" ja-kanji)
("" "Q" "cuNm Q" "cuNm Q" ja-kanji)
("" "R" "N`wj E" "N`wj E" ja-kanji)
("" "R[" "N`wj R[" "N`wj R[" ja-kanji)
("" "V" "\Vm V" "\Vm V" ja-kanji)
("" "W" "mEm L[W" "mEm L[W" ja-kanji)
("" "V" "ENm V" "ENm V" ja-kanji)
("" "\" "\VNm \" "\VNm \" ja-kanji)
("" "h" "N`wj bR" "N`wj bR" ja-kanji)
("" "gc" "niVJm niV" "niVJm niV" ja-kanji)
("" "t" "N`wj t" "N`wj t" ja-kanji)
("" "z[" "zGm z[" "zGm z[" ja-kanji)
("" "AC" "nNm AC" "nNm AC" ja-kanji)
("" "KN" "IhNm L[W" "IhNm L[W" ja-kanji)
("" "J" "J[m J" "J[m J" ja-kanji)
("" "L" "Jm L" "Jm L" ja-kanji)
("" "R[" "Jm R[" "Jm R[" ja-kanji)
("" "R[" "hLm R[" "hLm R[" ja-kanji)
("" "R[" "N`wj S[" "N`wj S[" ja-kanji)
("" "V" "nJm V" "nJm V" ja-kanji)
("@" "V" "^m V" "^m V" ja-kanji)
("A" "V" "N`wj jV" "N`wj jV" ja-kanji)
("B" "^" "N`wj ^N" "N`wj ^N" ja-kanji)
("C" "[" "N`wj [" "N`wj [" ja-kanji)
("D" "J" "Jm P" "Jm P" ja-kanji)
("E" "R[" "N`wj CN" "N`wj CN" ja-kanji)
("F" "J" "E^m J" "E^m J" ja-kanji)
("G" "K" "N`wj " "N`wj " ja-kanji)
("H" "L" "N`wj }" "N`wj }" ja-kanji)
("I" "S" "N`wj S" "N`wj S" ja-kanji)
("J" "R[" "N`wj T" "N`wj T" ja-kanji)
("K" "R[" "z[R[m R[" "z[R[m R[" ja-kanji)
("L" "RN" "iNm RN" "iNm RN" ja-kanji)
("M" "z" "zj[Cm z" "zj[Cm z" ja-kanji)
("N" "[" "TGYm [" "TGYm [" ja-kanji)
("O" "I" "N`wj ICe" "N`wj ICe" ja-kanji)
("P" "KC" "N`wj KC" "N`wj KC" ja-kanji)
("Q" "K" "tNm K" "tNm K" ja-kanji)
("R" "R[" "N`wj \" "N`wj \" ja-kanji)
("S" "V[" "Em V[" "Em V[" ja-kanji)
("T" "Zc" "XXm Zc" "XXm Zc" ja-kanji)
("U" "^N" "N`wj ^N" "N`wj ^N" ja-kanji)
("V" "^" "P^Jm ^" "P^Jm ^" ja-kanji)
("W" "^" "NEm ^" "NEm ^" ja-kanji)
("X" "e" "Eim e" "Eim e" ja-kanji)
("Y" "C" "N`wj h" "N`wj h" ja-kanji)
("Z" "J" "N`wj " "N`wj " ja-kanji)
("[" "JC" "N`oVm JC" "N`oVm JC" ja-kanji)
("\" "JN" "JbPcm JN" "JbPcm JN" ja-kanji)
("]" "JN" "JNm CW" "JNm CW" ja-kanji)
("^" "J" "JZCm J" "JZCm J" ja-kanji)
("_" "L" "iQNm L" "iQNm L" ja-kanji)
("`" "V" "^_jm V" "^_jm V" ja-kanji)
("a" "V[" "iNm V[" "iNm V[" ja-kanji)
("b" "[" "[\Nm [" "[\Nm [" ja-kanji)
("c" "\N" "iNm \N" "iNm \N" ja-kanji)
("d" "^" "^m L[W" "^m L[W" ja-kanji)
("e" "eC" "iNm eC" "iNm eC" ja-kanji)
("f" "i" "N`wj ~i~" "N`wj ~i~" ja-kanji)
("g" "" "TgXm " "TgXm " ja-kanji)
("h" "" "bpm " "bpm " ja-kanji)
("i" "[" "zKJm [" "zKJm [" ja-kanji)
("j" "I" "IGcm I" "IGcm I" ja-kanji)
("k" "L[" "JOm L[" "JOm L[" ja-kanji)
("l" "T" "iQNm T" "iQNm T" ja-kanji)
("m" "T" "VKm T" "VKm T" ja-kanji)
("n" "V" "VR[qm V" "VR[qm V" ja-kanji)
("o" "V" "AUPEm V" "AUPEm V" ja-kanji)
("p" "V" "N`wj }Rg" "N`wj }Rg" ja-kanji)
("q" "I[" "nNm I[" "nNm I[" ja-kanji)
("r" "S[" "J}rXVC" "J}rXVC" ja-kanji)
("s" "TN" "N`wj ZL" "N`wj ZL" ja-kanji)
("t" "\[" "\\mJXm \[" "\\mJXm \[" ja-kanji)
("u" "\[" "ZLm \[" "ZLm \[" ja-kanji)
("v" "}" "N`wj }" "N`wj }" ja-kanji)
("w" "" "N`wj " "N`wj " ja-kanji)
("x" "Cc" "Zum Cc" "Zum Cc" ja-kanji)
("y" "L" "Ecm L[W" "Ecm L[W" ja-kanji)
("z" "GC" "Cgim L[W" "Cgim L[W" ja-kanji)
("{" "V" "N`oVm V" "N`oVm V" ja-kanji)
("|" "ZC" "CiiNm ZC" "CiiNm ZC" ja-kanji)
("}" "`[" "AUPm `[" "AUPm `[" ja-kanji)
("~" "u" "N`wj iV" "N`wj iV" ja-kanji)
("" "C" "AAm C" "AAm C" ja-kanji)
("" "L" "cOm L" "cOm L" ja-kanji)
("" "V[" "E\uNm V[" "E\uNm V[" ja-kanji)
("" "[C" "Jm [C" "Jm [C" ja-kanji)
("" "\[" "TOm \[" "TOm \[" ja-kanji)
("" "R[" "R[Vm R[" "R[Vm R[" ja-kanji)
("" "eC" "N`wj eC" "N`wj eC" ja-kanji)
("" "q" "JJAm q" "JJAm q" ja-kanji)
("" "[" "[[m [" "[[m [" ja-kanji)
("" "eC" "NV~m eC" "NV~m eC" ja-kanji)
("" "eC" "NV~m ]NW" "NV~m ]NW" ja-kanji)
("" "G" "m~Rm G" "m~Rm G" ja-kanji)
("" "L[" "TLjm L[" "TLjm L[" ja-kanji)
("" "I[" "iNm I[" "iNm I[" ja-kanji)
("" "Q" "LrVCm L[W" "LrVCm L[W" ja-kanji)
("" "S[" "J}rXVC" "J}rXVC" ja-kanji)
("" "VN" "\VNm VN" "\VNm VN" ja-kanji)
("" "V[" "TTNm V[" "TTNm V[" ja-kanji)
("" "e" "TGYm e" "TGYm e" ja-kanji)
("" "QC" "^Sgm QC" "^Sgm QC" ja-kanji)
("" "\" "N`wj IN" "N`wj IN" ja-kanji)
("" "VN" "VNm L[W" "VNm L[W" ja-kanji)
("" "Qc" "JWm L[W" "JWm L[W" ja-kanji)
("" "C" "NjK}Gm C" "NjK}Gm C" ja-kanji)
("" "J" "Igm J" "Igm J" ja-kanji)
("" "C" "NjK}Gj C" "NjK}Gj C" ja-kanji)
("" "RN" "~cNjm Nj" "~cNjm Nj" ja-kanji)
("" "[" "NjK}Gj A" "NjK}Gj A" ja-kanji)
("" "S" "NjK}Gj " "NjK}Gj " ja-kanji)
("" "M" "NjK}Gj T`" "NjK}Gj T`" ja-kanji)
("" "P" "Pm L[W" "Pm L[W" ja-kanji)
("" "RN" "Njm L[W" "Njm L[W" ja-kanji)
("" "C" "JRm L[W" "JRm L[W" ja-kanji)
("" "G" "}hJm L[W" "}hJm L[W" ja-kanji)
("" "_" "J^}m L[W" "J^}m L[W" ja-kanji)
("" "Y" "nJm L[W" "nJm L[W" ja-kanji)
("" "VN" "VNm VN" "VNm VN" ja-kanji)
("" "G" "}Cm G" "}Cm G" ja-kanji)
("" "j[" "c`wj nC" "c`wj nC" ja-kanji)
("" "J" "c`wj V^" "c`wj V^" ja-kanji)
("" "T" "c`wj }" "c`wj }" ja-kanji)
("" "J" "JZCm J" "JZCm J" ja-kanji)
("" "L" "TJCm L" "TJCm L" ja-kanji)
("" "V" "W[Vm V" "W[Vm V" ja-kanji)
("" "nC" "gWm nC" "gWm nC" ja-kanji)
("" "J" "c{m J" "c{m J" ja-kanji)
("" "XC" "^m L[W" "^m L[W" ja-kanji)
("" "J" "Jj c`" "Jj c`" ja-kanji)
("" "n" "cc~m n" "cc~m n" ja-kanji)
("" "t" "c`wj t" "c`wj t" ja-kanji)
("" "z[" "c`wj cc" "c`wj cc" ja-kanji)
("" "KC" "INm ZoC" "INm ZoC" ja-kanji)
("" "M" "c`wj S" "c`wj S" ja-kanji)
("" "R[" "c`wj CN" "c`wj CN" ja-kanji)
("" "ec" "c`wj C^" "c`wj C^" ja-kanji)
("" "n" "c`wj wC" "c`wj wC" ja-kanji)
("" "g[Q" "^Im g[Q" "^Im g[Q" ja-kanji)
("" "AC" "`m AC" "`m AC" ja-kanji)
("" "JN" "c`wj JN" "c`wj JN" ja-kanji)
("" "z" "c`wj z" "c`wj z" ja-kanji)
("" "`" "`m ZCW" "`m ZCW" ja-kanji)
("" "`" "`iCm `" "`iCm `" ja-kanji)
("" "A" "Vc`m A" "Vc`m A" ja-kanji)
("" "J" "c`wj ni" "c`wj ni" ja-kanji)
("" "TC" "c`wj \c" "c`wj \c" ja-kanji)
("" "z[" "AY`m z[" "AY`m z[" ja-kanji)
("" "C" "tTOm C" "tTOm C" ja-kanji)
("" "J" "c{m J" "c{m J" ja-kanji)
("" "W[" "om L[W" "om L[W" ja-kanji)
("" "z" "gfm z" "gfm z" ja-kanji)
("" "I" "c`wj JX" "c`wj JX" ja-kanji)
("" "GC" "nJm GC" "nJm GC" ja-kanji)
("" "JC" "E~j c`" "E~j c`" ja-kanji)
("" "L" "Lz[wm L" "Lz[wm L" ja-kanji)
("" "V" "c`wj gL" "c`wj gL" ja-kanji)
("" "R[" "IJm R[" "IJm R[" ja-kanji)
("" "U" "US[m U" "US[m U" ja-kanji)
("" "" "mj c`" "mj c`" ja-kanji)
("" "J" "c`wj AC_" "c`wj AC_" ja-kanji)
("" "L" "nCLm L" "nCLm L" ja-kanji)
("" "\" "c`wj \" "c`wj \" ja-kanji)
("" "I[" "c`wj IN" "c`wj IN" ja-kanji)
("" "JC" "Rm L[W" "Rm L[W" ja-kanji)
("" "V[" "JLlm V[" "JLlm V[" ja-kanji)
("" "`" "c`wj `" "c`wj `" ja-kanji)
("" "_" "i}Pm L[W" "i}Pm L[W" ja-kanji)
("" "[" "tZOm [" "tZOm [" ja-kanji)
("" "Ac" "ITGm L[W" "ITGm L[W" ja-kanji)
("" "KN" "^jm KN" "^jm KN" ja-kanji)
("" "W" "c`wj cNX" "c`wj cNX" ja-kanji)
("" "R[" "c`wj R[" "c`wj R[" ja-kanji)
("" "C" "Cm L[W" "Cm L[W" ja-kanji)
("" "e" "c`wj e" "c`wj e" ja-kanji)
("" "^" "rdm r" "rdm r" ja-kanji)
("" "W[" "W[m L[W" "W[m L[W" ja-kanji)
("" "[" "c`j [" "c`j [" ja-kanji)
("" "\[" "\[m L[W" "\[m L[W" ja-kanji)
("" "R" "c{m L[W" "c{m L[W" ja-kanji)
("" "C`" "qgcm L[W" "qgcm L[W" ja-kanji)
("" "ZC" "Rm ]NW" "Rm ]NW" ja-kanji)
("" "R" "RNm R" "RNm R" ja-kanji)
("" "W" "RguLm L[W" "RguLm L[W" ja-kanji)
("" "`" "tKVm `" "tKVm `" ja-kanji)
("" "XC" "XCj[m XC" "XCj[m XC" ja-kanji)
("" "PC" "nJm PC" "nJm PC" ja-kanji)
("" "^" "IICm L[W" "IICm L[W" ja-kanji)
("" "" "m L[W" "m L[W" ja-kanji)
("" "J" "Ir^_VCm J" "Ir^_VCm J" ja-kanji)
("" "Pc" "Lm Pc" "Lm Pc" ja-kanji)
("" "[" "[Zcm [" "[Zcm [" ja-kanji)
("" "z" "gm L[W" "gm L[W" ja-kanji)
("" "R" "zRm R" "zRm R" ja-kanji)
("" "L[" "nTm L[" "nTm L[" ja-kanji)
("" "L" "Lm ]NW" "Lm ]NW" ja-kanji)
("" "GL" "oN`m GL" "oN`m GL" ja-kanji)
("" "J" "TJim J" "TJim J" ja-kanji)
("" "PC" "PCEm PC" "PCEm PC" ja-kanji)
("" "PC" "CYN]m PC" "CYN]m PC" ja-kanji)
("" "\[" "\[j IILC" "\[j IILC" ja-kanji)
("" "V" "ISm V" "ISm V" ja-kanji)
("" "e" "}cm e" "}cm e" ja-kanji)
("" "I[" "INm L[W" "INm L[W" ja-kanji)
("" "V[" "XXm L[W" "XXm L[W" ja-kanji)
("" "" "JK~oRm " "JK~oRm " ja-kanji)
("@" "J" "RV}m J" "RV}m J" ja-kanji)
("A" "VN" "oCVNm VN" "oCVNm VN" ja-kanji)
("B" "\[" "\IEm \[" "\IEm \[" ja-kanji)
("C" "C" "RV}m C" "RV}m C" ja-kanji)
("D" "C" "RV}m CW" "RV}m CW" ja-kanji)
("E" "q" "iL nnm q" "iL nnm q" ja-kanji)
("F" "_c" "_bLm _c" "_bLm _c" ja-kanji)
("G" "{" "Iiwj nn" "Iiwj nn" ja-kanji)
("H" "C" "Iom C" "Iom C" ja-kanji)
("I" "L[" "nWJ~m L[" "nWJ~m L[" ja-kanji)
("J" "P" "EcNVCm P" "EcNVCm P" ja-kanji)
("K" "j" "nm CW" "nm CW" ja-kanji)
("L" "[" "[R[m [" "[R[m [" ja-kanji)
("M" "K" "Iiwj " "Iiwj " ja-kanji)
("N" "P" "PPm P" "PPm P" ja-kanji)
("O" "V" "Vom V" "Vom V" ja-kanji)
("P" "_" "ViJm _" "ViJm _" ja-kanji)
("Q" "wC" "gm wC" "gm wC" ja-kanji)
("R" "i" "Iij IgR" "Iij IgR" ja-kanji)
("S" "A" "Iiwj A" "Iiwj A" ja-kanji)
("T" "C" "~_m L[W" "~_m L[W" ja-kanji)
("U" "G" "GLNm G" "GLNm G" ja-kanji)
("V" "V" "Iiwj g" "Iiwj g" ja-kanji)
("W" "V" "gm V" "gm V" ja-kanji)
("X" "q" "QWm q" "QWm q" ja-kanji)
("Y" "" "T{m " "T{m " ja-kanji)
("Z" "r" "Rrm r" "Rrm r" ja-kanji)
("[" "I[" "I[im I[" "I[im I[" ja-kanji)
("\" "R[" "CcNVm R[" "CcNVm R[" ja-kanji)
("]" "W[" "^IJm W[" "^IJm W[" ja-kanji)
("^" "\[" "Ajm \[" "Ajm \[" ja-kanji)
("_" "{" "Iiwj E}" "Iiwj E}" ja-kanji)
("`" "G" "G[m G" "G[m G" ja-kanji)
("a" "I[" "nnm I[" "nnm I[" ja-kanji)
("b" "R[" "Iiwj cl" "Iiwj cl" ja-kanji)
("c" "h" "JCm h" "JCm h" ja-kanji)
("d" "q[" "JCm q[" "JCm q[" ja-kanji)
("e" "J" "~rJm J" "~rJm J" ja-kanji)
("f" "J" "Jm CW" "Jm CW" ja-kanji)
("g" "L[" "i}JVC" "i}JVC" ja-kanji)
("h" "Z" "ZPm Z" "ZPm Z" ja-kanji)
("i" "wC" "wC`[m wC" "wC`[m wC" ja-kanji)
("j" "W[" "IgRm iu" "IgRm iu" ja-kanji)
("k" "W[" "Iim iu" "Iim iu" ja-kanji)
("l" "q" "qm q" "qm q" ja-kanji)
("m" "r" "Iiwj ni" "Iiwj ni" ja-kanji)
("n" "" "Iiwj C" "Iiwj C" ja-kanji)
("o" "W[" "Xm L[W" "Xm L[W" ja-kanji)
("p" "Z" "JCm Z" "JCm Z" ja-kanji)
("q" "\[" "m \[" "m \[" ja-kanji)
("r" "Qc" "Qccm Qc" "Qccm Qc" ja-kanji)
("s" "[" "nm [" "nm [" ja-kanji)
("t" "t" "^}Sm t" "^}Sm t" ja-kanji)
("u" "nC" "z[L{Vm nC" "z[L{Vm nC" ja-kanji)
("v" "h" "bRj V" "bRj V" ja-kanji)
("w" "KC" "~hSm KC" "~hSm KC" ja-kanji)
("x" "WN" "CYm WN" "CYm WN" ja-kanji)
("y" "W" "VQm W" "VQm W" ja-kanji)
("z" "t" "JGm t" "JGm t" ja-kanji)
("{" "KN" "}ium L[W" "}ium L[W" ja-kanji)
("|" "KN" "}ium ]NW" "}ium ]NW" ja-kanji)
("}" "W" "WVm W" "WVm W" ja-kanji)
("" "^" "zJm ^" "zJm ^" ja-kanji)
("" "J" "JKm J" "JKm J" ja-kanji)
("" "V" "Vscm V" "Vscm V" ja-kanji)
("" "G" "E~m ]NW" "E~m ]NW" ja-kanji)
("" "R[" "A_m ]NW" "A_m ]NW" ja-kanji)
("" "JN" "AKm JN" "AKm JN" ja-kanji)
("" "VN" "}Rgjm VN" "}Rgjm VN" ja-kanji)
("" "r" "lm r" "lm r" ja-kanji)
("" "S" "Tm S" "Tm S" ja-kanji)
("" "Wc" "~mm L[W" "~mm L[W" ja-kanji)
("" "V" "lm L[W" "lm L[W" ja-kanji)
("" "oN" "TrVCm oN" "TrVCm oN" ja-kanji)
("" "[" "TrVCm [" "TrVCm [" ja-kanji)
("" "V" "EcXm L[W" "EcXm L[W" ja-kanji)
("" "J" "JNm J" "JNm J" ja-kanji)
("" "z[" "^Jm L[W" "^Jm L[W" ja-kanji)
("" "z[" "^Jm ]NW" "^Jm ]NW" ja-kanji)
("" "RN" "Jcm RN" "Jcm RN" ja-kanji)
("" "V[" "V[m L[W" "V[m L[W" ja-kanji)
("" "Z" "bpm L[W" "bpm L[W" ja-kanji)
("" "^C" "cCm L[W" "cCm L[W" ja-kanji)
("" "W" "jm NW" "jm NW" ja-kanji)
("" "Z" "XNiCm Z" "XNiCm Z" ja-kanji)
("" "I[" "}QAVm I[" "}QAVm I[" ja-kanji)
("" "{[" "NCkm {[" "NCkm {[" ja-kanji)
("" "V" "VJolm V" "VJolm V" ja-kanji)
("" "C" "ITm C" "ITm C" ja-kanji)
("" "q" "wm q" "wm q" ja-kanji)
("" "JC" "ghPm L[W" "ghPm L[W" ja-kanji)
("" "V" "N\m V" "N\m V" ja-kanji)
("" "L" "qCLm L" "qCLm L" ja-kanji)
("" "PL" "nLmm PL" "nLmm PL" ja-kanji)
("" "wC" "r[um r[" "r[um r[" ja-kanji)
("" "Z" "Cm Z" "Cm Z" ja-kanji)
("" "VN" "]Nm L[W" "]Nm L[W" ja-kanji)
("" "ec" "ec" "ec" ja-kanji)
("" "KC" "}j co" "}j co" ja-kanji)
("" "g[" "}j J^i" "}j J^i" ja-kanji)
("" "Lc" "\rGm Lc" "\rGm Lc" ja-kanji)
("" "L[" "}j Iu" "}j Iu" ja-kanji)
("" "V" "~lm V" "~lm V" ja-kanji)
("" "^" "tj }" "tj }" ja-kanji)
("" "V" "}j Ii" "}j Ii" ja-kanji)
("" "V[" "V[Em V[" "V[Em V[" ja-kanji)
("" "W" "}m i}G" "}m i}G" ja-kanji)
("" "nN" "}wj V" "}wj V" ja-kanji)
("" "wC" "}wj ^C" "}wj ^C" ja-kanji)
("" "r" "rUm r" "rUm r" ja-kanji)
("" "x" "}wj x" "}wj x" ja-kanji)
("" "Z" "}wj Z" "}wj Z" ja-kanji)
("" "R[" "}j AE" "}j AE" ja-kanji)
("" "W" "}wj e" "}wj e" ja-kanji)
("" "K" "}j " "}j " ja-kanji)
("" "L[" "nU}m CW" "nU}m CW" ja-kanji)
("" "R[" "}wj T" "}wj T" ja-kanji)
("" "V[" "PVCm V[" "PVCm V[" ja-kanji)
("" "g[" "V}m ]NW" "V}m ]NW" ja-kanji)
("" "N" "}wj ^j" "}wj ^j" ja-kanji)
("" "J" "JUm J" "JUm J" ja-kanji)
("" "KC" "KPm KC" "KPm KC" ja-kanji)
("" "R[" "niIJm IJ" "niIJm IJ" ja-kanji)
("" "L" "~TLm ]NW" "~TLm ]NW" ja-kanji)
("" "M" "}j L" "}j L" ja-kanji)
("" "Nc" "\o_cm Nc" "\o_cm Nc" ja-kanji)
("" "R" "Rm R" "Rm R" ja-kanji)
("" "TC" "}j tg" "}j tg" ja-kanji)
("" "\[" "\[R[m \[" "\[R[m \[" ja-kanji)
("" "[" "}wj [" "}wj [" ja-kanji)
("" "" "Rm " "Rm " ja-kanji)
("" "" "m CW" "m CW" ja-kanji)
("" "J" "nRm J" "nRm J" ja-kanji)
("" "K" "Cm K" "Cm K" ja-kanji)
("" "O[" "}wj O[" "}wj O[" ja-kanji)
("" "r" "}wj r" "}wj r" ja-kanji)
("" "JC" "}j Ij" "}j Ij" ja-kanji)
("" "T" "}j T" "}j T" ja-kanji)
("" "WN" "}wj C" "}wj C" ja-kanji)
("" "N" "}wj N" "}wj N" ja-kanji)
("" "U" "U[m U" "U[m U" ja-kanji)
("" "V[" "}wj V[" "}wj V[" ja-kanji)
("" "M[" "}wj M[" "}wj M[" ja-kanji)
("" "g[" "}wj m {" "}wj m {" ja-kanji)
("" "M" "}wj M" "}wj M" ja-kanji)
("" "P" "PVCm P" "PVCm P" ja-kanji)
("" "KN" "^Pm L[W" "^Pm L[W" ja-kanji)
("" "[" "}j ^JV" "}j ^JV" ja-kanji)
("" "M" "}j E^KE" "}j E^KE" ja-kanji)
("" "V" "V}m V" "V}m V" ja-kanji)
("" "T" "TKm T" "TKm T" ja-kanji)
("" "M" "M[m M" "M[m M" ja-kanji)
("" "e" "C^_Lm e" "C^_Lm e" ja-kanji)
("" "" "z[m " "z[m " ja-kanji)
("" "K" "PVCm L[W" "PVCm L[W" ja-kanji)
("" "Z" "Jm Z" "Jm Z" ja-kanji)
("" "u" "~Rm u" "~Rm u" ja-kanji)
("" "C" "Gjm C" "Gjm C" ja-kanji)
("" "V" "TJYLm V" "TJYLm V" ja-kanji)
("" "V" "J~m CW" "J~m CW" ja-kanji)
("" "\[" "z[Lm \[" "z[Lm \[" ja-kanji)
("" "`c" "t~dc~m ec" "t~dc~m ec" ja-kanji)
("" "h" "JlCm h" "JlCm h" ja-kanji)
("" "nN" "Lkm nN" "Lkm nN" ja-kanji)
("" "^C" "Irm L[W" "Irm L[W" ja-kanji)
("" "C" "J^rm C" "J^rm C" ja-kanji)
("" "AN" "gom AN" "gom AN" ja-kanji)
("" "C" "gom C" "gom C" ja-kanji)
("" "eC" "nMkm eC" "nMkm eC" ja-kanji)
("" "xL" "JIJNVm xL" "JIJNVm xL" ja-kanji)
("" "JN" "nowj Nj" "nowj Nj" ja-kanji)
("" "}" "}}Nm }" "}}Nm }" ja-kanji)
("" "V" "m{m V" "m{m V" ja-kanji)
("" "g[" "n^m g[" "n^m g[" ja-kanji)
("" "wC" "kTm L[W" "kTm L[W" ja-kanji)
("" "z[" "^CR`m z[" "^CR`m z[" ja-kanji)
("" "P" "^Cm P" "^Cm P" ja-kanji)
("" "wC" "AZm wC" "AZm wC" ja-kanji)
("" "[" "CgKVm [" "CgKVm [" ja-kanji)
("" "}" "CJjm }" "CJjm }" ja-kanji)
("" "Q" "}_m Q" "}_m Q" ja-kanji)
("" "V[" "}_j qcW" "}_j qcW" ja-kanji)
("" "V" "Jm V" "Jm V" ja-kanji)
("" "V[" "}_j AE" "}_j AE" ja-kanji)
("" "J" "}_j ic" "}_j ic" ja-kanji)
("" "L[" "E}m L[" "E}m L[" ja-kanji)
("" "L[" "L[m CW" "L[m CW" ja-kanji)
("@" "[" "}_j [" "}_j [" ja-kanji)
("A" "R[" "qCm L[W" "qCm L[W" ja-kanji)
("B" "V" "Rmm V" "Rmm V" ja-kanji)
("C" "`[" "Nm CW" "Nm CW" ja-kanji)
("D" "e" "~Zm e" "~Zm e" ja-kanji)
("E" "nC" "X^m L[W" "X^m L[W" ja-kanji)
("F" "u" "}_j iV" "}_j iV" ja-kanji)
("G" "JC" "}_j gN" "}_j gN" ja-kanji)
("H" "" "Nm " "Nm " ja-kanji)
("I" "" "Um " "Um " ja-kanji)
("J" "[" "Om [" "Om [" ja-kanji)
("K" "`[" "`[m L[W" "`[m L[W" ja-kanji)
("L" "`[" "}_j LN" "}_j LN" ja-kanji)
("M" "C" "Gj[m C" "Gj[m C" ja-kanji)
("N" "eL" "~`m eL" "~`m eL" ja-kanji)
("O" "L[" "jW[AVm L[" "jW[AVm L[" ja-kanji)
("P" "L" "Xem ]NW" "Xem ]NW" ja-kanji)
("Q" "\[" "\[m L[W" "\[m L[W" ja-kanji)
("R" "C" "clm C" "clm C" ja-kanji)
("S" "C" "Cm ]NW" "Cm ]NW" ja-kanji)
("T" "N" "VLK}Gm N" "VLK}Gm N" ja-kanji)
("U" "V" "RXm V" "RXm V" ja-kanji)
("V" "e" "~j C`" "~j C`" ja-kanji)
("W" "h" "CV~m h" "CV~m h" ja-kanji)
("X" "r" "m r" "m r" ja-kanji)
("Y" "z[" "~wj z[" "~wj z[" ja-kanji)
("Z" "J" "~wj J" "~wj J" ja-kanji)
("[" "_" "qNm L[W" "qNm L[W" ja-kanji)
("\" "r" "m L[W" "m L[W" ja-kanji)
("]" "" "qNm L[W" "qNm L[W" ja-kanji)
("^" "" "qNm " "qNm " ja-kanji)
("_" "PC" "PCKVm PC" "PCKVm PC" ja-kanji)
("`" "^" "^Wm ^" "^Wm ^" ja-kanji)
("a" "XC" "XCZCm XC" "XCZCm XC" ja-kanji)
("b" "C" "nlY~m C" "nlY~m C" ja-kanji)
("c" "T" "TdNm T" "TdNm T" ja-kanji)
("d" "z[" "cd~m Ig" "cd~m Ig" ja-kanji)
("e" "eL" "M[jx" "M[jx" ja-kanji)
("f" "z[" "T}Em z[" "T}Em z[" ja-kanji)
("g" "I[" "Nm ]NW" "Nm ]NW" ja-kanji)
("h" "\" "IM[ \Cm \" "IM[ \Cm \" ja-kanji)
("i" "tc" "zgPm L[W" "zgPm L[W" ja-kanji)
("j" "JC" "nCJCm JC" "nCJCm JC" ja-kanji)
("k" "R" "nin_m R" "nin_m R" ja-kanji)
("l" "PC" "R~`m L[W" "R~`m L[W" ja-kanji)
("m" "W" "giGm W" "giGm W" ja-kanji)
("n" "W[" "V^KEm L[W" "V^KEm L[W" ja-kanji)
("o" "V" "Ecm V" "Ecm V" ja-kanji)
("p" "nC" "nCJCm nC" "nCJCm nC" ja-kanji)
("q" "C" "Nm RW" "Nm RW" ja-kanji)
("r" "R[" "z[R[m R[" "z[R[m R[" ja-kanji)
("s" "[" "uNm [" "uNm [" ja-kanji)
("t" "L[" "gm L[" "gm L[" ja-kanji)
("u" "\" "IVnJm \" "IVnJm \" ja-kanji)
("v" "L" "Vxj L" "Vxj L" ja-kanji)
("w" "S" "TJEm S" "TJEm S" ja-kanji)
("x" "WN" "WNW^m WN" "WNW^m WN" ja-kanji)
("y" "V" "}SRm V" "}SRm V" ja-kanji)
("z" "e" "J^WPiC" "J^WPiC" ja-kanji)
("{" "gN" "mm gN" "mm gN" ja-kanji)
("|" "t" "IRm t" "IRm t" ja-kanji)
("}" "C" "Rum C" "Rum C" ja-kanji)
("~" "JC" "AVCm CW" "AVCm CW" ja-kanji)
("" "R" "RWm R" "RWm R" ja-kanji)
("" "N" "bVxj N" "bVxj N" ja-kanji)
("" "W" "WNW^m W" "WNW^m W" ja-kanji)
("" "V" "CJfm V" "CJfm V" ja-kanji)
("" "\[" "jJm \[" "jJm \[" ja-kanji)
("" "^" "C^m ^" "C^m ^" ja-kanji)
("" "nN" "I\m nN" "I\m nN" ja-kanji)
("" "tc" "CJm tc" "CJm tc" ja-kanji)
("" "z[" "Vxj wC" "Vxj wC" ja-kanji)
("" "I[" "bVxj I[" "bVxj I[" ja-kanji)
("" "GC" "RGm GC" "RGm GC" ja-kanji)
("" "C" "Ctm C" "Ctm C" ja-kanji)
("" "j" "jm j" "jm j" ja-kanji)
("" "JN" "JNVNm JN" "JNVNm JN" ja-kanji)
("" "L[" "Xj RR" "Xj RR" ja-kanji)
("" "L[" "J}rXVC" "J}rXVC" ja-kanji)
("" "L[" "L[m ]NW" "L[m ]NW" ja-kanji)
("" "R[" "clm L[W" "clm L[W" ja-kanji)
("" "R[" "R[Rcm R[" "R[Rcm R[" ja-kanji)
("" "V" "zVC}}m V" "zVC}}m V" ja-kanji)
("" "W" "^mm W" "^mm W" ja-kanji)
("" "Wc" "Am Wc" "Am Wc" ja-kanji)
("" "V" "}Rgm V" "}Rgm V" ja-kanji)
("" "e" "XJm e" "XJm e" ja-kanji)
("" "h[" "IhXm h[" "IhXm h[" ja-kanji)
("" "[" "LKJm [" "LKJm [" ja-kanji)
("" "G" "CJm G" "CJm G" ja-kanji)
("" "J" "C}TVCm J" "C}TVCm J" ja-kanji)
("" "O" "bVxj O" "bVxj O" ja-kanji)
("" "R" "Rz[m R" "Rz[m R" ja-kanji)
("" "V[" "I\m V[" "I\m V[" ja-kanji)
("" "V[" "V[[m V[" "V[[m V[" ja-kanji)
("" "V" "A^m V" "A^m V" ja-kanji)
("" "nC" "gm nC" "gm nC" ja-kanji)
("" "o" "Vxj " "Vxj " ja-kanji)
("" "[" "EGm [" "EGm [" ja-kanji)
("" "" "JVRCm " "JVRCm " ja-kanji)
("" "" "l^m " "l^m " ja-kanji)
("" "AN" "Cm L[W" "Cm L[W" ja-kanji)
("" "L" "I\m L" "I\m L" ja-kanji)
("" "PC" "Om L[W" "Om L[W" ja-kanji)
("" "P" "Em P" "Em P" ja-kanji)
("" "XC" "ZKm XC" "ZKm XC" ja-kanji)
("" "XC" "V[XCm XC" "V[XCm XC" ja-kanji)
("" "ZC" "ZCTm ZC" "ZCTm ZC" ja-kanji)
("" "`[" "Em `[" "Em `[" ja-kanji)
("" "`[" "`[Rm `[" "`[Rm `[" ja-kanji)
("" "{[" "{[{[m {[" "{[{[m {[" ja-kanji)
("" "I" "CJm I" "CJm I" ja-kanji)
("" "KN" "IhNm KN" "IhNm KN" ja-kanji)
("" "P" "A}m P" "A}m P" ja-kanji)
("" "R[" "Aem R[" "Aem R[" ja-kanji)
("" "V" "IJm V" "IJm V" ja-kanji)
("" "V[" "Vxj AL" "Vxj AL" ja-kanji)
("" "YC" "I\m YC" "I\m YC" ja-kanji)
("" "ZC" "Vxj zV" "Vxj zV" ja-kanji)
("" "P" "Vxj Z" "Vxj Z" ja-kanji)
("" "\[" "bVxj \[" "bVxj \[" ja-kanji)
("" "\N" "\NCm \N" "\NCm \N" ja-kanji)
("" "m[" "im L[W" "im L[W" ja-kanji)
("" "~" "Am ~" "Am ~" ja-kanji)
("" "tN" "gm tN" "gm tN" ja-kanji)
("" "C" "lSm C" "lSm C" ja-kanji)
("" "KC" "bVxj L" "bVxj L" ja-kanji)
("" "JN" "}Rgm JN" "}Rgm JN" ja-kanji)
("" "L" "ULm L" "ULm L" ja-kanji)
("" "P" "AL^iC" "AL^iC" ja-kanji)
("" "Q" "ccVm Q" "ccVm Q" ja-kanji)
("" "V" "ccVm L[W" "ccVm L[W" ja-kanji)
("" "TN" "Eb^Gm TN" "Eb^Gm TN" ja-kanji)
("" "\[" "C^m \[" "C^m \[" ja-kanji)
("" "nN" "~NnNm nN" "~NnNm nN" ja-kanji)
("" "[" "XXm [" "XXm [" ja-kanji)
("" "c" "IrGm c" "IrGm c" ja-kanji)
("" "J" "uTJm J" "uTJm J" ja-kanji)
("" "R[" "iQNm R[" "iQNm R[" ja-kanji)
("" "T" "~Wm L[W" "~Wm L[W" ja-kanji)
("" "U" "nWm CW" "nWm CW" ja-kanji)
("" "U" "ULm U" "ULm U" ja-kanji)
("" "V[" "XXm V[" "XXm V[" ja-kanji)
("" "V[" "I\m V[" "I\m V[" ja-kanji)
("" "V[" "Vxj LY" "Vxj LY" ja-kanji)
("" "][" "^VJm ][" "^VJm ][" ja-kanji)
("" "^" "EGm ^" "EGm ^" ja-kanji)
("" "h[" "h[RNm h[" "h[RNm h[" ja-kanji)
("" "gN" "PKm gN" "PKm gN" ja-kanji)
("" "q[" "bVxj q[" "bVxj q[" ja-kanji)
("" "[" "mECm [" "mECm [" ja-kanji)
("" "L" "Ruj RR" "Ruj RR" ja-kanji)
("" "M" "VoNm M" "VoNm M" ja-kanji)
("" "PC" "CR[m ]NW" "CR[m ]NW" ja-kanji)
("" "PC" "ARKm PC" "ARKm PC" ja-kanji)
("" "V[" "cm V[" "cm V[" ja-kanji)
("" "^" "noJm ^" "noJm ^" ja-kanji)
("" "nC" "cJm nC" "cJm nC" ja-kanji)
("" "q[" "cNm q[" "cNm q[" ja-kanji)
("" "r" "Am r" "Am r" ja-kanji)
("" "u" "KbJXm u" "KbJXm u" ja-kanji)
("" "GL" "Rrm GL" "Rrm GL" ja-kanji)
("" "I[" "}km I[" "}km I[" ja-kanji)
("" "I[" "PCI[m I[" "PCI[m I[" ja-kanji)
("" "JC" "JCm L[W" "JCm L[W" ja-kanji)
("" "JC" "IR^m JC" "IR^m JC" ja-kanji)
("" "L" "cgm L" "cgm L" ja-kanji)
("" "\[" "bVxj \[" "bVxj \[" ja-kanji)
("" "^" "Vxj ^" "Vxj ^" ja-kanji)
("" "{[" "cgm {[" "cgm {[" ja-kanji)
("" "" "JJm " "JJm " ja-kanji)
("" "" "I\m " "I\m " ja-kanji)
("" "_" "INr[m _" "INr[m _" ja-kanji)
("" "}" "_Gm " "_Gm " ja-kanji)
("" "" "IR^m " "IR^m " ja-kanji)
("" "U" "NCm L[W" "NCm L[W" ja-kanji)
("" "U" "NCm U" "NCm U" ja-kanji)
("" "C" "CgNm C" "CgNm C" ja-kanji)
("" "J" "Rum J" "Rum J" ja-kanji)
("" "N" "N[m N" "N[m N" ja-kanji)
("" "V[" "I\m V[" "I\m V[" ja-kanji)
("" "" "RCm L[W" "RCm L[W" ja-kanji)
("" "J" "Jm zR" "Jm zR" ja-kanji)
("" "Gc" "}TJm Gc" "}TJm Gc" ja-kanji)
("" "W" "}m W" "}m W" ja-kanji)
("" "Wc" "W[jVm Ck" "W[jVm Ck" ja-kanji)
("" "T" "\RiEm T" "\RiEm T" ja-kanji)
("" "Jc" "RXm Jc" "RXm Jc" ja-kanji)
("@" "Jc" "Jcm ]NW" "Jcm ]NW" ja-kanji)
("A" "J" "RXm J" "RXm J" ja-kanji)
("B" "Zc" "Zc[m Zc" "Zc[m Zc" ja-kanji)
("C" "N" "TcNm N" "TcNm N" ja-kanji)
("D" "Z" "^^JEm L[W" "^^JEm L[W" ja-kanji)
("E" "M" "^m L[W" "^m L[W" ja-kanji)
("F" "^N" "TXm ^N" "TXm ^N" ja-kanji)
("G" "w" "wg[m w" "wg[m w" ja-kanji)
("H" "Tc" "ewj co" "ewj co" ja-kanji)
("I" "J" "tZOm J" "tZOm J" ja-kanji)
("J" "R[" "^^Nm R[" "^^Nm R[" ja-kanji)
("K" "R[" "AQm R[" "AQm R[" ja-kanji)
("L" "T" "Xm T" "Xm T" ja-kanji)
("M" "W" "RNWm Te" "RNWm Te" ja-kanji)
("N" "N" "NTcm N" "NTcm N" ja-kanji)
("O" "L[" "ewj I[" "ewj I[" ja-kanji)
("P" "Pc" "GOm Pc" "GOm Pc" ja-kanji)
("Q" "\[" "TKXm \[" "TKXm \[" ja-kanji)
("R" "W" "WW[m W" "WW[m W" ja-kanji)
("S" "\[" "c}m \[" "c}m \[" ja-kanji)
("T" "g" "g\[m g" "g\[m g" ja-kanji)
("U" "oc" "kNm L[W" "kNm L[W" ja-kanji)
("V" "x" "Ecm x" "Ecm x" ja-kanji)
("W" "z[" "z[hm z[" "z[hm z[" ja-kanji)
("X" "[" "Xlm [" "Xlm [" ja-kanji)
("Y" "J" "cOm J" "cOm J" ja-kanji)
("Z" "V" "ewj [X" "ewj [X" ja-kanji)
("[" "_" "gGm _" "gGm _" ja-kanji)
("\" "_" "_zXm _" "_zXm _" ja-kanji)
("]" "^N" "qNm ^N" "qNm ^N" ja-kanji)
("^" "^" "jiEm L[W" "jiEm L[W" ja-kanji)
("_" "f" "qm f" "qm f" ja-kanji)
("`" "nC" "IKm L[W" "IKm L[W" ja-kanji)
("a" "n" "JNnm n" "JNnm n" ja-kanji)
("b" "t" "ifm t" "ifm t" ja-kanji)
("c" "tc" "nEm L[W" "nEm L[W" ja-kanji)
("d" "{" "Irm {" "Irm {" ja-kanji)
("e" "z[" "iQXem z[" "iQXem z[" ja-kanji)
("f" "" "`Xm " "`Xm " ja-kanji)
("g" "JN" "Ecm JN" "Ecm JN" ja-kanji)
("h" "Lc" "nAEm Lc" "nAEm Lc" ja-kanji)
("i" "L[" "R}lNm L[" "R}lNm L[" ja-kanji)
("j" "E" "ewj n" "ewj n" ja-kanji)
("k" "PC" "JPm PC" "JPm PC" ja-kanji)
("l" "Pc" "qbTQm Pc" "qbTQm Pc" ja-kanji)
("m" "W[" "^XPm W[" "^XPm W[" ja-kanji)
("n" "\" "VcGm \" "VcGm \" ja-kanji)
("o" "G" "GLm G" "GLm G" ja-kanji)
("p" "L[" "nTm L[" "nTm L[" ja-kanji)
("q" "J" "tZOm J" "tZOm J" ja-kanji)
("r" "\[" "TKXm L[W" "TKXm L[W" ja-kanji)
("s" "lc" "lc][m lc" "lc][m lc" ja-kanji)
("t" "GL" "^XPm GL" "^XPm GL" ja-kanji)
("u" "L" "ewj YVC" "ewj YVC" ja-kanji)
("v" "L" "AQm L" "AQm L" ja-kanji)
("w" "V" "ewj g" "ewj g" ja-kanji)
("x" "XC" "` Ecm XC" "` Ecm XC" ja-kanji)
("y" "ZC" "qNm ZC" "qNm ZC" ja-kanji)
("z" "g[" "Xm X" "Xm X" ja-kanji)
("{" "g[" "tm g[" "tm g[" ja-kanji)
("|" "eC" "ILem eC" "ILem eC" ja-kanji)
("}" "C" "ewj Cm `" "ewj Cm `" ja-kanji)
("~" "" "qm " "qm " ja-kanji)
("" "c" "lWm c" "lWm c" ja-kanji)
("" "G" "G]Nm G" "G]Nm G" ja-kanji)
("" "JC" "JCVm JC" "JCVm JC" ja-kanji)
("" "J" "Gum J" "Gum J" ja-kanji)
("" "L" "qNV[ CbLm L" "qNV[ CbLm L" ja-kanji)
("" "V" "nJm V" "nJm V" ja-kanji)
("" "W[" "m W[" "m W[" ja-kanji)
("" "\[" "TXm L[W" "TXm L[W" ja-kanji)
("" "" "JJEm " "JJEm " ja-kanji)
("" "" "JJEm " "JJEm " ja-kanji)
("" "[" "m L[W" "m L[W" ja-kanji)
("" "P" "gm P" "gm P" ja-kanji)
("" "R[" "J}Gm R[" "J}Gm R[" ja-kanji)
("" "T" "ewj T" "ewj T" ja-kanji)
("" "WN" "Jm WN" "Jm WN" ja-kanji)
("" "\[" "ewj N" "ewj N" ja-kanji)
("" "Zc" "ITm L[W" "ITm L[W" ja-kanji)
("" "g[" "ewj V}" "ewj V}" ja-kanji)
("" "g[" "Xm g[" "Xm g[" ja-kanji)
("" "nN" "Ecm nN" "Ecm nN" ja-kanji)
("" "TC" "N_Nm TC" "N_Nm TC" ja-kanji)
("" "V" "gm V" "gm V" ja-kanji)
("" "^" "Ecm ^" "Ecm ^" ja-kanji)
("" "R[" "NNm R[" "NNm R[" ja-kanji)
("" "JN" "~_Xm L[W" "~_Xm L[W" ja-kanji)
("" "ZC" "ewj V" "ewj V" ja-kanji)
("" "h[" "^m h[" "^m h[" ja-kanji)
("" "nc" "o`m nc" "o`m nc" ja-kanji)
("" "[" "~_m [" "~_m [" ja-kanji)
("" "[" "gm [" "gm [" ja-kanji)
("" "J" "ESJXm J" "ESJXm J" ja-kanji)
("" "L" "m L[W" "m L[W" ja-kanji)
("" "L" "gRm L" "gRm L" ja-kanji)
("" "Z" "zVC}}m Z" "zVC}}m Z" ja-kanji)
("" "^N" "Gum L[W" "Gum L[W" ja-kanji)
("" "^c" "` Ecm ^c" "` Ecm ^c" ja-kanji)
("" "nN" "cUNm nN" "cUNm nN" ja-kanji)
("" "C" "ewj J~i" "ewj J~i" ja-kanji)
("" "JN" "tf INm IN" "tf INm IN" ja-kanji)
("" "L" "AQm L[W" "AQm L[W" ja-kanji)
("" "L" "AQm CW" "AQm CW" ja-kanji)
("" "ZC" "IXm ZC" "IXm ZC" ja-kanji)
("" "^C" "^Qm L[W" "^Qm L[W" ja-kanji)
("" "^C" "^Qm _C" "^Qm _C" ja-kanji)
("" "g[" "Ecm g[" "Ecm g[" ja-kanji)
("" "q" "V]Pm q" "V]Pm q" ja-kanji)
("" "" "gm " "gm " ja-kanji)
("" "Z" "^_Xm Z" "^_Xm Z" ja-kanji)
("" "JN" "qQm L[W" "qQm L[W" ja-kanji)
("" "eL" "eL_m eL" "eL_m eL" ja-kanji)
("" "nC" "qNm nC" "qNm nC" ja-kanji)
("" "n" "Wm{m n" "Wm{m n" ja-kanji)
("" "N" "Ecm N" "Ecm N" ja-kanji)
("" "W[" "W[Cm W[" "W[Cm W[" ja-kanji)
("" "PC" "^YTGm L[W" "^YTGm L[W" ja-kanji)
("" "T" "Ac}m T" "Ac}m T" ja-kanji)
("" "^" "qNm ^" "qNm ^" ja-kanji)
("" "" "qLcm " "qLcm " ja-kanji)
("" "JN" "cJm JN" "cJm JN" ja-kanji)
("" "{N" "g}^m {N" "g}^m {N" ja-kanji)
("" "{N" "mum {N" "mum {N" ja-kanji)
("" "R[" "JKGm RW" "JKGm RW" ja-kanji)
("" "V[" "ITm L[W" "ITm L[W" ja-kanji)
("" "[" "gRm [" "gRm [" ja-kanji)
("" "e" "^jm u" "^jm u" ja-kanji)
("" "R[" "LNm L[W" "LNm L[W" ja-kanji)
("" "S[" "A\um S[" "A\um S[" ja-kanji)
("" "`N" "C}Vm L[W" "C}Vm L[W" ja-kanji)
("" "W" "mxm ZCW" "mxm ZCW" ja-kanji)
("" "W" "mxm L[W" "mxm L[W" ja-kanji)
("" "V[" "qPm V[" "qPm V[" ja-kanji)
("" "wC" "um wC" "um wC" ja-kanji)
("" "R[" "XCR[m R[" "XCR[m R[" ja-kanji)
("" "X[" "JYm L[W" "JYm L[W" ja-kanji)
("" "" "ITm " "ITm " ja-kanji)
("" "wC" "^Im wC" "^Im wC" ja-kanji)
("" "w" "Jm L[W" "Jm L[W" ja-kanji)
("" "RN" "}Xm RN" "}Xm RN" ja-kanji)
("" "V" "Nm V" "Nm V" ja-kanji)
("" "VN" "Lm VN" "Lm VN" ja-kanji)
("" "_" "^cm L[W" "^cm L[W" ja-kanji)
("" "Z" "Z_m Z" "Z_m Z" ja-kanji)
("" "nC" "n^m nC" "n^m nC" ja-kanji)
("" "z[" "cNm z[" "cNm z[" ja-kanji)
("" "{[" "{[M[m {[" "{[M[m {[" ja-kanji)
("" "ZC" "n^m ZC" "n^m ZC" ja-kanji)
("" "[" "iKm [" "iKm [" ja-kanji)
("" "n" "nm ]NW" "nm ]NW" ja-kanji)
("" "n" "n^m n" "n^m n" ja-kanji)
("" "" "iCm RW" "iCm RW" ja-kanji)
("" "L" "Xfm cN" "Xfm cN" ja-kanji)
("" "J" "qfm J" "qfm J" ja-kanji)
("" "R[" "ALJm R[" "ALJm R[" ja-kanji)
("" "R[" "\m R[" "\m R[" ja-kanji)
("" "\N" "J^Nm \N" "J^Nm \N" ja-kanji)
("" "r" "\m r" "\m r" ja-kanji)
("" "[" "nJm [" "nJm [" ja-kanji)
("" "Wc" "`JdNm Wc" "`JdNm Wc" ja-kanji)
("" "`[" "qTVCm `[" "qTVCm `[" ja-kanji)
("" "{[" "Xom {[" "Xom {[" ja-kanji)
("" "[" "RUgiVm [" "RUgiVm [" ja-kanji)
("" "A" "AY~mm AY" "AY~mm AY" ja-kanji)
("" "R[" "qJm L[W" "qJm L[W" ja-kanji)
("" "V" "XXm ZCW" "XXm ZCW" ja-kanji)
("" "`[" "ZCm qgc" "ZCm qgc" ja-kanji)
("" "L" "JNm L" "JNm L" ja-kanji)
("" "`[" "qm L[W" "qm L[W" ja-kanji)
("" "S" "qwj " "qwj " ja-kanji)
("" "R[" "qwj cQ" "qwj cQ" ja-kanji)
("" "V" "AV^m V" "AV^m V" ja-kanji)
("" "ZC" "ALJm ZC" "ALJm ZC" ja-kanji)
("" "ZL" "ALJm ZL" "ALJm ZL" ja-kanji)
("" "ZL" "CZLm ZL" "CZLm ZL" ja-kanji)
("" "q" "qj AY" "qj AY" ja-kanji)
("" "E" "JTm E" "JTm E" ja-kanji)
("" "GC" "qwj GC" "qwj GC" ja-kanji)
("" "L" "JKNm L" "JKNm L" ja-kanji)
("" "P" "A^^Jm P" "A^^Jm P" ja-kanji)
("" "[" "qmfm [" "qmfm [" ja-kanji)
("" "C" "~m C" "~m C" ja-kanji)
("" "L" "Irm L" "Irm L" ja-kanji)
("" "Z" "XXm Z" "XXm Z" ja-kanji)
("" "M[" "AJcLm L[W" "AJcLm L[W" ja-kanji)
("" "g" "qwj g" "qwj g" ja-kanji)
("" "xc" "C`xcm xc" "C`xcm xc" ja-kanji)
("@" "[" "JKNm [" "JKNm [" ja-kanji)
("A" "[" "ALJm CW" "ALJm CW" ja-kanji)
("B" "AC" "AC}Cm AC" "AC}Cm AC" ja-kanji)
("C" "[" "[}Cm [" "[}Cm [" ja-kanji)
("D" "R[" "R[hm R[" "R[hm R[" ja-kanji)
("E" "R[" "R[m NW" "R[m NW" ja-kanji)
("F" "M" "^C[m M" "^C[m M" ja-kanji)
("G" "m[" "TLjm m[" "TLjm m[" ja-kanji)
("H" "Gc" "CNm Gc" "CNm Gc" ja-kanji)
("I" "GC" "qNm ]NW" "qNm ]NW" ja-kanji)
("J" "Jc" "CYN]m Jc" "CYN]m Jc" ja-kanji)
("K" "q" "~JdLm q" "~JdLm q" ja-kanji)
("L" "[" "zKJm L[W" "zKJm L[W" ja-kanji)
("M" "L" "Sm L[W" "Sm L[W" ja-kanji)
("N" "[" "I{m [" "I{m [" ja-kanji)
("O" "[" "[[m [" "[[m [" ja-kanji)
("P" "n" "n^KVm L[W" "n^KVm L[W" ja-kanji)
("Q" "Wc" "EPm Wc" "EPm Wc" ja-kanji)
("R" "V" "gQm V" "gQm V" ja-kanji)
("S" "_" "ESJXm _" "ESJXm _" ja-kanji)
("T" "j[" "Lwj nC" "Lwj nC" ja-kanji)
("U" "N" "Lwj `J" "Lwj `J" ja-kanji)
("V" "g[" "Lwj J^i" "Lwj J^i" ja-kanji)
("W" "J" "eRm J" "eRm J" ja-kanji)
("X" "L" "L[m L" "L[m L" ja-kanji)
("Y" "R[" "nVm R[" "nVm R[" ja-kanji)
("Z" "N" "NCm N" "NCm N" ja-kanji)
("[" "T" "Lwj }" "Lwj }" ja-kanji)
("\" "}" "Lm g`m CW" "Lm g`m CW" ja-kanji)
("]" "I[" "Lwj I[" "Lwj I[" ja-kanji)
("^" "Pc" "Pcm ]NW" "Pcm ]NW" ja-kanji)
("_" "V[" "Lm }cm CW" "Lm }cm CW" ja-kanji)
("`" "`" "NkMm ~" "NkMm ~" ja-kanji)
("a" "r[" "RYGm r[" "RYGm r[" ja-kanji)
("b" "t" "Lwj P" "Lwj P" ja-kanji)
("c" "z[" "Lwj z[" "Lwj z[" ja-kanji)
("d" "" "Lwj g" "Lwj g" ja-kanji)
("e" "V[" "Lwj }X" "Lwj }X" ja-kanji)
("f" "PC" "Lwj PC" "Lwj PC" ja-kanji)
("g" "J" "JVm J" "JVm J" ja-kanji)
("h" "J" "G_m J" "G_m J" ja-kanji)
("i" "JC" "cGm JC" "cGm JC" ja-kanji)
("j" "J" "Gum J" "Gum J" ja-kanji)
("k" "L" "Lm J^`" "Lm J^`" ja-kanji)
("l" "L[" "qcMm L[" "qcMm L[" ja-kanji)
("m" "N" "Lm P{iV" "Lm P{iV" ja-kanji)
("n" "T" "Lm R{P" "Lm R{P" ja-kanji)
("o" "TN" "Lm TN" "Lm TN" ja-kanji)
("p" "^N" "q[VMm ^N" "q[VMm ^N" ja-kanji)
("q" "eC" "lm eC" "lm eC" ja-kanji)
("r" "gc" "Lwj f" "Lwj f" ja-kanji)
("s" "zI" "Lm z" "Lm z" ja-kanji)
("t" "t" "Lwj cN" "Lwj cN" ja-kanji)
("u" "[" "Lwj ^c" "Lwj ^c" ja-kanji)
("v" "R" "Lwj E" "Lwj E" ja-kanji)
("w" "JC" "qm Lm L[W" "qm Lm L[W" ja-kanji)
("x" "J" "VIm J" "VIm J" ja-kanji)
("y" "L[" "J}`m L[" "J}`m L[" ja-kanji)
("z" "N" "Lwj n" "Lwj n" ja-kanji)
("{" "Pc" "ncPm Pc" "ncPm Pc" ja-kanji)
("|" "R" "Lwj nJ}" "Lwj nJ}" ja-kanji)
("}" "R[" "Lwj R[" "Lwj R[" ja-kanji)
("~" "Vc" "Lwj C^" "Lwj C^" ja-kanji)
("" "\" "NVm \" "NVm \" ja-kanji)
("" "Z" "Lwj \" "Lwj \" ja-kanji)
("" "{[" "zRm {[" "zRm {[" ja-kanji)
("" "g[" "Lwj A^" "Lwj A^" ja-kanji)
("" "JN" "^Lm JN" "^Lm JN" ja-kanji)
("" "J" "Lm J" "Lm J" ja-kanji)
("" "L[" "gm tN[" "gm tN[" ja-kanji)
("" "RN" "Lwj cQ" "Lwj cQ" ja-kanji)
("" "T" "qm T" "qm T" ja-kanji)
("" "V" "Lm N`iV" "Lm N`iV" ja-kanji)
("" "W[" "W[m L[W" "W[m L[W" ja-kanji)
("" "_" "Lm iM" "Lm iM" ja-kanji)
("" "eC" "eRm e" "eRm e" ja-kanji)
("" "g[" "LJum L[W" "LJum L[W" ja-kanji)
("" "r" "Lwj wC" "Lwj wC" ja-kanji)
("" "t" "CJ_m t" "CJ_m t" ja-kanji)
("" "{" "{m[m {" "{m[m {" ja-kanji)
("" "" "Lwj " "Lwj " ja-kanji)
("" "" "nVj V^" "nVj V^" ja-kanji)
("" "A" "Lwj A" "Lwj A" ja-kanji)
("" "G" "J}m Gm AE" "J}m Gm AE" ja-kanji)
("" "N" "Lwj L~" "Lwj L~" ja-kanji)
("" "JN" "qcMm JN" "qcMm JN" ja-kanji)
("" "L" "V[Mm CW" "V[Mm CW" ja-kanji)
("" "LN" "Lwj LN" "Lwj LN" ja-kanji)
("" "LN" "gQm LN" "gQm LN" ja-kanji)
("" "JN" "Lwj Nj" "Lwj Nj" ja-kanji)
("" "P" "Lwj P" "Lwj P" ja-kanji)
("" "R[" "Lwj IJ" "Lwj IJ" ja-kanji)
("" "R[" "Lwj \" "Lwj \" ja-kanji)
("" "R" "R{[m R" "R{[m R" ja-kanji)
("" "R" "Lwj R" "Lwj R" ja-kanji)
("" "T" "ToVm T" "ToVm T" ja-kanji)
("" "V" "Vm V" "Vm V" ja-kanji)
("" "V" "Vm CW" "Vm CW" ja-kanji)
("" "V[" "RV[m V[" "RV[m V[" ja-kanji)
("" "Zc" "cOm Zc" "cOm Zc" ja-kanji)
("" "\[" "Lm ic" "Lm ic" ja-kanji)
("" "eC" "jEm eC" "jEm eC" ja-kanji)
("" "`" "Lwj V" "Lwj V" ja-kanji)
("" "g[" "TIm g[" "TIm g[" ja-kanji)
("" "g[" "Lm }iV" "Lm }iV" ja-kanji)
("" "j" "Lwj l" "Lwj l" ja-kanji)
("" "g[" "Lwj t" "Lwj t" ja-kanji)
("" "wC" "Lwj iu" "Lwj iu" ja-kanji)
("" "L[" "Lm NkM" "Lm NkM" ja-kanji)
("" "e" "Lwj e" "Lwj e" ja-kanji)
("" "g[" "Lwj C^" "Lwj C^" ja-kanji)
("" "" "NXm Lm ^OC" "NXm Lm ^OC" ja-kanji)
("" "GC" "nVm GC" "nVm GC" ja-kanji)
("" "JC" "JCVm JC" "JCVm JC" ja-kanji)
("" "R" "Lwj R" "Lwj R" ja-kanji)
("" "V[" "Lwj AL" "Lwj AL" ja-kanji)
("" "V[" "JWm V[" "JWm V[" ja-kanji)
("" "Zc" "NTrm Zc" "NTrm Zc" ja-kanji)
("" "Z" "Lwj CY~" "Lwj CY~" ja-kanji)
("" "`" "Lm R[]" "Lm R[]" ja-kanji)
("" "W" "Lm T" "Lm T" ja-kanji)
("" "eC" "Lwj eC" "Lwj eC" ja-kanji)
("" "e" "^Lm e" "^Lm e" ja-kanji)
("" "{[" "TJGm {[" "TJGm {[" ja-kanji)
("" "" "Vm ~m " "Vm ~m " ja-kanji)
("" "" "Lm j" "Lm j" ja-kanji)
("" "K" "[Km K" "[Km K" ja-kanji)
("" "" "Lm I[`" "Lm I[`" ja-kanji)
("" "Vc" "Lwj Vc" "Lwj Vc" ja-kanji)
("" "[" "Lwj n" "Lwj n" ja-kanji)
("" "I" "Lwj I" "Lwj I" ja-kanji)
("" "GC" "TJGm L[W" "TJGm L[W" ja-kanji)
("" "JC" "Lm ViGW" "Lm ViGW" ja-kanji)
("" "KC" "nm Lm KC" "nm Lm KC" ja-kanji)
("" "R[" "Lwj ^JC" "Lwj ^JC" ja-kanji)
("" "R[" "R[Jm R[" "R[Jm R[" ja-kanji)
("" "Rc" "Lwj zl" "Lwj zl" ja-kanji)
("" "T" "Lwj T" "Lwj T" ja-kanji)
("" "TC" "gfm TC" "gfm TC" ja-kanji)
("" "TN" "zRm TN" "zRm TN" ja-kanji)
("" "g[" "Lwj V}" "Lwj V}" ja-kanji)
("" "g[" "RVJPm g[" "RVJPm g[" ja-kanji)
("" "n" "^Cm n" "^Cm n" ja-kanji)
("" "q" "Lm J" "Lm J" ja-kanji)
("" "G" "Lj P" "Lj P" ja-kanji)
("" "t" "Nm t" "Nm t" ja-kanji)
("" "xC" "Lwj C" "Lwj C" ja-kanji)
("" "{[" "t_m {[" "t_m {[" ja-kanji)
("" "[" "Lwj [" "Lwj [" ja-kanji)
("" "[" "Lm UN" "Lm UN" ja-kanji)
("" "[" "Lwj ^c" "Lwj ^c" ja-kanji)
("" "JN" "qcMm JN" "qcMm JN" ja-kanji)
("" "N" "^mVCm L[W" "^mVCm L[W" ja-kanji)
("" "L[" "}Km L[" "}Km L[" ja-kanji)
("" "L" "Lm NQ" "Lm NQ" ja-kanji)
("" "P" "Pm L[W" "Pm L[W" ja-kanji)
("" "R[" "Lwj R[" "Lwj R[" ja-kanji)
("" "RN" "Lm JV" "Lm JV" ja-kanji)
("" "[" "t_m [" "t_m [" ja-kanji)
("" "V[" "Lm ~" "Lm ~" ja-kanji)
("" "XC" "Lwj IgG" "Lwj IgG" ja-kanji)
("" "X[" "Jim L[W" "Jim L[W" ja-kanji)
("" "VN" "JGfm ^OC" "JGfm ^OC" ja-kanji)
("" "\[" "Lwj X" "Lwj X" ja-kanji)
("" "^" "qcMVm ^" "qcMVm ^" ja-kanji)
("" "n" "gJSm n" "gJSm n" ja-kanji)
("" "~c" "Lwj ~c" "Lwj ~c" ja-kanji)
("" "~c" "~cm CW" "~cm CW" ja-kanji)
("" "[" "T}m L[W" "T}m L[W" ja-kanji)
("" "[" "Om L[W" "Om L[W" ja-kanji)
("" "J" "Lwj J" "Lwj J" ja-kanji)
("" "J" "Lwj ckN" "Lwj ckN" ja-kanji)
("" "L" "Lwj Ru" "Lwj Ru" ja-kanji)
("" "TC" "Lwj TC" "Lwj TC" ja-kanji)
("" "V[" "Lwj V[" "Lwj V[" ja-kanji)
("" "[C" "\m [C" "\m [C" ja-kanji)
("" "_" "_Gm L[W" "_Gm L[W" ja-kanji)
("" "g[" "_C_Cm g[" "_C_Cm g[" ja-kanji)
("" "h[" "Lwj h[" "Lwj h[" ja-kanji)
("" "h[" "^m h[" "^m h[" ja-kanji)
("" "{N" "Lwj {N" "Lwj {N" ja-kanji)
("" "`[" "Lwj g" "Lwj g" ja-kanji)
("" "G" "qTVm G" "qTVm G" ja-kanji)
("" "IN" "Lm `" "Lm `" ja-kanji)
("" "PC" "gVrm PC" "gVrm PC" ja-kanji)
("" "QL" "QLum QL" "QLum QL" ja-kanji)
("" "P" "Vxm L[W" "Vxm L[W" ja-kanji)
("" "V[" "zoVm V[" "zoVm V[" ja-kanji)
("@" "nN" "Ln_m nN" "Ln_m nN" ja-kanji)
("A" "nN" "nNm CW" "nNm CW" ja-kanji)
("B" "J" "Im J" "Im J" ja-kanji)
("C" "L" "qcm L" "qcm L" ja-kanji)
("D" "g[" "Lm JC" "Lm JC" ja-kanji)
("E" "C" "m " "m " ja-kanji)
("F" "r" "r[Wm r" "r[Wm r" ja-kanji)
("G" "[" "m " "m " ja-kanji)
("H" "G" "Lwj G" "Lwj G" ja-kanji)
("I" "C" "^m C" "^m C" ja-kanji)
("J" "L" "Lm NkM" "Lm NkM" ja-kanji)
("K" "L" "Lm CW" "Lm CW" ja-kanji)
("L" "" "Vm " "Vm " ja-kanji)
("M" "L" "Lwj L" "Lwj L" ja-kanji)
("N" "I[" "TNm L[W" "TNm L[W" ja-kanji)
("O" "L" "Lm PL" "Lm PL" ja-kanji)
("P" "nN" "nNm CW" "nNm CW" ja-kanji)
("Q" "C" "Wm C" "Wm C" ja-kanji)
("R" "" "_m " "_m " ja-kanji)
("S" "" "Lwj " "Lwj " ja-kanji)
("T" "Ec" "VQm L[W" "VQm L[W" ja-kanji)
("U" "cL" "Lm cL" "Lm cL" ja-kanji)
("V" "AC" "iQNm AC" "iQNm AC" ja-kanji)
("W" "L" "XXiNm L" "XXiNm L" ja-kanji)
("X" "g[" "kXm L[W" "kXm L[W" ja-kanji)
("Y" "C" "\o_em C" "\o_em C" ja-kanji)
("Z" "C" "mm L[W" "mm L[W" ja-kanji)
("[" "Pc" "Xm Pc" "Xm Pc" ja-kanji)
("\" "\[" "XXm \[" "XXm \[" ja-kanji)
("]" "P" "AL^iC" "AL^iC" ja-kanji)
("^" "I[" "nNm L[W" "nNm L[W" ja-kanji)
("_" "L[" "Ac}m L[" "Ac}m L[" ja-kanji)
("`" "L" "XXiNm L" "XXiNm L" ja-kanji)
("a" "J" "m]m J" "m]m J" ja-kanji)
("b" "" "Jm " "Jm " ja-kanji)
("c" "J" "Rum L[W" "Rum L[W" ja-kanji)
("d" "L" "JGm L[W" "JGm L[W" ja-kanji)
("e" "Kc" "Vkm Kc" "Vkm Kc" ja-kanji)
("f" "{c" "iNim {c" "iNim {c" ja-kanji)
("g" "[" "JWjm [" "JWjm [" ja-kanji)
("h" "e" "cNXm e" "cNXm e" ja-kanji)
("i" "[" "UCm [" "UCm [" ja-kanji)
("j" "q[" "C`^wj t" "C`^wj t" ja-kanji)
("k" "U" "mRm L[W" "mRm L[W" ja-kanji)
("l" "t[" "NTm t[" "NTm t[" ja-kanji)
("m" "C" "I`m C" "I`m C" ja-kanji)
("n" "V[" "JWjm V[" "JWjm V[" ja-kanji)
("o" "GC" "^IXm GC" "^IXm GC" ja-kanji)
("p" "^" "cNXm ^" "cNXm ^" ja-kanji)
("q" "q" "qL[m q" "qL[m q" ja-kanji)
("r" "Z" "Zm ZCW" "Zm ZCW" ja-kanji)
("s" "Z" "z{Xm Z" "z{Xm Z" ja-kanji)
("t" "V" "}^m V" "}^m V" ja-kanji)
("u" "C" "Cm Njm C" "Cm Njm C" ja-kanji)
("v" "RN" "Jm L[W" "Jm L[W" ja-kanji)
("w" "I[" "iOm L[W" "iOm L[W" ja-kanji)
("x" "u" "iKm u" "iKm u" ja-kanji)
("y" "CN" "\_cm CN" "\_cm CN" ja-kanji)
("z" "Z" "P kNm V" "P kNm V" ja-kanji)
("{" "L[" "}m L[" "}m L[" ja-kanji)
("|" "S[" "tfm TLm S[" "tfm TLm S[" ja-kanji)
("}" "[C" "jRQm [C" "jRQm [C" ja-kanji)
("~" "^" "PImm ^" "PImm ^" ja-kanji)
("" "L" "LJm L" "LJm L" ja-kanji)
("" "Z" "[Zm Z" "[Zm Z" ja-kanji)
("" "{[" "\[{[m {[" "\[{[m {[" ja-kanji)
("" "L" "LK}Gm L" "LK}Gm L" ja-kanji)
("" "t" "UCm t" "UCm t" ja-kanji)
("" "C" "CR[m C" "CR[m C" ja-kanji)
("" "L" "CLm L[W" "CLm L[W" ja-kanji)
("" "R[" "XCMm R[" "XCMm R[" ja-kanji)
("" "T" "TYCj }" "TYCj }" ja-kanji)
("" "g" "TYCj c`" "TYCj c`" ja-kanji)
("" "I[" "TYCj I[" "TYCj I[" ja-kanji)
("" "L" "t`m L" "t`m L" ja-kanji)
("" "S" "TYCj S" "TYCj S" ja-kanji)
("" "V" "iMTm V" "iMTm V" ja-kanji)
("" "V" "V~m V" "V~m V" ja-kanji)
("" "nC" "TYCj C`" "TYCj C`" ja-kanji)
("" "t" "TYCj P" "TYCj P" ja-kanji)
("" "xL" "TYCj q" "TYCj q" ja-kanji)
("" "w" "TYCj n" "TYCj n" ja-kanji)
("" "{c" "{cm CW" "{cm CW" ja-kanji)
("" "N" "NNm N" "NNm N" ja-kanji)
("" "Zc" "nCZcm Zc" "nCZcm Zc" ja-kanji)
("" "I[" "TYCj I[" "TYCj I[" ja-kanji)
("" "I[" "~Y^}m I[" "~Y^}m I[" ja-kanji)
("" "R" "RPm R" "RPm R" ja-kanji)
("" "V" "TYCj " "TYCj " ja-kanji)
("" "V[" "TYCj V[" "TYCj V[" ja-kanji)
("" "\" "TJm{m \" "TJm{m \" ja-kanji)
("" "V" "nom V" "nom V" ja-kanji)
("" "_" "CPm _" "CPm _" ja-kanji)
("" "e" "EIEm e" "EIEm e" ja-kanji)
("" "e" "TYCj ^" "TYCj ^" ja-kanji)
("" "{[" "TYCj {[" "TYCj {[" ja-kanji)
("" "~" "zrm ~" "zrm ~" ja-kanji)
("" "z[" "TYCj ^C" "TYCj ^C" ja-kanji)
("" "C" "i~_m CW" "i~_m CW" ja-kanji)
("" "C" "ni~Ym C" "ni~Ym C" ja-kanji)
("" "G" "tGm G" "tGm G" ja-kanji)
("" "L[" "Nm L[" "Nm L[" ja-kanji)
("" "LN" "~]m LN" "~]m LN" ja-kanji)
("" "R[" "A}lNm R[" "A}lNm R[" ja-kanji)
("" "R[" "TYCj qJ" "TYCj qJ" ja-kanji)
("" "V" "TYCj V" "TYCj V" ja-kanji)
("" "W" "}Rgm W" "}Rgm W" ja-kanji)
("" "W" "TYCj m" "TYCj m" ja-kanji)
("" "V" "Vm V" "Vm V" ja-kanji)
("" "c" "TYCj c" "TYCj c" ja-kanji)
("" "J" "J`[m J" "J`[m J" ja-kanji)
("" "P" "PWm P" "PWm P" ja-kanji)
("" "R[" "TYCj qV" "TYCj qV" ja-kanji)
("" "V" "TEm V" "TEm V" ja-kanji)
("" "V[" "A}lVm V[" "A}lVm V[" ja-kanji)
("" "Zc" "TYCj I" "TYCj I" ja-kanji)
("" "Z" "_m G" "_m G" ja-kanji)
("" "eC" "i~_m eC" "i~_m eC" ja-kanji)
("" "g[" "i~m L[W" "i~m L[W" ja-kanji)
("" "l" "lnm l" "lnm l" ja-kanji)
("" "G" "Cm G" "Cm G" ja-kanji)
("" "G" "t`m CW" "t`m CW" ja-kanji)
("" "G" "t`m L[W" "t`m L[W" ja-kanji)
("" "J" "q^Xm J" "q^Xm J" ja-kanji)
("" "L" "LXCm L" "LXCm L" ja-kanji)
("" "J" "TYCj L" "TYCj L" ja-kanji)
("" "R" "qAKm R" "qAKm R" ja-kanji)
("" "R[" "}Wm R[" "}Wm R[" ja-kanji)
("" "TC" "TCCm TC" "TCCm TC" ja-kanji)
("" "V[" "TYCj }c" "TYCj }c" ja-kanji)
("" "V[" "IIi~m V[" "IIi~m V[" ja-kanji)
("" "W[" "LCm L[W" "LCm L[W" ja-kanji)
("" "ZC" "XSCm CW" "XSCm CW" ja-kanji)
("" "ZL" "gOm ZL" "gOm ZL" ja-kanji)
("" "Z" "ATCm L[W" "ATCm L[W" ja-kanji)
("" "\[" "\\Om \[" "\\Om \[" ja-kanji)
("" "I" "hm I" "hm I" ja-kanji)
("" "N" "TYCm N" "TYCm N" ja-kanji)
("" "" "VYm " "VYm " ja-kanji)
("" "C" "GiWm G" "GiWm G" ja-kanji)
("" "C" "TYCj C" "TYCj C" ja-kanji)
("" "C" "Ccm C" "Ccm C" ja-kanji)
("" "J" "TYCj J" "TYCj J" ja-kanji)
("" "J" "Jpcm J" "Jpcm J" ja-kanji)
("" "G" "LCm G" "LCm G" ja-kanji)
("" "R[" "TYCj R[" "TYCj R[" ja-kanji)
("" "R" "RVm R" "RVm R" ja-kanji)
("" "T" "JXm T" "JXm T" ja-kanji)
("" "V[" "TYCj AL" "TYCj AL" ja-kanji)
("" "Zc" "TEm Zc" "TEm Zc" ja-kanji)
("" "Z" "TYCj CY~" "TYCj CY~" ja-kanji)
("" "^" "nZm ^" "nZm ^" ja-kanji)
("" "eC" "gh}m eC" "gh}m eC" ja-kanji)
("" "nC" "TYCj IK" "TYCj IK" ja-kanji)
("" "r[" "nJm r[" "nJm r[" ja-kanji)
("" "" "TYCj " "TYCj " ja-kanji)
("" "{c" "{bJCm {c" "{bJCm {c" ja-kanji)
("" "}" "~cm L[W" "~cm L[W" ja-kanji)
("" "" "Jm " "Jm " ja-kanji)
("" "[" "IOm [" "IOm [" ja-kanji)
("" "c" "nccm c" "nccm c" ja-kanji)
("" "PC" "^jm L[W" "^jm L[W" ja-kanji)
("" "R[" "^`}`m R[" "^`}`m R[" ja-kanji)
("" "R[" "qCm R[" "qCm R[" ja-kanji)
("" "R" "PKm R" "PKm R" ja-kanji)
("" "V" "JXm V" "JXm V" ja-kanji)
("" "WN" "VAcCm WN" "VAcCm WN" ja-kanji)
("" "\" "TJm{m \" "TJm{m \" ja-kanji)
("" "\[" "\[JCm \[" "\[JCm \[" ja-kanji)
("" "\[" "TYCj \[" "TYCj \[" ja-kanji)
("" "g[" "~iMm g[" "~iMm g[" ja-kanji)
("" "g[" "Njm im g[" "Njm im g[" ja-kanji)
("" "g[" "CPm g[" "CPm g[" ja-kanji)
("" "t" "tem t" "tem t" ja-kanji)
("" "{[" "{[[m {[" "{[[m {[" ja-kanji)
("" "C" "NCm C" "NCm C" ja-kanji)
("" "GC" "mMm GC" "mMm GC" ja-kanji)
("" "KC" "JKCm KC" "JKCm KC" ja-kanji)
("" "J" "JKCm J" "JKCm J" ja-kanji)
("" "R" "VnCm CC" "VnCm CC" ja-kanji)
("" "R" "XCRfm R" "XCRfm R" ja-kanji)
("" "R" "^Mm R" "^Mm R" ja-kanji)
("" "V[" "m~mm V[" "m~mm V[" ja-kanji)
("" "V" "jWm V" "jWm V" ja-kanji)
("" "\[" "XXOm \[" "XXOm \[" ja-kanji)
("" "^C" "^Cm L[W" "^Cm L[W" ja-kanji)
("" "`[" "~iMm `[" "~iMm `[" ja-kanji)
("" "eL" "AEm eL" "AEm eL" ja-kanji)
("@" "[" "^_Em [" "^_Em [" ja-kanji)
("A" "" "V~m " "V~m " ja-kanji)
("B" "" "TYCj n" "TYCj n" ja-kanji)
("C" "M[" "EXCm M[" "EXCm M[" ja-kanji)
("D" "Z" "Z[m Z" "Z[m Z" ja-kanji)
("E" "T" "T[m T" "T[m T" ja-kanji)
("F" "W[" "VuCm CW" "VuCm CW" ja-kanji)
("G" "W[" "VuCm L[W" "VuCm L[W" ja-kanji)
("H" "W" "W[m W" "W[m W" ja-kanji)
("I" "Z" "Om L[W" "Om L[W" ja-kanji)
("J" "Z" "Om CW" "Om CW" ja-kanji)
("K" "^" "tJCm ^" "tJCm ^" ja-kanji)
("L" "`[" "Xm `[" "Xm `[" ja-kanji)
("M" "g[" "TYCj h[" "TYCj h[" ja-kanji)
("N" "n" "V~Ym n" "V~Ym n" ja-kanji)
("O" "z[" "TYCj z[" "TYCj z[" ja-kanji)
("P" "[" "^}m L[W" "^}m L[W" ja-kanji)
("Q" "" "EXCm " "EXCm " ja-kanji)
("R" "[" "^}~Ym [" "^}~Ym [" ja-kanji)
("S" "I[" "TYCj IN" "TYCj IN" ja-kanji)
("T" "J" "TYCj ~L" "TYCj ~L" ja-kanji)
("U" "\[" "AEm \[" "AEm \[" ja-kanji)
("V" "^N" "Tm L[W" "Tm L[W" ja-kanji)
("W" "^" "A^m ^" "A^m ^" ja-kanji)
("X" "t" "TYCj t" "TYCj t" ja-kanji)
("Y" "C" "~IcNVm ~I" "~IcNVm ~I" ja-kanji)
("Z" "TC" "Xm L[W" "Xm L[W" ja-kanji)
("[" "Vc" "Vm L[W" "Vm L[W" ja-kanji)
("\" "V" "TEm V" "TEm V" ja-kanji)
("]" "r" "rCm r" "rCm r" ja-kanji)
("^" "C" "kJ~m C" "kJ~m C" ja-kanji)
("_" "q" "n}m L[W" "n}m L[W" ja-kanji)
("`" "{N" "TYCj {N" "TYCj {N" ja-kanji)
("a" "[" "[}Cm [" "[}Cm [" ja-kanji)
("b" "V" "nNm V" "nNm V" ja-kanji)
("c" "V" "V[m V" "V[m V" ja-kanji)
("d" "Z" "\\Om Z" "\\Om Z" ja-kanji)
("e" "oN" "^Lm oN" "^Lm oN" ja-kanji)
("f" "[" "TYCj [" "TYCj [" ja-kanji)
("g" "[" "zKJm [" "zKJm [" ja-kanji)
("h" "" "RXm " "RXm " ja-kanji)
("i" "GC" "E~m GC" "E~m GC" ja-kanji)
("j" "J" "JJCm J" "JJCm J" ja-kanji)
("k" "`" "TYCj `" "TYCj `" ja-kanji)
("l" "L" "V^^m L" "V^^m L" ja-kanji)
("m" "" "XCm " "XCm " ja-kanji)
("n" "V[" "V[Vm V[" "V[Vm V[" ja-kanji)
("o" "r" "^m r" "^m r" ja-kanji)
("p" "" "i~m " "i~m " ja-kanji)
("q" "" "TYCj " "TYCj " ja-kanji)
("r" "TC" "VNm TC" "VNm TC" ja-kanji)
("s" "" "CGm L[W" "CGm L[W" ja-kanji)
("t" "V" "Aum V" "Aum V" ja-kanji)
("u" "\[" "C^m \[" "C^m \[" ja-kanji)
("v" "PC" "PCKm PC" "PCKm PC" ja-kanji)
("w" "PC" "PCm ]NW" "PCm ]NW" ja-kanji)
("x" "L" "R^cm R" "R^cm R" ja-kanji)
("y" "TN" "TNcm TN" "TNcm TN" ja-kanji)
("z" "wC" "ALJm wC" "ALJm wC" ja-kanji)
("{" "z[" "Aum z[" "Aum z[" ja-kanji)
("|" "G" "Pm CW" "Pm CW" ja-kanji)
("}" "L[" "Xj ce" "Xj ce" ja-kanji)
("~" "W[" "XXm W[" "XXm W[" ja-kanji)
("" "N" "NCm N" "NCm N" ja-kanji)
("" "G" "V[Gm G" "V[Gm G" ja-kanji)
("" "z[" "mVm z[" "mVm z[" ja-kanji)
("" "R" "qwj }W" "qwj }W" ja-kanji)
("" "z[" "Aum z[" "Aum z[" ja-kanji)
("" "J" "Jpcm J" "Jpcm J" ja-kanji)
("" "L" "qJm L" "qJm L" ja-kanji)
("" "L" "Lm CW" "Lm CW" ja-kanji)
("" "N" "A^^Jm N" "A^^Jm N" ja-kanji)
("" "PC" "qgm PC" "qgm PC" ja-kanji)
("" "R[" "JKNm R[" "JKNm R[" ja-kanji)
("" "_" "_m _" "_m _" ja-kanji)
("" "[" "Aum [" "Aum [" ja-kanji)
("" "N" "CuXm N" "CuXm N" ja-kanji)
("" "N" "CuXm CW" "CuXm CW" ja-kanji)
("" "\N" "LGm \N" "LGm \N" ja-kanji)
("" "R[" "qwj ~cO" "qwj ~cO" ja-kanji)
("" "C" "mVm C" "mVm C" ja-kanji)
("" "S[" "Cm S[" "Cm S[" ja-kanji)
("" "J" "A^^m J" "A^^m J" ja-kanji)
("" "L" "JXJim L" "JXJim L" ja-kanji)
("" "V" "TJm V" "TJm V" ja-kanji)
("" "V[" "Nm L[W" "Nm L[W" ja-kanji)
("" "g" "qwm g" "qwm g" ja-kanji)
("" "n" "Nm n" "Nm n" ja-kanji)
("" "[" "JKrm [" "JKrm [" ja-kanji)
("" "I[" "A^^Jm I[" "A^^Jm I[" ja-kanji)
("" "L" "Nm L" "Nm L" ja-kanji)
("" "XC" "qE`m XC" "qE`m XC" ja-kanji)
("" "^c" "R^cm ^c" "R^cm ^c" ja-kanji)
("" "W" "GJXm W" "GJXm W" ja-kanji)
("" "Z" "wCZm Z" "wCZm Z" ja-kanji)
("" "[" "JKNm [" "JKNm [" ja-kanji)
("" "VN" "JKNm VN" "JKNm VN" ja-kanji)
("" "" "Cm L[W" "Cm L[W" ja-kanji)
("" "" "}m " "}m " ja-kanji)
("" "T" "J}hm T" "J}hm T" ja-kanji)
("" "\[" "A\Em L[W" "A\Em L[W" ja-kanji)
("" "n" "n`[Cm n" "n`[Cm n" ja-kanji)
("" "G" "RRjm G" "RRjm G" ja-kanji)
("" "C" "^m L[W" "^m L[W" ja-kanji)
("" "R[" "}Wm R[" "}Wm R[" ja-kanji)
("" "\" "}iC^m \" "}iC^m \" ja-kanji)
("" "V[" "V[w" "V[w" ja-kanji)
("" "\[" "TJm L[W" "TJm L[W" ja-kanji)
("" "V[" "JLlm V[" "JLlm V[" ja-kanji)
("" "Z" "t_m Z" "t_m Z" ja-kanji)
("" "gN" "t_m gN" "t_m gN" ja-kanji)
("" "eC" "A^m eC" "A^m eC" ja-kanji)
("" "S" "EVwj " "EVwj " ja-kanji)
("" "C" "XLm C" "XLm C" ja-kanji)
("" "C" "Cm CW" "Cm CW" ja-kanji)
("" "z" "nVm z" "nVm z" ja-kanji)
("" "R[" "lMEm R[" "lMEm R[" ja-kanji)
("" "N" "}_EVm N" "}_EVm N" ja-kanji)
("" "gN" "R[Vm gN" "R[Vm gN" ja-kanji)
("" "M" "CPjGm L[W" "CPjGm L[W" ja-kanji)
("" "[" "Pmwj [" "Pmwj [" ja-kanji)
("" "TC" "}Ckm TC" "}Ckm TC" ja-kanji)
("" "W[" "im W[" "im W[" ja-kanji)
("" "`[" "Ckm `" "Ckm `" ja-kanji)
("" "eL" "GrXm eL" "GrXm eL" ja-kanji)
("" "R[" "im R[" "im R[" ja-kanji)
("" "q" "Tm qq" "Tm qq" ja-kanji)
("" "JN" "Wim oN" "Wim oN" ja-kanji)
("" "R" "nin_VCm R" "nin_VCm R" ja-kanji)
("" "R[" "YCm R[" "YCm R[" ja-kanji)
("" "L[" "Z}Cm L[W" "Z}Cm L[W" ja-kanji)
("" "P" "PJCm P" "PJCm P" ja-kanji)
("" "VN" "X~Jm VN" "X~Jm VN" ja-kanji)
("" "C" "Pmwj L" "Pmwj L" ja-kanji)
("" "QC" "QCJm QC" "QCJm QC" ja-kanji)
("" "TC" "E^KEm TC" "E^KEm TC" ja-kanji)
("" "V[" "V[Pcm V[" "V[Pcm V[" ja-kanji)
("" "\c" "jJm \c" "jJm \c" ja-kanji)
("" "R[" "Pmwj R[" "Pmwj R[" ja-kanji)
("" "^" "CmVVm ^" "CmVVm ^" ja-kanji)
("" "V[" "Pmwj zV" "Pmwj zV" ja-kanji)
("" "C" "~_m C" "~_m C" ja-kanji)
("" "Jc" "YCm Jc" "YCm Jc" ja-kanji)
("" "V[" "XXm L[W" "XXm L[W" ja-kanji)
("" "oN" "oNm CW" "oNm CW" ja-kanji)
("" "N" "_}m L[W" "_}m L[W" ja-kanji)
("" "Pc" "^Pm Pc" "^Pm Pc" ja-kanji)
("" "JC" "YCm JC" "YCm JC" ja-kanji)
("" "hN" "qgm L[W" "qgm L[W" ja-kanji)
("" "h[" "h[[m h[" "h[[m h[" ja-kanji)
("" "W[" "Pmm L[W" "Pmm L[W" ja-kanji)
("" "[" "Jm L[W" "Jm L[W" ja-kanji)
("" "P" "TTQm L[W" "TTQm L[W" ja-kanji)
("" "^c" "JE\m ^c" "JE\m ^c" ja-kanji)
("" "J" "R[q[m R" "R[q[m R" ja-kanji)
("" "^C" "^C}Cm ^C" "^C}Cm ^C" ja-kanji)
("" "`" "YVCm CW" "YVCm CW" ja-kanji)
("" "n" "KXm n" "KXm n" ja-kanji)
("" "nN" "RnNm nN" "RnNm nN" ja-kanji)
("" "W" "I[wj ~~" "I[wj ~~" ja-kanji)
("" "nC" "I[wj nC" "I[wj nC" ja-kanji)
("" "N" "I[wj JN" "I[wj JN" ja-kanji)
("" "" "m L[W" "m L[W" ja-kanji)
("" "[" "[Jm [" "[Jm [" ja-kanji)
("" "[" "[m L[W" "[m L[W" ja-kanji)
("" "R" "RnNm R" "RnNm R" ja-kanji)
("" "S" "I[wj " "I[wj " ja-kanji)
("" "q" "R[q[m q" "R[q[m q" ja-kanji)
("" "z[" "I[wj z[" "I[wj z[" ja-kanji)
("" "J" "LYm J" "LYm J" ja-kanji)
("" "R" "I[wj O" "I[wj O" ja-kanji)
("" "Vc" "LVcm Vc" "LVcm Vc" ja-kanji)
("" "m[" "m[m m[" "m[m m[" ja-kanji)
("" "{[" "I[wj IJX" "I[wj IJX" ja-kanji)
("" "" "I[wj " "I[wj " ja-kanji)
("" "GC" "JKNm GC" "JKNm GC" ja-kanji)
("" "JC" "I[wj Ij" "I[wj Ij" ja-kanji)
("" "T" "YVCm T" "YVCm T" ja-kanji)
("" "" "m[m " "m[m " ja-kanji)
("" "[" "^}m CW" "^}m CW" ja-kanji)
("" "L" "I[wj L" "I[wj L" ja-kanji)
("" "V[" "I[wj V[" "I[wj V[" ja-kanji)
("" "nN" "A^}m nN" "A^}m nN" ja-kanji)
("" "wL" "JyLm yL" "JyLm yL" ja-kanji)
("" "PC" "^}m PC" "^}m PC" ja-kanji)
("" "[" "I[wj ^c" "I[wj ^c" ja-kanji)
("" "GC" "NrJUm GC" "NrJUm GC" ja-kanji)
("" "GC" "GCm NW" "GCm NW" ja-kanji)
("@" "R" "qTSm R" "qTSm R" ja-kanji)
("A" "x" "xm L[W" "xm L[W" ja-kanji)
("C" "LO" "ITm ^C" "ITm ^C" ja-kanji)
("D" "I[" "Jm I[" "Jm I[" ja-kanji)
("E" "g" "ITm ^C" "ITm ^C" ja-kanji)
("G" "~O" "ITm ^C" "ITm ^C" ja-kanji)
("H" "wN^O" "ITm ^C" "ITm ^C" ja-kanji)
("I" "V" "Lmm V" "Lmm V" ja-kanji)
("J" "P" "XGmm P" "XGmm P" ja-kanji)
("K" "V[" "VLKm V[" "VLKm V[" ja-kanji)
("L" "Z`O" "ITm ^C" "ITm ^C" ja-kanji)
("M" "I[" "Jm I[" "Jm I[" ja-kanji)
("N" "Z" "Jm Z" "Jm Z" ja-kanji)
("O" "{[" "Jm {[" "Jm {[" ja-kanji)
("P" "I[" "Jm I[" "Jm I[" ja-kanji)
("Q" "wL" "VLKm wL" "VLKm wL" ja-kanji)
("R" "V[" "im CW" "im CW" ja-kanji)
("S" "\" "~KGm \" "~KGm \" ja-kanji)
("T" "[" "Nm [" "Nm [" ja-kanji)
("U" "`[" "}`m CW" "}`m CW" ja-kanji)
("V" "[" "ghm NW" "ghm NW" ja-kanji)
("W" "JC" "^wj JC" "^wj JC" ja-kanji)
("X" "R[" "^KXm R[" "^KXm R[" ja-kanji)
("Y" "t" "^wj Ibg" "^wj Ibg" ja-kanji)
("Z" "V" "VCLm V" "VCLm V" ja-kanji)
("[" "z" "Elm zW" "Elm zW" ja-kanji)
("\" "z" "bRm z" "bRm z" ja-kanji)
("]" "C" "^wj R" "^wj R" ja-kanji)
("^" "W" "^wj e" "^wj e" ja-kanji)
("_" "N" "A}Vm CW" "A}Vm CW" ja-kanji)
("`" "K" "GKNm L[W" "GKNm L[W" ja-kanji)
("a" "" "^wj A}" "^wj A}" ja-kanji)
("b" "L" "YVCm L" "YVCm L" ja-kanji)
("c" "g[" "A^m L[W" "A^m L[W" ja-kanji)
("d" "L[" "TJCm L[" "TJCm L[" ja-kanji)
("e" "`[" "TLjm `[" "TLjm `[" ja-kanji)
("f" "`[" "`[m NW" "`[m NW" ja-kanji)
("g" "W[" "^^~m L[W" "^^~m L[W" ja-kanji)
("h" "W[" "^^~m zW" "^^~m zW" ja-kanji)
("i" "W[" "^^~m W[" "^^~m W[" ja-kanji)
("j" "`[" "}C_j `[" "}C_j `[" ja-kanji)
("k" "L[" "}Cm L[" "}Cm L[" ja-kanji)
("l" "Z" "nC^m Z" "nC^m Z" ja-kanji)
("m" "JC" "JCZm JC" "JCZm JC" ja-kanji)
("n" "[" "C{m [" "C{m [" ja-kanji)
("o" "J" "}C_j J" "}C_j J" ja-kanji)
("p" "J" "qLcPm J" "qLcPm J" ja-kanji)
("q" "Q" "}C_j Q" "}C_j Q" ja-kanji)
("r" "V" "LYm V" "LYm V" ja-kanji)
("s" "\" "^\r[m \" "^\r[m \" ja-kanji)
("t" "^" "I[_m ^" "I[_m ^" ja-kanji)
("u" "g[" "EYNm g[" "EYNm g[" ja-kanji)
("v" "z[" "z[\[m z[" "z[\[m z[" ja-kanji)
("w" "C" "LYm C" "LYm C" ja-kanji)
("x" "Z" "PbZm Z" "PbZm Z" ja-kanji)
("y" "[" "JCm [" "JCm [" ja-kanji)
("z" "" "qLcm PC" "qLcm " ja-kanji)
("{" "V" "AUm V" "AUm V" ja-kanji)
("|" "q" "}C_j q" "}C_j q" ja-kanji)
("}" "A" "}Cm A" "}Cm A" ja-kanji)
("~" "C" "iGm C" "iGm C" ja-kanji)
("" "R" "iKYCm R" "iKYCm R" ja-kanji)
("" "XC" "cJm XC" "cJm XC" ja-kanji)
("" "^" "^com ^" "^com ^" ja-kanji)
("" "q" "J^}qm q" "J^}qm q" ja-kanji)
("" "}" "Vrm }" "Vrm }" ja-kanji)
("" "" "r[m " "r[m " ja-kanji)
("" "t[" "L[Wm t[" "L[Wm t[" ja-kanji)
("" "[" "nmm [" "nmm [" ja-kanji)
("" "" "iIm " "iIm " ja-kanji)
("" "I" "IGLm I" "IGLm I" ja-kanji)
("" "MN" "MNr[m MN" "MNr[m MN" ja-kanji)
("" "\[" "Zm \[" "Zm \[" ja-kanji)
("" "\[" "LYm \[" "LYm \[" ja-kanji)
("" "n" "Agm n" "Agm n" ja-kanji)
("" "[" "Rum [" "Rum [" ja-kanji)
("" "V[" "V[Lm V[" "V[Lm V[" ja-kanji)
("" "C" "CLm C" "CLm C" ja-kanji)
("" "[" "ZVm [" "ZVm [" ja-kanji)
("" "J" "JVNm J" "JVNm J" ja-kanji)
("" "nC" "nCWm nC" "nCWm nC" ja-kanji)
("" "[" "C^m [" "C^m [" ja-kanji)
("" "f" "}C_j hm" "}C_j hm" ja-kanji)
("" "C" "CLm C" "CLm C" ja-kanji)
("" "`" "IJm L[W" "IJm L[W" ja-kanji)
("" "[" "JCm CW" "JCm CW" ja-kanji)
("" "JN" "JNm JN" "JNm JN" ja-kanji)
("" "C" "Cr[m C" "Cr[m C" ja-kanji)
("" "VN" "JVNm VN" "JVNm VN" ja-kanji)
("" "L" "}C_j L" "}C_j L" ja-kanji)
("" "Z" "JCZm Z" "JCZm Z" ja-kanji)
("" "[" "[\m [" "[\m [" ja-kanji)
("" "e" "t[em e" "t[em e" ja-kanji)
("" "nc" "ncKV" "ncKV" ja-kanji)
("" "L" "~Ym gm L" "~Ym gm L" ja-kanji)
("" "nc" "nicm L[W" "nicm L[W" ja-kanji)
("" "L[" "JoVCm L[" "JoVCm L[" ja-kanji)
("" "{[" "J^hm CW" "J^hm CW" ja-kanji)
("" "L" "JGm CW" "JGm CW" ja-kanji)
("" "R[" "TcLm CW" "TcLm CW" ja-kanji)
("" "R[" "Vwj R[" "Vwj R[" ja-kanji)
("" "J" "Vwj J" "Vwj J" ja-kanji)
("" "R[" "Vwj cQ" "Vwj cQ" ja-kanji)
("" "ZL" "nNZLm ZL" "nNZLm ZL" ja-kanji)
("" "KC" "KC[m KC" "KC[m KC" ja-kanji)
("" "z[" "jLrm z[" "jLrm z[" ja-kanji)
("" "V" "qrm V" "qrm V" ja-kanji)
("" "N" "AJMm N" "AJMm N" ja-kanji)
("" "N" "Nm CW" "Nm CW" ja-kanji)
("" "V[" "Vm V[" "Vm V[" ja-kanji)
("" "E" "n`m E" "n`m E" ja-kanji)
("" "R[" "i]m R[" "i]m R[" ja-kanji)
("" "KC" "P_Vm CW" "P_Vm CW" ja-kanji)
("" "S[" "nS[m S[" "nS[m S[" ja-kanji)
("" "T" "TJYLm T" "TJYLm T" ja-kanji)
("" "W" "cLm L[W" "cLm L[W" ja-kanji)
("" "J" "^Cm J" "^Cm J" ja-kanji)
("" "" "ZCm " "ZCm " ja-kanji)
("" "g[" "ESNm g[" "ESNm g[" ja-kanji)
("" "g[" "g[m CW" "g[m CW" ja-kanji)
("" "PC" "wj PC" "wj PC" ja-kanji)
("" "^" "jm ^" "jm ^" ja-kanji)
("" "r[" "XKm r[" "XKm r[" ja-kanji)
("" "x" "Rxm x" "Rxm x" ja-kanji)
("" "Q" "}uVCm Q" "}uVCm Q" ja-kanji)
("" "C" "wj A}" "wj A}" ja-kanji)
("" "V" "}Rgm L[W" "}Rgm L[W" ja-kanji)
("" "ZC" "ZCm CW" "ZCm CW" ja-kanji)
("" "ZC" "}iWm ZC" "}iWm ZC" ja-kanji)
("" "}C" "NCm }C" "NCm }C" ja-kanji)
("" "P" "JG~m P" "JG~m P" ja-kanji)
("" "{[" "qg~m {[" "qg~m {[" ja-kanji)
("" "eC" "wj Ig[g" "wj Ig[g" ja-kanji)
("" "KC" "jm KC" "jm KC" ja-kanji)
("" "QC" "jm QC" "jm QC" ja-kanji)
("" "V[" "}cQm V[" "}cQm V[" ja-kanji)
("" "ZC" "qg~m ZC" "qg~m ZC" ja-kanji)
("" "wC" "wCQCm wC" "wCQCm wC" ja-kanji)
("" "GC" "GCm CW" "GCm CW" ja-kanji)
("" "R[" "R[Km R[" "R[Km R[" ja-kanji)
("" "g" "~m g" "~m g" ja-kanji)
("" "Jc" "wj KC" "wj KC" ja-kanji)
("" "V" "CJm V" "CJm V" ja-kanji)
("" "C" "C\[m C" "C\[m C" ja-kanji)
("" "h[" "h[Nm h[" "h[Nm h[" ja-kanji)
("" "}" "M}m }" "M}m }" ja-kanji)
("" "J" "~IXm J" "~IXm J" ja-kanji)
("" "L" "wj g[gC" "wj g[gC" ja-kanji)
("" "AC" "wj AC" "wj AC" ja-kanji)
("" "N" "N[m N" "N[m N" ja-kanji)
("" "P" "}u^m P" "}u^m P" ja-kanji)
("" "R" "m[~m I" "m[~m I" ja-kanji)
("" "Z" "Z{[m Z" "Z{[m Z" ja-kanji)
("" "[" "[}Cm [" "[}Cm [" ja-kanji)
("" "JN" "JNVNm JN" "JNVNm JN" ja-kanji)
("" "`N" "`NVcm `N" "`NVcm `N" ja-kanji)
("" "VN" "VNNm VN" "VNNm VN" ja-kanji)
("" "L" "ccVm L" "ccVm L" ja-kanji)
("" "C" "j " "j " ja-kanji)
("" "C" "CV[m C" "CV[m C" ja-kanji)
("" "R[" "CVwj R[" "CVwj R[" ja-kanji)
("" "ZC" "CVwj L" "CVwj L" ja-kanji)
("" "q" "q\m q" "q\m q" ja-kanji)
("" "R[" "AKlm R[" "AKlm R[" ja-kanji)
("" "V" "CV}m V" "CV}m V" ja-kanji)
("" "C" "gCVm g" "gCVm g" ja-kanji)
("" "PC" "PC\m PC" "PC\m PC" ja-kanji)
("" "TC" "N_Nm TC" "N_Nm TC" ja-kanji)
("" "JL" "CVwj ni" "CVwj ni" ja-kanji)
("" "n" "i~j CV" "i~j CV" ja-kanji)
("" "z[" "z[\m z[" "z[\m z[" ja-kanji)
("" "nC" "nCCm nC" "nCCm nC" ja-kanji)
("" "N" "N_Jm N" "N_Jm N" ja-kanji)
("" "Pc" "CVu~m Pc" "CVu~m Pc" ja-kanji)
("" "ZL" "CVwj eC" "CVwj eC" ja-kanji)
("" "`" "Lk^m CW" "Lk^m CW" ja-kanji)
("" "m[" "CVwj m[" "CVwj m[" ja-kanji)
("" "KC" "CVwj KC" "CVwj KC" ja-kanji)
("" "Jc" "CVwj zl" "CVwj zl" ja-kanji)
("" "T" "~KNm T" "~KNm T" ja-kanji)
("" "^N" "ncPm ^N" "ncPm ^N" ja-kanji)
("" "e" "CVwj e" "CVwj e" ja-kanji)
("" "l" "^Cm [h" "^Cm [h" ja-kanji)
("" "z[" "^Cm |h" "^Cm |h" ja-kanji)
("" "C" "CNm C" "CNm C" ja-kanji)
("" "PC" "PCZcm PC" "PCZcm PC" ja-kanji)
("@" "ZL" "CVwj ZL" "CVwj ZL" ja-kanji)
("A" "Z" "Jm Z" "Jm Z" ja-kanji)
("B" "R[" "R[JNm R[" "R[JNm R[" ja-kanji)
("C" "g[" "CVwj m{" "CVwj m{" ja-kanji)
("D" "CN" "^}j j^ CV" "^}j j^ CV" ja-kanji)
("E" "M" "C\m M" "C\m M" ja-kanji)
("F" "g[" "CVwj A^" "CVwj A^" ja-kanji)
("G" "KC" "CVwj M" "CVwj M" ja-kanji)
("H" "o" "~[om o" "~[om o" ja-kanji)
("I" "L" "cuem L" "cuem L" ja-kanji)
("J" "V" "}cm V" "}cm V" ja-kanji)
("K" "V" "zRm V" "zRm V" ja-kanji)
("L" "V" "ccVm V" "ccVm V" ja-kanji)
("M" "XC" "^^m XC" "^^m XC" ja-kanji)
("N" "\" "TCCm \" "TCCm \" ja-kanji)
("O" "q" "qm q" "qm q" ja-kanji)
("P" "tc" "nEm tc" "nEm tc" ja-kanji)
("Q" "L" "VXwj \m" "VXwj \m" ja-kanji)
("R" "N" "Nm L[W" "Nm L[W" ja-kanji)
("S" "PC" "~\Mm PC" "~\Mm PC" ja-kanji)
("T" "VN" "VXwm VN" "VXwm VN" ja-kanji)
("U" "L" "VXwj L" "VXwj L" ja-kanji)
("V" "TC" "TCm L[W" "TCm L[W" ja-kanji)
("W" "[" "[m L[W" "[m L[W" ja-kanji)
("X" "C" "Cm L[W" "Cm L[W" ja-kanji)
("Y" "W[" "nEm W[" "nEm W[" ja-kanji)
("Z" "E" "EI[m E" "EI[m E" ja-kanji)
("[" "O" "IiKUm O" "IiKUm O" ja-kanji)
("\" "wC" "gm wC" "gm wC" ja-kanji)
("]" "q" "VCim q" "VCim q" ja-kanji)
("^" "I[" "iGm I[" "iGm I[" ja-kanji)
("_" "L" "NLrm Lr" "NLrm Lr" ja-kanji)
("`" "nc" "C^m nc" "C^m nc" ja-kanji)
("a" "}c" "}OTm }c" "}OTm }c" ja-kanji)
("b" "J" "m J" "m J" ja-kanji)
("c" "V[" "m V[" "m V[" ja-kanji)
("d" "L" "mMwj \m" "mMwj \m" ja-kanji)
("e" "`N" "mMwj `N" "mMwj `N" ja-kanji)
("f" "`[" "mMwj V[" "mMwj V[" ja-kanji)
("g" "" "Mm " "Mm " ja-kanji)
("h" "" "m CW" "m CW" ja-kanji)
("i" "V[" "^^Gm L[W" "^^Gm L[W" ja-kanji)
("j" "g[" "Clm CW" "Clm CW" ja-kanji)
("k" "R[" "m R[" "m R[" ja-kanji)
("l" "VN" "Lrm VN" "Lrm VN" ja-kanji)
("m" "[" "mMwj [" "mMwj [" ja-kanji)
("n" "XC" "zm CW" "zm CW" ja-kanji)
("o" "`" "ITiCm CW" "ITiCm CW" ja-kanji)
("p" "VN" "VNtm VN" "VNtm VN" ja-kanji)
("q" "AC" "PKm AC" "PKm AC" ja-kanji)
("r" "I" "I_Jm L[W" "I_Jm L[W" ja-kanji)
("s" "V[" "ALm L[W" "ALm L[W" ja-kanji)
("t" "W[" "~mm L[W" "~mm L[W" ja-kanji)
("u" "L[" "L[em L[" "L[em L[" ja-kanji)
("v" "ZC" "IgVAim ZC" "IgVAim ZC" ja-kanji)
("w" "[" "[`[m [" "[`[m [" ja-kanji)
("x" "\[" "}hm L[W" "}hm L[W" ja-kanji)
("y" "`[" "[`[m `[" "[`[m `[" ja-kanji)
("z" "L" "NVm L" "NVm L" ja-kanji)
("{" "R[" "AiOm R[" "AiOm R[" ja-kanji)
("|" "J" "Aim J" "Aim J" ja-kanji)
("}" "\[" "J}hm L[W" "J}hm L[W" ja-kanji)
("~" "[" "J}m L[W" "J}m L[W" ja-kanji)
("" "N" "cm N" "cm N" ja-kanji)
("" "L[" "Aim L[" "Aim L[" ja-kanji)
("" "U" "Uj[m U" "Uj[m U" ja-kanji)
("" "[" "Aij [" "Aij [" ja-kanji)
("" "XC" "tJCm XC" "tJCm XC" ja-kanji)
("" "g[" "Aim g[" "Aim g[" ja-kanji)
("" "Zc" "kXm L[W" "kXm L[W" ja-kanji)
("" "fJbg" "^CZLm ^C" "^CZLm ^C" ja-kanji)
("" "Lbg" "^CZLm ^C" "^CZLm ^C" ja-kanji)
("" "fVbg" "^CZLm ^C" "^CZLm ^C" ja-kanji)
("" "~bg" "^CZLm ^C" "^CZLm ^C" ja-kanji)
("" "^" "^cm ^" "^cm ^" ja-kanji)
("" "`" "^^Ym `" "^^Ym `" ja-kanji)
("" "wC" "ium L[W" "ium L[W" ja-kanji)
("" "wNgbg" "^CZLm ^C" "^CZLm ^C" ja-kanji)
("" "V" "}cm V" "}cm V" ja-kanji)
("" "V[" "V[[m V[" "V[[m V[" ja-kanji)
("" "Pc" "cNXm Pc" "cNXm Pc" ja-kanji)
("" "Z`bg" "^CZLm ^C" "^CZLm ^C" ja-kanji)
("" "K" "^Pj }" "^Pj }" ja-kanji)
("" "Rc" "VNm Rc" "VNm Rc" ja-kanji)
("" "\[" "Um \[" "Um \[" ja-kanji)
("" "n" "^Pj gG" "^Pj gG" ja-kanji)
("" "J" "tGm CbVm J" "tGm CbVm J" ja-kanji)
("" "Z" "^Pj V" "^Pj V" ja-kanji)
("" "V[" "tGm V[" "tGm V[" ja-kanji)
("" "`" "`m `" "`m `" ja-kanji)
("" "n" "^Pj n" "^Pj n" ja-kanji)
("" "z" "^Pj z" "^Pj z" ja-kanji)
("" "V" "^PJj " "^PJj " ja-kanji)
("" "L[" "JSm L[" "JSm L[" ja-kanji)
("" "L[" "em A L[" "em A L[" ja-kanji)
("" "PC" "JUVm PC" "JUVm PC" ja-kanji)
("" "V" "^PmRm V" "^PmRm V" ja-kanji)
("" "\[" "Rgm \[" "Rgm \[" ja-kanji)
("" "Z" "tZSm Z" "tZSm Z" ja-kanji)
("" "Z" "TTm Z" "TTm Z" ja-kanji)
("" "G" "Vm G" "Vm G" ja-kanji)
("" "L" "nRm L" "nRm L" ja-kanji)
("" "L[" "^Pj L[" "^Pj L[" ja-kanji)
("" "P" "JPqm P" "JPqm P" ja-kanji)
("" "TN" "^Pj cN" "^Pj cN" ja-kanji)
("" "V[" "Vmm CW" "Vmm CW" ja-kanji)
("" "ZC" "^Pj i" "^Pj i" ja-kanji)
("" "ZC" "EiEm ZC" "EiEm ZC" ja-kanji)
("" "J" "JR[m J" "JR[m J" ja-kanji)
("" "L" "^Pj L" "^Pj L" ja-kanji)
("" "L" "^Pj R" "^Pj R" ja-kanji)
("" "R" "^Km R" "^Km R" ja-kanji)
("" "N" "^Pj \" "^Pj \" ja-kanji)
("" "g[" "TXm g[" "TXm g[" ja-kanji)
("" "Z" "t_m Z" "t_m Z" ja-kanji)
("" "\[" "z[Lm \[" "z[Lm \[" ja-kanji)
("" "\[" "Rgm \[" "Rgm \[" ja-kanji)
("" "\[" "\[m ]NW" "\[m ]NW" ja-kanji)
("" "tN" "Grm tN" "Grm tN" ja-kanji)
("" "L[" "JSm L[" "JSm L[" ja-kanji)
("" "R[" "^Jm R[" "^Jm R[" ja-kanji)
("" "R[" "^Pj R[" "^Pj R[" ja-kanji)
("" "J" "nRm J" "nRm J" ja-kanji)
("" "V" "C}Vm V" "C}Vm V" ja-kanji)
("" "e" "eRNm e" "eRNm e" ja-kanji)
("" "R[" "JKrm JK" "JKrm JK" ja-kanji)
("" "V" "tCm V" "tCm V" ja-kanji)
("" "T" "^Pm ~m" "^Pm ~m" ja-kanji)
("" "T" "~mm ]NW" "~mm ]NW" ja-kanji)
("" "wC" "JUVm wC" "JUVm wC" ja-kanji)
("" "c" "^Pj N" "^Pj N" ja-kanji)
("" "[" "JSm L[W" "JSm L[W" ja-kanji)
("" "TN" "XmRm T" "XmRm T" ja-kanji)
("" "\[" "Km \[" "Km \[" ja-kanji)
("" "`[" "^Pj z" "^Pj z" ja-kanji)
("" "qc" "Com qc" "Com qc" ja-kanji)
("" "z[" "g}m z[" "g}m z[" ja-kanji)
("" "[" "im [" "im [" ja-kanji)
("" "" "^PJSm " "^PJSm " ja-kanji)
("" "G" "^Pj G" "^Pj G" ja-kanji)
("" "L" "bRm L" "bRm L" ja-kanji)
("" "R[" "tGm V^m R[" "tGm V^m R[" ja-kanji)
("" "V" "JUVm V" "JUVm V" ja-kanji)
("" "e" "Vm e" "Vm e" ja-kanji)
("" "G" "mLm G" "mLm G" ja-kanji)
("" "V[" "tGm V[" "tGm V[" ja-kanji)
("" "Z" "ZI[m Z" "ZI[m Z" ja-kanji)
("" "`[" "nJSgm `[" "nJSgm `[" ja-kanji)
("" "" "JSm " "JSm " ja-kanji)
("" "\[" "um L[W" "um L[W" ja-kanji)
("" "L" "^Pj n^" "^Pj n^" ja-kanji)
("" "`[" "^Pj [" "^Pj [" ja-kanji)
("" "g[" "cm g[" "cm g[" ja-kanji)
("" "g[" "cm CW" "cm CW" ja-kanji)
("" "C" "^Pj C" "^Pj C" ja-kanji)
("" "Z" "I~NWm Z" "I~NWm Z" ja-kanji)
("" "Z" "I~NWm ZCW" "I~NWm ZCW" ja-kanji)
("" "N" "KbLm N" "KbLm N" ja-kanji)
("" "" "}KLm " "}KLm " ja-kanji)
("" "q" "VCim q" "VCim q" ja-kanji)
("" "g" "Rwj g" "Rwj g" ja-kanji)
("" "Gc" "Jgm CC" "Jgm CC" ja-kanji)
("" "S[" "Rwj S[" "Rwj S[" ja-kanji)
("" "V" "Lrm V" "Lrm V" ja-kanji)
("" "W" "Rj VJVe" "Rj VJVe" ja-kanji)
("" "g[" "`}Lm g[" "`}Lm g[" ja-kanji)
("" "R[" "E`m R[" "E`m R[" ja-kanji)
("" "T" "JKNm T" "JKNm T" ja-kanji)
("" "[" "Am [" "Am [" ja-kanji)
("" "[" "Jem CW" "Jem CW" ja-kanji)
("" "XC" "CLm CW" "CLm CW" ja-kanji)
("" "\[" "`}Lm \[" "`}Lm \[" ja-kanji)
("" "J" "Rwj ni" "Rwj ni" ja-kanji)
("" "W[" "}Wm W[" "}Wm W[" ja-kanji)
("" "T" "Rwj W" "Rwj W" ja-kanji)
("" "J" "Rwj CG" "Rwj CG" ja-kanji)
("" "r" "zVCCm r" "zVCCm r" ja-kanji)
("" "r" "^_m r" "^_m r" ja-kanji)
("" "" "Rwj " "Rwj " ja-kanji)
("" "VN" "Jm CW" "Jm CW" ja-kanji)
("" "_" "`Sm _" "`Sm _" ja-kanji)
("" "C" "NSm C" "NSm C" ja-kanji)
("" "eL" "JClm eL" "JClm eL" ja-kanji)
("" "`[" "Elm `[" "Elm `[" ja-kanji)
("" "L[" "AUiEm L[" "AUiEm L[" ja-kanji)
("" "E" "}Km E" "}Km E" ja-kanji)
("@" "`[" "`[I[m `[" "`[I[m `[" ja-kanji)
("A" "E" "Cgwj C[" "Cgwj C[" ja-kanji)
("B" "q" "Cgwj R" "Cgwj R" ja-kanji)
("C" "r" "~_m r" "~_m r" ja-kanji)
("D" "PC" "qLVm PC" "qLVm PC" ja-kanji)
("E" "R[" "nbR[m R[" "nbR[m R[" ja-kanji)
("F" "Tc" "gh}m Tc" "gh}m Tc" ja-kanji)
("G" "Zc" "LYim Zc" "LYim Zc" ja-kanji)
("H" "^C" "Cgwj _C" "Cgwj _C" ja-kanji)
("I" "`" "ATkmm `" "ATkmm `" ja-kanji)
("J" "n" "LYim n" "LYim n" ja-kanji)
("K" "R[" "AJCm R[" "AJCm R[" ja-kanji)
("L" "R[" "Cgwj qJ" "Cgwj qJ" ja-kanji)
("M" "R[" "Cgwj CN" "Cgwj CN" ja-kanji)
("N" "V" "Cgm L[W" "Cgm L[W" ja-kanji)
("O" "W[" "W[^m W[" "W[^m W[" ja-kanji)
("P" "W" "^m W" "^m W" ja-kanji)
("Q" "Zc" "Zcm ]NW" "Zcm ]NW" ja-kanji)
("R" "z[" "JXm z[" "JXm z[" ja-kanji)
("S" "PC" "wm L[W" "wm L[W" ja-kanji)
("T" "V[" "VV[m ]NW" "VV[m ]NW" ja-kanji)
("U" "W[" "Ti_m W[" "Ti_m W[" ja-kanji)
("V" "XC" "XJm XC" "XJm XC" ja-kanji)
("W" "" "LkImm " "LkImm " ja-kanji)
("X" "j" "Cgwj Vmu" "Cgwj Vmu" ja-kanji)
("Y" "L" "Am L" "Am L" ja-kanji)
("Z" "PC" "L[Vm PC" "L[Vm PC" ja-kanji)
("[" "P" "Cgwj }N" "Cgwj }N" ja-kanji)
("\" "TC" "Am TC" "Am TC" ja-kanji)
("]" "V" "NMkm V" "NMkm V" ja-kanji)
("^" "VN" "A_im A_" "A_im A_" ja-kanji)
("_" "Z" "Zm L[W" "Zm L[W" ja-kanji)
("`" "\[" "Xxem L[W" "Xxem L[W" ja-kanji)
("a" "`[" "hXm `[" "hXm `[" ja-kanji)
("b" "g[" "iEm g[" "iEm g[" ja-kanji)
("c" "x" "^m zW" "^m zW" ja-kanji)
("d" "" "Vm " "Vm " ja-kanji)
("e" "C" "Cgwj h" "Cgwj h" ja-kanji)
("f" "" "ciOm " "ciOm " ja-kanji)
("g" "J" "gWm J" "gWm J" ja-kanji)
("h" "V[" "gGm V[" "gGm V[" ja-kanji)
("i" "Zc" "Zcm CW" "Zcm CW" ja-kanji)
("j" "^" "W[^m ^" "W[^m ^" ja-kanji)
("k" "`" "Cgwj C^" "Cgwj C^" ja-kanji)
("l" "r[" "nJm r[" "nJm r[" ja-kanji)
("m" "r" "cCgm r" "cCgm r" ja-kanji)
("n" "C" "Cgwj C" "Cgwj C" ja-kanji)
("o" "GC" "Nrm GC" "Nrm GC" ja-kanji)
("p" "P" "AK^m L[W" "AK^m L[W" ja-kanji)
("q" "TC" "RgLm TC" "RgLm TC" ja-kanji)
("r" "V" "m V" "m V" ja-kanji)
("s" "W[" "^em L[W" "^em L[W" ja-kanji)
("t" "WN" "YVC" "YVC" ja-kanji)
("u" "V" "AJCLkm V" "AJCLkm V" ja-kanji)
("v" "cC" "XKm cC" "XKm cC" ja-kanji)
("w" "g[" "JoLm g[" "JoLm g[" ja-kanji)
("x" "r[" "A}m r[" "A}m r[" ja-kanji)
("y" "L[" "Cgwj cC" "Cgwj cC" ja-kanji)
("z" "r" "ciOm r" "ciOm r" ja-kanji)
("{" "}" "Wm Lkm }" "Wm Lkm }" ja-kanji)
("|" "q[" "ni_m q[" "ni_m q[" ja-kanji)
("}" "z[" "Cgj NY" "Cgj NY" ja-kanji)
("~" "" "Qm " "Qm " ja-kanji)
("" "C" "ciOm C" "ciOm C" ja-kanji)
("" "" "cm " "cm " ja-kanji)
("" "E" "Cgwj N" "Cgwj N" ja-kanji)
("" "J" "jVLm A" "jVLm A" ja-kanji)
("" "T" "Cgwj `" "Cgwj `" ja-kanji)
("" "W[" "Om W[" "Om W[" ja-kanji)
("" "n" "qgNm n" "qgNm n" ja-kanji)
("" "[" "[m [" "[m [" ja-kanji)
("" "GL" "^Ylm GL" "^Ylm GL" ja-kanji)
("" "JC" "Gm L[W" "Gm L[W" ja-kanji)
("" "W[" "im L[W" "im L[W" ja-kanji)
("" "PC" "cOm L[W" "cOm L[W" ja-kanji)
("" "V" "EXMkm V" "EXMkm V" ja-kanji)
("" "ZC" "JXm zW" "JXm zW" ja-kanji)
("" "ZC" "JXm CW" "JXm CW" ja-kanji)
("" "q" "qum q" "qum q" ja-kanji)
("" "x" "xpcm x" "xpcm x" ja-kanji)
("" "" "{m " "{m " ja-kanji)
("" "Pc" "R[Pcm Pc" "R[Pcm Pc" ja-kanji)
("" "T" "cOm T" "cOm T" ja-kanji)
("" "]N" "cdNm L[W" "cdNm L[W" ja-kanji)
("" "e" "}g[m e" "}g[m e" ja-kanji)
("" "R[" "R[Pcm R[" "R[Pcm R[" ja-kanji)
("" "GC" "iKCm GC" "iKCm GC" ja-kanji)
("" "TC" "YJjm TC" "YJjm TC" ja-kanji)
("" "Z" "Zm ZCW" "Zm ZCW" ja-kanji)
("" "Z" "Zm L[W" "Zm L[W" ja-kanji)
("" "g[" "n^m g[" "n^m g[" ja-kanji)
("" "" "gdim " "gdim " ja-kanji)
("" "R[" "Jm R[" "Jm R[" ja-kanji)
("" "Pc" "JPm L[W" "JPm L[W" ja-kanji)
("" "J" "qrm J" "qrm J" ja-kanji)
("" "I[" "PVm P" "PVm P" ja-kanji)
("" "C" "TJ_m C" "TJ_m C" ja-kanji)
("" "h" "rm CW" "rm CW" ja-kanji)
("" "J" "Jm L[W" "Jm L[W" ja-kanji)
("" "{[" "A~KVm {[" "A~KVm {[" ja-kanji)
("" "J" "}jm J" "}jm J" ja-kanji)
("" "[" "A~m [" "A~m [" ja-kanji)
("" "t" "KVj t" "KVj t" ja-kanji)
("" "R" "A~m R" "A~m R" ja-kanji)
("" "r" "lY~im i" "lY~im i" ja-kanji)
("" "A" "A~m A" "A~m A" ja-kanji)
("" "g[" "JSm g[" "JSm g[" ja-kanji)
("" "" "j nV" "j nV" ja-kanji)
("" "oc" "c~m ]NW" "c~m ]NW" ja-kanji)
("" "P" "j Lk" "j Lk" ja-kanji)
("" "q" "h[ucm qO}" "h[ucm qO}" ja-kanji)
("" "xL" "Lm xL" "Lm xL" ja-kanji)
("" "L" "^dim ZCW" "^dim ZCW" ja-kanji)
("" "L" "Lm L" "Lm L" ja-kanji)
("" "L[" "GrXm L[" "GrXm L[" ja-kanji)
("" "R[" "RqcWm R[" "RqcWm R[" ja-kanji)
("" "V[" "nWm V[" "nWm V[" ja-kanji)
("" "eC" "IqcWm eC" "IqcWm eC" ja-kanji)
("" "C" "JVJm C" "JVJm C" ja-kanji)
("" "O" "m CW" "m CW" ja-kanji)
("" "Jc" "LZCV^ qcW" "LZCV^ qcW" ja-kanji)
("" "M" "tNMm M" "tNMm M" ja-kanji)
("" "R[" "Acmm R[" "Acmm R[" ja-kanji)
("" "R[" "R[m CW" "R[m CW" ja-kanji)
("" "Z" "i}OTCm Z" "i}OTCm Z" ja-kanji)
("" "C" "Zm C" "Zm C" ja-kanji)
("" "[" "Cm L[W" "Cm L[W" ja-kanji)
("" "V" "qm V" "qm V" ja-kanji)
("" "XC" "JZ~m XC" "JZ~m XC" ja-kanji)
("" "N" "^XPm N" "^XPm N" ja-kanji)
("" "L[" "Ac}m L[" "Ac}m L[" ja-kanji)
("" "V[" "no^Nm V[" "no^Nm V[" ja-kanji)
("" "q" "JZ~m q" "JZ~m q" ja-kanji)
("" "Z" "Lm ZCW" "Lm ZCW" ja-kanji)
("" "w" "qKGm w" "qKGm w" ja-kanji)
("" "GC" "JQm GC" "JQm GC" ja-kanji)
("" "M[" "AQm M[" "AQm M[" ja-kanji)
("" "z" "qKGm z" "qKGm z" ja-kanji)
("" "L" "gVm L" "gVm L" ja-kanji)
("" "[" "IC{m [" "IC{m [" ja-kanji)
("" "ec" "gVm ec" "gVm ec" ja-kanji)
("" "C" "XLm C" "XLm C" ja-kanji)
("" "E" "NTMm E" "NTMm E" ja-kanji)
("" "n" "XLwj gG" "XLwj gG" ja-kanji)
("" "V" "XLm V" "XLm V" ja-kanji)
("" "W" "XNm W" "XNm W" ja-kanji)
("" "WN" "XNm WN" "XNm WN" ja-kanji)
("" "R[" "ALJm R[" "ALJm R[" ja-kanji)
("" "`" "nWm CW" "nWm CW" ja-kanji)
("" "[" "u[m [" "u[m [" ja-kanji)
("" "C" "LNm C" "LNm C" ja-kanji)
("" "Jc" "J}rXVC" "J}rXVC" ja-kanji)
("" "wC" "}lNm wC" "}lNm wC" ja-kanji)
("" "W" "Ac}m W" "Ac}m W" ja-kanji)
("" "ZC" "Rm CW" "Rm CW" ja-kanji)
("" "W[" "~~j T_}" "~~j T_}" ja-kanji)
("" "" "cim CW" "cim CW" ja-kanji)
("" "V[" "\rGm V[" "\rGm V[" ja-kanji)
("" "ZC" "RGm L[W" "RGm L[W" ja-kanji)
("" "\[" "TgVm L[W" "TgVm L[W" ja-kanji)
("" "W[" "TTNm W[" "TTNm W[" ja-kanji)
("" "fC" "~~AJm fC" "~~AJm fC" ja-kanji)
("" "`[" "LNm L[W" "LNm L[W" ja-kanji)
("" "Cc" "tfdNm Cc" "tfdNm Cc" ja-kanji)
("" "C" "iEm C" "iEm C" ja-kanji)
("" "V" "~Zm V" "~Zm V" ja-kanji)
("" "VN" "ccVm L[W" "ccVm L[W" ja-kanji)
("" "R[" "R[m R[" "R[m R[" ja-kanji)
("" "R[" "R[[m [" "R[[m [" ja-kanji)
("" "g" "nm g" "nm g" ja-kanji)
("" "gc" "cLwj E`" "cLwj E`" ja-kanji)
("" "{[" "IJXm CW" "IJXm CW" ja-kanji)
("" "[" "C{m [" "C{m [" ja-kanji)
("" "R[" "cLwj R[" "cLwj R[" ja-kanji)
("" "V" "Vm V" "Vm V" ja-kanji)
("" "TN" "qmMm TN" "qmMm TN" ja-kanji)
("" "`" "^Rm `" "^Rm `" ja-kanji)
("" "`[" "Jb`[m `[" "Jb`[m `[" ja-kanji)
("" "nC" "nCKm nC" "nCKm nC" ja-kanji)
("" "n" "^Jm n" "^Jm n" ja-kanji)
("" "~N" "~Nm ]NW" "~Nm ]NW" ja-kanji)
("" "R" "}^m R" "}^m R" ja-kanji)
("" "R[" "{[R[m R[" "{[R[m R[" ja-kanji)
("" "PC" "Xlm PC" "Xlm PC" ja-kanji)
("" "V[" "ITm V[" "ITm V[" ja-kanji)
("" "V" "N`rm CW" "N`rm CW" ja-kanji)
("" "z" "zWVm z" "zWVm z" ja-kanji)
("" "GL" "Lm V^m L" "Lm V^m L" ja-kanji)
("@" "YC" "g[m }Gm YC" "g[m }Gm YC" ja-kanji)
("A" "e" "AcCm e" "AcCm e" ja-kanji)
("B" "q" "q][m q" "q][m q" ja-kanji)
("C" "q" "qRcm q" "qRcm q" ja-kanji)
("D" "t" "n^m t" "n^m t" ja-kanji)
("E" "w" "^Rm w" "^Rm w" ja-kanji)
("F" "P" "XWm P" "XWm P" ja-kanji)
("G" "TC" "Gm TC" "Gm TC" ja-kanji)
("H" "ZC" "i}OTCm ZC" "i}OTCm ZC" ja-kanji)
("I" "m[" "m[m L[W" "m[m L[W" ja-kanji)
("J" "" "RGm " "RGm " ja-kanji)
("K" "Ic" "IbgZCm Ic" "IbgZCm Ic" ja-kanji)
("L" "JN" "JN}Nm JN" "JN}Nm JN" ja-kanji)
("M" "nN" "W[nNm nN" "W[nNm nN" ja-kanji)
("N" "{[" "{[R[m {[" "{[R[m {[" ja-kanji)
("O" "" "Z{lm " "Z{lm " ja-kanji)
("P" "R[" "jJm R[" "jJm R[" ja-kanji)
("Q" "RN" "cLwj Nj" "cLwj Nj" ja-kanji)
("R" "Zc" "cLwj L" "cLwj L" ja-kanji)
("S" "`c" "`cm ZCW" "`cm ZCW" ja-kanji)
("T" "`c" "WZCLm `c" "WZCLm `c" ja-kanji)
("U" "`[" "n^m CW" "n^m CW" ja-kanji)
("V" "W" "AuMm W" "AuMm W" ja-kanji)
("W" "n" "qmMm n" "qmMm n" ja-kanji)
("X" "XC" "XC][m XC" "XC][m XC" ja-kanji)
("Y" "JC" "i}Xm JC" "i}Xm JC" ja-kanji)
("Z" "YC" "Xlm CW" "Xlm CW" ja-kanji)
("[" "^" "Lm CW" "Lm CW" ja-kanji)
("\" "f" "Vm f" "Vm f" ja-kanji)
("]" "q" "qWm q" "qWm q" ja-kanji)
("^" "[" "A^m [" "A^m [" ja-kanji)
("_" "P" "JIm P" "JIm P" ja-kanji)
("`" "ZC" "z]m ZC" "z]m ZC" ja-kanji)
("a" "W" "Xlm W" "Xlm W" ja-kanji)
("b" "G" "mhm G" "mhm G" ja-kanji)
("c" "[" "[Qcm [" "[Qcm [" ja-kanji)
("d" "[" "W[[m [" "W[[m [" ja-kanji)
("e" "" "cLwj " "cLwj " ja-kanji)
("f" "][" "n^m L[W" "n^m L[W" ja-kanji)
("g" "" "R}Mm " "R}Mm " ja-kanji)
("h" "][" "Cm ][" "Cm ][" ja-kanji)
("i" "_C" "_Cm L[W" "_Cm L[W" ja-kanji)
("j" "V" "C^m V" "C^m V" ja-kanji)
("k" "" "qLgm " "qLgm " ja-kanji)
("l" "" "JSm " "JSm " ja-kanji)
("m" "V[" "EXYNm V[" "EXYNm V[" ja-kanji)
("n" "L[" "V[gm L[" "V[gm L[" ja-kanji)
("o" "" "A^Gm L[W" "A^Gm L[W" ja-kanji)
("p" "L[" "tCm L[W" "tCm L[W" ja-kanji)
("q" "V" "hm L[W" "hm L[W" ja-kanji)
("r" "V" "im V" "im V" ja-kanji)
("s" "z" "~Zm L[W" "~Zm L[W" ja-kanji)
("t" "Z" "tlwj L~" "tlwj L~" ja-kanji)
("u" "z[" "tlwj z[" "tlwj z[" ja-kanji)
("v" "J" "tlwj J" "tlwj J" ja-kanji)
("w" "`N" "wTLm `N" "wTLm `N" ja-kanji)
("x" "t" "nVPm t" "nVPm t" ja-kanji)
("y" "\[" "Z\[m \[" "Z\[m \[" ja-kanji)
("z" "\[" "tl J]G" "tl J]G" ja-kanji)
("{" "Z" "tlwj L" "tlwj L" ja-kanji)
("|" "\[" "tlwj \[" "tlwj \[" ja-kanji)
("}" "g[" "tlwm h[" "tlwm h[" ja-kanji)
("~" "M" "tlm \[r" "tlm \[r" ja-kanji)
("" "V[" "tlwj V[" "tlwj V[" ja-kanji)
("" "[" "[V[m [" "[V[m [" ja-kanji)
("" "" "tlwj " "tlwj " ja-kanji)
("" "" "gdim g" "gdim g" ja-kanji)
("" "" "gm NW" "gm NW" ja-kanji)
("" "J" "Jim J" "Jim J" ja-kanji)
("" "G" "Gm L[W" "Gm L[W" ja-kanji)
("" "\[" "NTm \[" "NTm \[" ja-kanji)
("" "KC" "Mm KC" "Mm KC" ja-kanji)
("" "VN" "VNNm VN" "VNNm VN" ja-kanji)
("" "{[" "XXLm {[" "XXLm {[" ja-kanji)
("" "J" "NTj Q" "NTj Q" ja-kanji)
("" "T" "Jm T" "Jm T" ja-kanji)
("" "X[" "nX[m X[" "nX[m X[" ja-kanji)
("" "t" "JIm t" "JIm t" ja-kanji)
("" "C" "IIoRm C" "IIoRm C" ja-kanji)
("" "L" "NTJj L" "NTJj L" ja-kanji)
("" "R[" "CVNm R[" "CVNm R[" ja-kanji)
("" "[" "[WNm [" "[WNm [" ja-kanji)
("" "V" "ATm V" "ATm V" ja-kanji)
("" "g[" "tLm g[" "tLm g[" ja-kanji)
("" "}C" "C`Sm }C" "C`Sm }C" ja-kanji)
("" "}C" "C`Sm ZCW" "C`Sm ZCW" ja-kanji)
("" "n" "ZCm qgc" "ZCm qgc" ja-kanji)
("" "t" "NTJj t" "NTJj t" ja-kanji)
("" "wC" "ELNTm wC" "ELNTm wC" ja-kanji)
("" "z[" "cc~m z[" "cc~m z[" ja-kanji)
("" "{[" "WTCm {[" "WTCm {[" ja-kanji)
("" "N" "NTJj " "NTJj " ja-kanji)
("" "oc" "WX~m oc" "WX~m oc" ja-kanji)
("" "[" "NTj ^c" "NTj ^c" ja-kanji)
("" "C" "Vglm C" "Vglm C" ja-kanji)
("" "JC" "ECL[m JC" "ECL[m JC" ja-kanji)
("" "JN" "}jm JN" "}jm JN" ja-kanji)
("" "V" "RRjm V" "RRjm V" ja-kanji)
("" "V" "O~m V" "O~m V" ja-kanji)
("" "W" "WVm W" "WVm W" ja-kanji)
("" "W" "fm W" "fm W" ja-kanji)
("" "Z" "VLjm Z" "VLjm Z" ja-kanji)
("" "g[" "NTj S[" "NTj S[" ja-kanji)
("" "tN" "NTj tZ" "NTj tZ" ja-kanji)
("" "{[" "{[[m {[" "{[[m {[" ja-kanji)
("" "C" "~[Km C" "~[Km C" ja-kanji)
("" "C" "IIjm C" "IIjm C" ja-kanji)
("" "" "NTj NC" "NTj NC" ja-kanji)
("" "G" "Vm G" "Vm G" ja-kanji)
("" "K" "cm Mm K" "cm Mm K" ja-kanji)
("" "K" "c{~m K" "c{~m K" ja-kanji)
("" "L[" "Tm L[" "Tm L[" ja-kanji)
("" "PC" "NLm L[W" "NLm L[W" ja-kanji)
("" "S" "NTj N" "NTj N" ja-kanji)
("" "T" "NTJj T" "NTJj T" ja-kanji)
("" "`" "NRm `" "NRm `" ja-kanji)
("" "\[" "VLm L[W" "VLm L[W" ja-kanji)
("" "g" "NTj A}" "NTj A}" ja-kanji)
("" "g" "ETMm CW" "ETMm CW" ja-kanji)
("" "g[" "}m g[" "}m g[" ja-kanji)
("" "W" "VmuOTm W" "VmuOTm W" ja-kanji)
("" "[" "lRWVm [" "lRWVm [" ja-kanji)
("" "" "WX~m " "WX~m " ja-kanji)
("" "[" "^oRm [" "^oRm [" ja-kanji)
("" "A" "CIm L[W" "CIm L[W" ja-kanji)
("" "M" "XiOTm M" "XiOTm M" ja-kanji)
("" "L" "X~m L" "X~m L" ja-kanji)
("" "R" "JIOTm R" "JIOTm R" ja-kanji)
("" "VN" "}m VN" "}m VN" ja-kanji)
("" "XC" "Ac}m XC" "Ac}m XC" ja-kanji)
("" "X[" "XYim X[" "XYim X[" ja-kanji)
("" "ZC" "NTj c}" "NTj c}" ja-kanji)
("" "ZC" "Jum ZC" "Jum ZC" ja-kanji)
("" "V[" "z[Lm V[" "z[Lm V[" ja-kanji)
("" "`[" "NTj iKC" "NTj iKC" ja-kanji)
("" "n" "NTj i~" "NTj i~" ja-kanji)
("" "q" "NTj AY" "NTj AY" ja-kanji)
("" "wC" "ELNTm wC" "ELNTm wC" ja-kanji)
("" "z[" "NTj A" "NTj A" ja-kanji)
("" "z[" "oGm z[" "oGm z[" ja-kanji)
("" "[" "NTm [" "NTm [" ja-kanji)
("" "" "N\[m im " "N\[m im " ja-kanji)
("" "[" "qVm [" "qVm [" ja-kanji)
("" "" "NTj nV" "NTj nV" ja-kanji)
("" "J" "Vm J" "Vm J" ja-kanji)
("" "J" "NTJj J" "NTJj J" ja-kanji)
("" "KN" "Eeim KN" "Eeim KN" ja-kanji)
("" "KN" "KNm L[W" "KNm L[W" ja-kanji)
("" "J" "NTj J" "NTj J" ja-kanji)
("" "N" "NVm N" "NVm N" ja-kanji)
("" "R" "jjNm N" "jjNm N" ja-kanji)
("" "X[" "zVOTm X[" "zVOTm X[" ja-kanji)
("" "^" "NQm ^" "NQm ^" ja-kanji)
("" "eC" "w^m eC" "w^m eC" ja-kanji)
("" "n" "Jxm n" "Jxm n" ja-kanji)
("" "z[" "JNXm z[" "JNXm z[" ja-kanji)
("" "}" "}m L[W" "}m L[W" ja-kanji)
("" "N" "IVxm Z^" "IVxm Z^" ja-kanji)
("" "V" "NTj zhRX" "NTj zhRX" ja-kanji)
("" "" "`Tm " "`Tm " ja-kanji)
("" "I[" "NTj ILi" "NTj ILi" ja-kanji)
("" "KC" "t^m CW" "t^m CW" ja-kanji)
("" "P" "NTj Jl" "NTj Jl" ja-kanji)
("" "R[" "Mm R[" "Mm R[" ja-kanji)
("" "R" "RjNm R" "RjNm R" ja-kanji)
("" "U" "NTj X" "NTj X" ja-kanji)
("" "V" "gMm V" "gMm V" ja-kanji)
("" "WN" "RjNm jN" "RjNm jN" ja-kanji)
("" "V[" "NTj IT" "NTj IT" ja-kanji)
("" "WN" "Vglm WN" "Vglm WN" ja-kanji)
("" "V" "NTm V" "NTm V" ja-kanji)
("" "ZL" "Vm ZL" "Vm ZL" ja-kanji)
("" "q" "q}m q" "q}m q" ja-kanji)
("" "{[" "S{[m {[" "S{[m {[" ja-kanji)
("" "TC" "NTj }c" "NTj }c" ja-kanji)
("" "VN" "NTj h" "NTj h" ja-kanji)
("" "W" "im V" "im V" ja-kanji)
("" "V" "Vg[m V" "Vg[m V" ja-kanji)
("" "T" "jWm W" "jWm W" ja-kanji)
("" "\" "AImm \" "AImm \" ja-kanji)
("" "\[" "Ac}m \[" "Ac}m \[" ja-kanji)
("" "^C" "w^m ^C" "w^m ^C" ja-kanji)
("" "tN" "ztNm tN" "ztNm tN" ja-kanji)
("" "[" "^fVim ^f" "^fVim ^f" ja-kanji)
("" "LN" "NTj LN" "NTj LN" ja-kanji)
("" "V" "NQm V" "NQm V" ja-kanji)
("" "W[" "^LMm W[" "^LMm W[" ja-kanji)
("" "V" "Lm Rm V" "Lm Rm V" ja-kanji)
("@" "W" "CNTm W" "CNTm W" ja-kanji)
("A" "YC" "Vxm YC" "Vxm YC" ja-kanji)
("B" "YC" "YCm ]NW" "YCm ]NW" ja-kanji)
("C" "[" "JKl\[" "JKl\[" ja-kanji)
("D" "E" "}cm E" "}cm E" ja-kanji)
("E" "JC" "bL[m JC" "bL[m JC" ja-kanji)
("F" "JC" "VQm JC" "VQm JC" ja-kanji)
("G" "L[" "nWJ~m L[" "nWJ~m L[" ja-kanji)
("H" "PC" "AU~m PC" "AU~m PC" ja-kanji)
("I" "R[" "~}Jm R[" "~}Jm R[" ja-kanji)
("J" "V[" "Mm V[" "Mm V[" ja-kanji)
("K" "VN" "om o" "om o" ja-kanji)
("L" "Zc" "Mm Zc" "Mm Zc" ja-kanji)
("M" "\[" "um L[W" "um L[W" ja-kanji)
("N" "r" "om " "om " ja-kanji)
("O" "wC" "wCm wC" "wCm wC" ja-kanji)
("P" "" "NTj AYP" "NTj AYP" ja-kanji)
("Q" "C" "c{~m C" "c{~m C" ja-kanji)
("R" "[" "NTj [" "NTj [" ja-kanji)
("S" "ZL" "[[Lm [L" "[[Lm [L" ja-kanji)
("T" "ZC" "iYim ZC" "iYim ZC" ja-kanji)
("U" "][" "Nm L[W" "Nm L[W" ja-kanji)
("V" "^C" "tLmg[m g[" "tLmg[m g[" ja-kanji)
("W" "r[" "JYm r[" "JYm r[" ja-kanji)
("X" "O[" "nXlm O[" "nXlm O[" ja-kanji)
("Y" "QC" "QCm L[W" "QCm L[W" ja-kanji)
("Z" "N" "NXm L[W" "NXm L[W" ja-kanji)
("[" "C" "AJUm C" "AJUm C" ja-kanji)
("\" "AC" "VQm AC" "VQm AC" ja-kanji)
("]" "E" "cm E" "cm E" ja-kanji)
("^" "\" "~KGm CW" "~KGm CW" ja-kanji)
("_" "q" "J^o~m q" "J^o~m q" ja-kanji)
("`" "C" "JM" "JM" ja-kanji)
("a" "" "COTm " "COTm " ja-kanji)
("b" "" "AVm L[W" "AVm L[W" ja-kanji)
("c" "[" "IIP_em [" "IIP_em [" ja-kanji)
("d" "Z" "RPm Z" "RPm Z" ja-kanji)
("e" "}" "NTm Jc" "NTm Jc" ja-kanji)
("f" "" "JYm " "JYm " ja-kanji)
("g" "R" "gKVm R" "gKVm R" ja-kanji)
("h" "R" "gm CW" "gm CW" ja-kanji)
("i" "P" "ccVm P" "ccVm P" ja-kanji)
("j" "S[" "TPum L[W" "TPum L[W" ja-kanji)
("k" "L" "JPm L" "JPm L" ja-kanji)
("l" "Vc" "V~m Vc" "V~m Vc" ja-kanji)
("m" "C" "~~Ym C" "~~Ym C" ja-kanji)
("n" "R[" "Jfm R[" "Jfm R[" ja-kanji)
("o" "V" "Em V" "Em V" ja-kanji)
("p" "g[" "Vwj g[" "Vwj g[" ja-kanji)
("q" "[C" "um [C" "um [C" ja-kanji)
("r" "z[" "huKCm z[" "huKCm z[" ja-kanji)
("s" "J" "Vwj A}C" "Vwj A}C" ja-kanji)
("t" "L[" "~~Ym L[" "~~Ym L[" ja-kanji)
("u" "R" "Pm R" "Pm R" ja-kanji)
("v" "V" "EWm V" "EWm V" ja-kanji)
("w" "[" "Vwj " "Vwj " ja-kanji)
("x" "C" "JQ[m [" "JQ[m [" ja-kanji)
("y" "C" "JLm C" "JLm C" ja-kanji)
("z" "z[" "Vwj cc" "Vwj cc" ja-kanji)
("{" "JC" "nm Vm JC" "nm Vm JC" ja-kanji)
("|" "Jc" "iNWm Jc" "iNWm Jc" ja-kanji)
("}" "L[" "RIMm L[" "RIMm L[" ja-kanji)
("~" "L[" "L[Im L[" "L[Im L[" ja-kanji)
("" "R[" "~Y`m R[" "~Y`m R[" ja-kanji)
("" "V" "Nm V" "Nm V" ja-kanji)
("" "[" "Vwj IC" "Vwj IC" ja-kanji)
("" "G" "Vwj m r" "Vwj m r" ja-kanji)
("" "P" "VW~m P" "VW~m P" ja-kanji)
("" "S" "SR[m S" "SR[m S" ja-kanji)
("" "VN" "CVm VN" "CVm VN" ja-kanji)
("" "V" "VL[m V" "VL[m V" ja-kanji)
("" "[C" "kPKm [C" "kPKm [C" ja-kanji)
("" "^" "^Wm ^" "^Wm ^" ja-kanji)
("" "t" "JQ[m JQ" "JQ[m JQ" ja-kanji)
("" "" "Vwj A}" "Vwj A}" ja-kanji)
("" "[" "TiMm [" "TiMm [" ja-kanji)
("" "" "ATm " "ATm " ja-kanji)
("" "GL" "Vwj GL" "Vwj GL" ja-kanji)
("" "G" "Vwj Ae" "Vwj Ae" ja-kanji)
("" "P" "jim P" "jim P" ja-kanji)
("" "ZC" "g{m ZC" "g{m ZC" ja-kanji)
("" "ZL" "gJQm ZL" "gJQm ZL" ja-kanji)
("" "`[" "qOVm `[" "qOVm `[" ja-kanji)
("" "q" "AuVm q" "AuVm q" ja-kanji)
("" "tN" "R[m " "R[m " ja-kanji)
("" "C" "CV[m C" "CV[m C" ja-kanji)
("" "J" "J^cm J" "J^cm J" ja-kanji)
("" "J" "I^}WNVm J" "I^}WNVm J" ja-kanji)
("" "Jc" "T\m Jc" "T\m Jc" ja-kanji)
("" "R" "R`[m m R" "R`[m m R" ja-kanji)
("" "R[" "CiSm R[" "CiSm R[" ja-kanji)
("" "Vc" "V~m CW" "V~m CW" ja-kanji)
("" "tN" "}Vm tN" "}Vm tN" ja-kanji)
("" "w" "R[m R[" "R[m R[" ja-kanji)
("" "" "Vwj " "Vwj " ja-kanji)
("" "[" "JQ[m [" "JQ[m [" ja-kanji)
("" "g[" "Nm CbVm g[" "Nm CbVm g[" ja-kanji)
("" "[" "nGm L[W" "nGm L[W" ja-kanji)
("" "PC" "z^m L[W" "z^m L[W" ja-kanji)
("" "C" "YCVm C" "YCVm C" ja-kanji)
("" "[" "g[[m [" "g[[m [" ja-kanji)
("" "S[" "nT~m S[" "nT~m S[" ja-kanji)
("" "Vc" "LMXm Vc" "LMXm Vc" ja-kanji)
("" "V[" "CiSm V[" "CiSm V[" ja-kanji)
("" "Vc" "Vwj \c" "Vwj \c" ja-kanji)
("" "W[" "Vwj cl" "Vwj cl" ja-kanji)
("" "XC" "CGhm XC" "CGhm XC" ja-kanji)
("" "ZL" "TXm ZL" "TXm ZL" ja-kanji)
("" "`c" "`bLm `c" "`bLm `c" ja-kanji)
("" "g[" "Vwj h[" "Vwj h[" ja-kanji)
("" "oN" "qLKGm oN" "qLKGm oN" ja-kanji)
("" "oN" "oNm CW" "oNm CW" ja-kanji)
("" "[" "IPm [" "IPm [" ja-kanji)
("" "M[" "M[`[m M[" "M[`[m M[" ja-kanji)
("" "`[" "Vm L[W" "Vm L[W" ja-kanji)
("" "n" "_J}m n" "_J}m n" ja-kanji)
("" "JC" "Jjm CW" "Jjm CW" ja-kanji)
("" "Jc" "T\m Jc" "T\m Jc" ja-kanji)
("" "Z" "qLKGm Z" "qLKGm Z" ja-kanji)
("" "eC" "}em eC" "}em eC" ja-kanji)
("" "g[" "g[[m g[" "g[[m g[" ja-kanji)
("" "{[" "Eo~m {[" "Eo~m {[" ja-kanji)
("" "{[" "{[m CW" "{[m CW" ja-kanji)
("" "GC" "Vwj TJG" "Vwj TJG" ja-kanji)
("" "JN" "VNgV" "VNgV" ja-kanji)
("" "[" "ESNm [" "ESNm [" ja-kanji)
("" "V" "ESNm V" "ESNm V" ja-kanji)
("" "C" "qTSm C" "qTSm C" ja-kanji)
("" "R" "}hXm R" "}hXm R" ja-kanji)
("" "T" "JCRm L[W" "JCRm L[W" ja-kanji)
("" "g" "gm ZCW" "gm ZCW" ja-kanji)
("" "g" "LNCVm g" "LNCVm g" ja-kanji)
("" "o" "GrXm L[W" "GrXm L[W" ja-kanji)
("" "WN" "NWPm WN" "NWPm WN" ja-kanji)
("" "WN" "NWPm CW" "NWPm CW" ja-kanji)
("" "Q" "eEm Q" "eEm Q" ja-kanji)
("" "K" "KiCm K" "KiCm K" ja-kanji)
("" "GC" "}m L[W" "}m L[W" ja-kanji)
("" "N" "`}^m N" "`}^m N" ja-kanji)
("" "T" "Wom T" "Wom T" ja-kanji)
("" "G" "GZCKC" "GZCKC" ja-kanji)
("" "L" "h[Lm L" "h[Lm L" ja-kanji)
("" "R" "RVNm R" "RVNm R" ja-kanji)
("" "Wc" "ARm Wc" "ARm Wc" ja-kanji)
("" "W" "Gm W" "Gm W" ja-kanji)
("" "W" "Wm CW" "Wm CW" ja-kanji)
("" "m[" "PTm m[" "PTm m[" ja-kanji)
("" "Pc" "^gm Pc" "^gm Pc" ja-kanji)
("" "V" "qgGmm V" "qgGmm V" ja-kanji)
("" "^" "n_kOm ^" "n_kOm ^" ja-kanji)
("" "l" "Rwj V[" "Rwj V[" ja-kanji)
("" "n" "Rwj V" "Rwj V" ja-kanji)
("" "n" "iKWom o" "iKWom o" ja-kanji)
("" "z[" "^Cm z[" "^Cm z[" ja-kanji)
("" "{[" "iKTm {[" "iKTm {[" ja-kanji)
("" "{" "nnj R" "nnj R" ja-kanji)
("" "PC" "E`JPm PC" "E`JPm PC" ja-kanji)
("" "tN" "tNTm tN" "tNTm tN" ja-kanji)
("" "J~V" "Rj W[Q" "Rj W[Q" ja-kanji)
("" "R[" "Rj CN" "Rj CN" ja-kanji)
("" "GC" "X\m GC" "X\m GC" ja-kanji)
("" "L[" "JSm L[" "JSm L[" ja-kanji)
("" "N" "X\m N" "X\m N" ja-kanji)
("" "\[" "\IEm L[W" "\IEm L[W" ja-kanji)
("" "J" "ccm J" "ccm J" ja-kanji)
("" "JC" "EMm JC" "EMm JC" ja-kanji)
("" "ZL" "n_kOm ZL" "n_kOm ZL" ja-kanji)
("" "nC" "iKC Lm" "iKC Lm" ja-kanji)
("" "q" "^XPm q" "^XPm q" ja-kanji)
("" "[" "E`JPm [" "E`JPm [" ja-kanji)
("" "TC" "Rj c}" "Rj c}" ja-kanji)
("" "R" "thVm R" "thVm R" ja-kanji)
("" "w" "Z}Cm w" "Z}Cm w" ja-kanji)
("" "z" "llRm z" "llRm z" ja-kanji)
("" "z[" "zm L[W" "zm L[W" ja-kanji)
("" "I" "hem I" "hem I" ja-kanji)
("" "WN" "WN\[m WN" "WN\[m WN" ja-kanji)
("" "^C" "Rj V]N" "Rj V]N" ja-kanji)
("" "`" "EoEm `" "EoEm `" ja-kanji)
("" "L[" "Rj cC" "Rj cC" ja-kanji)
("" "W[" "m{m W[" "m{m W[" ja-kanji)
("" "Zc" "CZcm Zc" "CZcm Zc" ja-kanji)
("" "V[" "q_m V[" "q_m V[" ja-kanji)
("" "" "Gm " "Gm " ja-kanji)
("" "^" "qgGm ^" "qgGm ^" ja-kanji)
("" "^" "^m NW" "^m NW" ja-kanji)
("" "g[" "E`JPm g[" "E`JPm g[" ja-kanji)
("" "wL" "q_m wL" "q_m wL" ja-kanji)
("@" "W" "iKWom W" "iKWom W" ja-kanji)
("A" "" "{m " "{m " ja-kanji)
("B" "Pc" "c}oTm Pc" "c}oTm Pc" ja-kanji)
("C" "xc" "^rm xc" "^rm xc" ja-kanji)
("D" "V" "n_Mm V" "n_Mm V" ja-kanji)
("E" "" "Rj " "Rj " ja-kanji)
("F" "L" "Rj AQ" "Rj AQ" ja-kanji)
("G" "A" "IIEm A" "IIEm A" ja-kanji)
("H" "^" "tJNe qC" "tJNe qC" ja-kanji)
("I" "JN" "Vxm JN" "Vxm JN" ja-kanji)
("J" "L" "^Yim L" "^Yim L" ja-kanji)
("K" "xL" "gm xL" "gm xL" ja-kanji)
("L" "e" "EJKEm e" "EJKEm e" ja-kanji)
("M" "QL" "IgRm ~R" "IgRm ~R" ja-kanji)
("N" "g" "~m g" "~m g" ja-kanji)
("O" "" "RClKEm " "RClKEm " ja-kanji)
("P" "L" "EJKEm L" "EJKEm L" ja-kanji)
("Q" "R[" "tg~m R[" "tg~m R[" ja-kanji)
("R" "L" "}~Gm L" "}~Gm L" ja-kanji)
("S" "JN" "I{Gm L[W" "I{Gm L[W" ja-kanji)
("T" "" "~m L[W" "~m L[W" ja-kanji)
("U" "eL" "}~Gm eL" "}~Gm eL" ja-kanji)
("V" "J" "~m L[W" "~m L[W" ja-kanji)
("W" "R" "TJYLm R" "TJYLm R" ja-kanji)
("X" "V" "N`oVm V" "N`oVm V" ja-kanji)
("Y" "eC" "cmwm eC" "cmwm eC" ja-kanji)
("Z" "JC" "cmwj qcW" "cmwj qcW" ja-kanji)
("[" "V[" "TJYLm V[" "TJYLm V[" ja-kanji)
("\" "VN" "Tm L[W" "Tm L[W" ja-kanji)
("]" "t" "tz[m t" "tz[m t" ja-kanji)
("^" "Lc" "Im Lc" "Im Lc" ja-kanji)
("_" "Pc" "AoNm Pc" "AoNm Pc" ja-kanji)
("`" "R[" "NYm R[" "NYm R[" ja-kanji)
("a" "J" "i}m J" "i}m J" ja-kanji)
("b" "J" "CuJm J" "CuJm J" ja-kanji)
("c" "gc" "hm gc" "hm gc" ja-kanji)
("d" "J" "VJm J" "VJm J" ja-kanji)
("e" "R" "~m R" "~m R" ja-kanji)
("f" "\" "mEm \" "mEm \" ja-kanji)
("g" "^C" "mRXm ^C" "mRXm ^C" ja-kanji)
("h" "eC" "\Vm eC" "\Vm eC" ja-kanji)
("i" "" "mmVm " "mmVm " ja-kanji)
("j" "JC" "IhPm JC" "IhPm JC" ja-kanji)
("k" "L" "\Nm L" "\Nm L" ja-kanji)
("l" "S" "nYJVm S" "nYJVm S" ja-kanji)
("m" "W" "nJm W" "nJm W" ja-kanji)
("n" "`[" "Zm `[" "Zm `[" ja-kanji)
("o" "`[" "AcGm `[" "AcGm `[" ja-kanji)
("p" "C" "CVm C" "CVm C" ja-kanji)
("q" "JC" "IVGm JC" "IVGm JC" ja-kanji)
("r" "JC" "C}Vm JC" "C}Vm JC" ja-kanji)
("s" "L[" "^uJXm L[" "^uJXm L[" ja-kanji)
("t" "R[" "cQm R[" "cQm R[" ja-kanji)
("u" "V[" "giGm V[" "giGm V[" ja-kanji)
("v" "V[" "Zm V[" "Zm V[" ja-kanji)
("w" "u" "VCm u" "VCm u" ja-kanji)
("x" "W" "NhCm W" "NhCm W" ja-kanji)
("y" "\[" "CTJEm \[" "CTJEm \[" ja-kanji)
("z" "e" "wcEm e" "wcEm e" ja-kanji)
("{" "W[" "VJgm W[" "VJgm W[" ja-kanji)
("|" "J" "CTm CW" "CTm CW" ja-kanji)
("}" "A" "\Wm A" "\Wm A" ja-kanji)
("~" "JC" "nCJCm JC" "nCJCm JC" ja-kanji)
("" "KN" "`NQm KN" "`NQm KN" ja-kanji)
("" "L" "Cm L" "Cm L" ja-kanji)
("" "MN" "^m MN" "^m MN" ja-kanji)
("" "P" "J}rXVC" "J}rXVC" ja-kanji)
("" "R" "IhPm R" "IhPm R" ja-kanji)
("" "t[" "t[Vm t[" "t[Vm t[" ja-kanji)
("" "w" "wQm w" "wQm w" ja-kanji)
("" "[" "wcEm [" "wcEm [" ja-kanji)
("" "J" "E^m J" "E^m J" ja-kanji)
("" "P" "}Rgm P" "}Rgm P" ja-kanji)
("" "V" "INim V" "INim V" ja-kanji)
("" "V" "Vm CW" "Vm CW" ja-kanji)
("" "VN" "^cm VN" "^cm VN" ja-kanji)
("" "qc" "XJm qc" "XJm qc" ja-kanji)
("" "{[" "\Vm {[" "\Vm {[" ja-kanji)
("" "[" "E^Em [" "E^Em [" ja-kanji)
("" "I[" "E^Em I[" "E^Em I[" ja-kanji)
("" "LN" "Lm LN" "Lm LN" ja-kanji)
("" "PC" "VuLm PC" "VuLm PC" ja-kanji)
("" "^N" "Zm ^N" "Zm ^N" ja-kanji)
("" "}" "Aihm }" "Aihm }" ja-kanji)
("" "{" "nJm {" "nJm {" ja-kanji)
("" "J" "ETCm J" "ETCm J" ja-kanji)
("" "J" "i}m J" "i}m J" ja-kanji)
("" "L" "\Vm L" "\Vm L" ja-kanji)
("" "Lc" "Ccm Lc" "Ccm Lc" ja-kanji)
("" "V[" "AJVm L[W" "AJVm L[W" ja-kanji)
("" "Z" "Zm CW" "Zm CW" ja-kanji)
("" "Z" "\Vm Z" "\Vm Z" ja-kanji)
("" "^" "niVm ^" "niVm ^" ja-kanji)
("" "Z" "^Sgm Z" "^Sgm Z" ja-kanji)
("" "\[" "TOm \[" "TOm \[" ja-kanji)
("" "q" "^gGm q" "^gGm q" ja-kanji)
("" "N" "Pm L[W" "Pm L[W" ja-kanji)
("" "P" "Zm P" "Zm P" ja-kanji)
("" "" "z}m L[W" "z}m L[W" ja-kanji)
("" "hN" "~Em ~" "~Em ~" ja-kanji)
("" "G" "TJm G" "TJm G" ja-kanji)
("" "V[" "J^Lm L[W" "J^Lm L[W" ja-kanji)
("" "U" "\Vm U" "\Vm U" ja-kanji)
("" "W[" "Ym L[W" "Ym L[W" ja-kanji)
("" "V" "VVm V" "VVm V" ja-kanji)
("" "J" "Rum J" "Rum J" ja-kanji)
("" "T" "^^Gm L[W" "^^Gm L[W" ja-kanji)
("" "J" "R_}m J" "R_}m J" ja-kanji)
("" "Jc" "qCm Jc" "qCm Jc" ja-kanji)
("" "PC" "^jm L[W" "^jm L[W" ja-kanji)
("" "L" "Ajm L" "Ajm L" ja-kanji)
("" "G" "Gh[m G" "Gh[m G" ja-kanji)
("" "W" "^em L[W" "^em L[W" ja-kanji)
("" "z[" "^Jm L[W" "^Jm L[W" ja-kanji)
("" "V" "CmRm V" "CmRm V" ja-kanji)
("" "P" "ViEm P" "ViEm P" ja-kanji)
("" "`" "CmVVm L[W" "CmVVm L[W" ja-kanji)
("" "`" "Wim `" "Wim `" ja-kanji)
("" "TC" "}Ckm TC" "}Ckm TC" ja-kanji)
("" "`[" "h[ucm `[" "h[ucm `[" ja-kanji)
("" "JN" "h[ucm Wi" "h[ucm Wi" ja-kanji)
("" "L[" "Wij X" "Wij X" ja-kanji)
("" "nN" "Pmm i}G" "Pmm i}G" ja-kanji)
("" "" "^kLm L[W" "^kLm L[W" ja-kanji)
("" "QC" "VJm Rm QC" "VJm Rm QC" ja-kanji)
("" "q" "qm ZCW" "qm ZCW" ja-kanji)
("" "q" "[W[m i" "[W[m i" ja-kanji)
("" "oN" "h[ucm oN" "h[ucm oN" ja-kanji)
("" "UC" "^Jm L[W" "^Jm L[W" ja-kanji)
("" "Vc" "^_Xm L[W" "^_Xm L[W" ja-kanji)
("" "h" "T{m h" "T{m h" ja-kanji)
("" "C" "INm C" "INm C" ja-kanji)
("" "V" "^Jm V" "^Jm V" ja-kanji)
("" "j[" "t^cm L[W" "t^cm L[W" ja-kanji)
("" "j" "t^cm CW" "t^cm CW" ja-kanji)
("" "w" "IgVm w" "IgVm w" ja-kanji)
("" "R" "V[jm R" "V[jm R" ja-kanji)
("" "z" "JUm z" "JUm z" ja-kanji)
("" "Z" "CVCm Z" "CVCm Z" ja-kanji)
("" "oC" "Em L[W" "Em L[W" ja-kanji)
("" "C" "^}m C" "^}m C" ja-kanji)
("" "TC" "TCZm TC" "TCZm TC" ja-kanji)
("" "^" "XJXm P" "XJXm P" ja-kanji)
("" "t" "INm t" "INm t" ja-kanji)
("" "V" "CPjGm jG" "CPjGm jG" ja-kanji)
("" "[C" "_m [C" "_m [C" ja-kanji)
("" "T" "^XPm ZCW" "^XPm ZCW" ja-kanji)
("" "C" "EcNVC T}" "EcNVC T}" ja-kanji)
("" "GC" "jVm " "jVm " ja-kanji)
("" "Z" "jMXm Z" "jMXm Z" ja-kanji)
("" "W" "niPm W" "niPm W" ja-kanji)
("" "ZC" "^Xm ZC" "^Xm ZC" ja-kanji)
("" "][" "][UCm ][" "][UCm ][" ja-kanji)
("" "\[" "JCwj V[" "JCwj V[" ja-kanji)
("" "q" "qCLm qC" "qCLm qC" ja-kanji)
("" "VN" "AKiEm VN" "AKiEm VN" ja-kanji)
("" "^" "AJm ^" "AJm ^" ja-kanji)
("" "V" "AJCm V" "AJCm V" ja-kanji)
("" "\[" "nVm CW" "nVm CW" ja-kanji)
("" "L[" "^P_PVC" "^P_PVC" ja-kanji)
("" "`" "nVm `" "nVm `" ja-kanji)
("" "`[" "RNCm `[" "RNCm `[" ja-kanji)
("" "L" "c}_em L" "c}_em L" ja-kanji)
("" "V" "AVm V" "AVm V" ja-kanji)
("" "t" "AOm t" "AOm t" ja-kanji)
("" "J" "Jtm J" "Jtm J" ja-kanji)
("" "T" "AVwj Tc" "AVwj Tc" ja-kanji)
("" "ZL" "AVwj CV" "AVwj CV" ja-kanji)
("" "ec" "c}YNm ec" "c}YNm ec" ja-kanji)
("" "n" "nR[m n" "nR[m n" ja-kanji)
("" "oc" "ocum oc" "ocum oc" ja-kanji)
("" "L" "qU}YNm L" "qU}YNm L" ja-kanji)
("" "L[" "AVIgm L[" "AVIgm L[" ja-kanji)
("" "R" "cPm R" "cPm R" ja-kanji)
("" "Z" "n_Vm Z" "n_Vm Z" ja-kanji)
("" "LN" "JKm LN" "JKm LN" ja-kanji)
("" "\" "EgCm CW" "EgCm CW" ja-kanji)
("" "[" "Nm [" "Nm [" ja-kanji)
("" "g" "AVwj nV" "AVwj nV" ja-kanji)
("" "J" "NuVm J" "NuVm J" ja-kanji)
("" "L" "EYN}m L" "EYN}m L" ja-kanji)
("" "Z" "IRiEm L[W" "IRiEm L[W" ja-kanji)
("" "`" "AVwj V" "AVwj V" ja-kanji)
("" "W[" "t~jWm W[" "t~jWm W[" ja-kanji)
("" "V[" "JJgm V[" "JJgm V[" ja-kanji)
("" "" "RGm " "RGm " ja-kanji)
("" "[" "Ihm CW" "Ihm CW" ja-kanji)
("" "PC" "~`m PC" "~`m PC" ja-kanji)
("@" "P" "PPm P" "PPm P" ja-kanji)
("A" "T" "c}YNm T" "c}YNm T" ja-kanji)
("B" "\[" "Nm \[" "Nm \[" ja-kanji)
("C" "ZL" "AVwj ZL" "AVwj ZL" ja-kanji)
("D" "g[" "tm CW" "tm CW" ja-kanji)
("E" "VN" "Z}m VN" "Z}m VN" ja-kanji)
("F" "V[" "Agm V[" "Agm V[" ja-kanji)
("G" "ZL" "AVEm ZL" "AVEm ZL" ja-kanji)
("H" "\[" "AVwj \[" "AVwj \[" ja-kanji)
("I" "}" "Nm }" "Nm }" ja-kanji)
("J" "qc" "TLoCm qc" "TLoCm qc" ja-kanji)
("K" "Pc" "c}YNm Pc" "c}YNm Pc" ja-kanji)
("L" "\" "EYN}m \" "EYN}m \" ja-kanji)
("M" "{N" "~YJLm zN" "~YJLm zN" ja-kanji)
("N" "\[" "TOm \[" "TOm \[" ja-kanji)
("O" "`" "^Em `" "^Em `" ja-kanji)
("P" "`N" "eL`Nm `N" "eL`Nm `N" ja-kanji)
("Q" "wL" "CUm wL" "CUm wL" ja-kanji)
("R" "ZC" "m{m ZC" "m{m ZC" ja-kanji)
("S" "`[" "^Em `[" "^Em `[" ja-kanji)
("T" "`" "c}YNm `" "c}YNm `" ja-kanji)
("U" "eL" "eL`Nm eL" "eL`Nm eL" ja-kanji)
("V" "e" "tm e" "tm e" ja-kanji)
("W" "" "tm " "tm " ja-kanji)
("X" "" "m CW" "m CW" ja-kanji)
("Y" "W[" "tm W[" "tm W[" ja-kanji)
("Z" "L[" "~YJm L[" "~YJm L[" ja-kanji)
("[" "^C" "J_m CW" "J_m CW" ja-kanji)
("\" "^C" "J_m L[W" "J_m L[W" ja-kanji)
("]" "^" "JXm ^" "JXm ^" ja-kanji)
("^" "r" "~wj EcNVC" "~wj EcNVC" ja-kanji)
("_" "J" "~wj J" "~wj J" ja-kanji)
("`" "I[" "~wj I[" "~wj I[" ja-kanji)
("a" "Ac" "LVm Ac" "LVm Ac" ja-kanji)
("b" "N" "NrLm N" "NrLm N" ja-kanji)
("c" "S[" "ghNm NW" "ghNm NW" ja-kanji)
("d" "ec" "N}wj TL" "N}wj TL" ja-kanji)
("e" "J" "N}wj J" "N}wj J" ja-kanji)
("f" "V" "C^m V" "C^m V" ja-kanji)
("g" "VN" "N}wj VL" "N}wj VL" ja-kanji)
("h" "`" "N}j C^" "N}j C^" ja-kanji)
("i" "JN" "N}wj JN" "N}wj JN" ja-kanji)
("j" "PC" "JCm L[W" "JCm L[W" ja-kanji)
("k" "`[" "Xi`m `[" "Xi`m `[" ja-kanji)
("l" "`[" "`[m ]NW" "`[m ]NW" ja-kanji)
("m" "o" "qNm o" "qNm o" ja-kanji)
("n" "V" "jO}m V" "jO}m V" ja-kanji)
("o" "ec" "ghm ec" "ghm ec" ja-kanji)
("p" "[" "[m L[W" "[m L[W" ja-kanji)
("q" "[" "V[m [" "V[m [" ja-kanji)
("r" "" "erLO}" "erLO}" ja-kanji)
("s" "\[" "Ac}m \[" "Ac}m \[" ja-kanji)
("t" "tN" "tNVm tN" "tNVm tN" ja-kanji)
("u" "tN" "N}j tN" "N}j tN" ja-kanji)
("v" "G" "iKGm G" "iKGm G" ja-kanji)
("w" "RN" "RVLm RN" "RVLm RN" ja-kanji)
("x" "e" "LVm e" "LVm e" ja-kanji)
("y" "V" "N}wj L" "N}wj L" ja-kanji)
("z" "e" "RKm L[W" "RKm L[W" ja-kanji)
("{" "N" "Nm N" "Nm N" ja-kanji)
("|" "L[" "JSm L[" "JSm L[" ja-kanji)
("}" "J" "tEm J" "tEm J" ja-kanji)
("~" "W" "qcMO}m W" "qcMO}m W" ja-kanji)
("" "L" "qNm L" "qNm L" ja-kanji)
("" "L" "N}wj L" "N}wj L" ja-kanji)
("" "" "Nm " "Nm " ja-kanji)
("" "R" "c~m R" "c~m R" ja-kanji)
("" "wL" "eVm wL" "eVm wL" ja-kanji)
("" "c" "JCm c" "JCm c" ja-kanji)
("" "W" "m L[W" "m L[W" ja-kanji)
("" "x" "xm L[W" "xm L[W" ja-kanji)
("" "Cc" "Vj[j C`" "Vj[j C`" ja-kanji)
("" "`[" "Vj[j iJ" "Vj[j iJ" ja-kanji)
("" "PC" "nJm PC" "nJm PC" ja-kanji)
("" "`[" "nJm `[" "nJm `[" ja-kanji)
("" "eL" "~`rNm eL" "~`rNm eL" ja-kanji)
("" "g[" "Vj[j \g" "Vj[j \g" ja-kanji)
("" "j[" "jm L[W" "jm L[W" ja-kanji)
("" "JC" "Vj[j JC" "Vj[j JC" ja-kanji)
("" "R[" "OAEm R[" "OAEm R[" ja-kanji)
("" "ZL" "Agm ZL" "Agm ZL" ja-kanji)
("" "iC" "Xi`m iC" "Xi`m iC" ja-kanji)
("" "L[" "Vj[j g" "Vj[j g" ja-kanji)
("" "PC" "R~`m CW" "R~`m CW" ja-kanji)
("" "V" "VWm V" "VWm V" ja-kanji)
("" "V[" "V[[m V[" "V[[m V[" ja-kanji)
("" "eC" "^N}VCm eC" "^N}VCm eC" ja-kanji)
("" "eL" "nJm eL" "nJm eL" ja-kanji)
("" "z" "mKm z" "mKm z" ja-kanji)
("" "RN" "Vj[j ^j" "Vj[j ^j" ja-kanji)
("" "C" "iij CNm C" "iij CNm C" ja-kanji)
("" "L" "~`m L" "~`m L" ja-kanji)
("" "^c" "^bXm L[W" "^bXm L[W" ja-kanji)
("" "z[" "zgoVm z[" "zgoVm z[" ja-kanji)
("" "Ac" "gm Ac" "gm Ac" ja-kanji)
("" "J" "nJm J" "nJm J" ja-kanji)
("" "R[" "Cg}m R[" "Cg}m R[" ja-kanji)
("" "V[" "V[m CW" "V[m CW" ja-kanji)
("" "V[" "cCm V[" "cCm V[" ja-kanji)
("" "eC" "TXKjm eC" "TXKjm eC" ja-kanji)
("" "" "RGm " "RGm " ja-kanji)
("" "i" "Vj[j ~i~" "Vj[j ~i~" ja-kanji)
("" "R[" "AEm R[" "AEm R[" ja-kanji)
("" "eC" "eCm L[W" "eCm L[W" ja-kanji)
("" "S[" "A\um S[" "A\um S[" ja-kanji)
("" "g" "mKm g" "mKm g" ja-kanji)
("" "W[" "Om W[" "Om W[" ja-kanji)
("" "YC" "YCm CW" "YCm CW" ja-kanji)
("" "`" "INm L[W" "INm L[W" ja-kanji)
("" "JC" "AEm JC" "AEm JC" ja-kanji)
("" "L" "X~Jjm L" "X~Jjm L" ja-kanji)
("" "}C" "XMm }C" "XMm }C" ja-kanji)
("" "[" "JGm [" "JGm [" ja-kanji)
("" "w" "A^m L[W" "A^m L[W" ja-kanji)
("" "w" "A^m CW" "A^m CW" ja-kanji)
("" "" "Om " "Om " ja-kanji)
("" "g" "m g" "m g" ja-kanji)
("" "J" "J^m J" "J^m J" ja-kanji)
("" "L[" "IJm L[" "IJm L[" ja-kanji)
("" "V[" "ZCm qgc" "ZCm qgc" ja-kanji)
("" "GC" "\m ~Rm i" "\m ~Rm i" ja-kanji)
("" "QL" "XL}m QL" "XL}m QL" ja-kanji)
("" "R" "ghm R" "ghm R" ja-kanji)
("" "t" "Nm t" "Nm t" ja-kanji)
("" "KN" "Cm Njm i" "Cm Njm i" ja-kanji)
("" "X[" "[Vm ZC`" "[Vm ZC`" ja-kanji)
("" "q" "CiJm q" "CiJm q" ja-kanji)
("" "^" "J^m ^" "J^m ^" ja-kanji)
("" "" "gim CW" "gim CW" ja-kanji)
("" "eC" "Em eC" "Em eC" ja-kanji)
("" "^" "^fLm ^" "^fLm ^" ja-kanji)
("" "g[" "g[Vm g[" "g[Vm g[" ja-kanji)
("" "J" "^Pim J" "^Pim J" ja-kanji)
("" "\" "TPwj m M" "TPwj m M" ja-kanji)
("" "C" "Em C" "Em C" ja-kanji)
("" "C" "XYNm C" "XYNm C" ja-kanji)
("" "eC" "tcJCm eC" "tcJCm eC" ja-kanji)
("" "TN" "Xm CW" "Xm CW" ja-kanji)
("" "XC" "Em L[W" "Em L[W" ja-kanji)
("" "" "~m " "~m " ja-kanji)
("" "JC" "VIJm JC" "VIJm JC" ja-kanji)
("" "C" "CXm L[W" "CXm L[W" ja-kanji)
("" "PC" "Xm PC" "Xm PC" ja-kanji)
("" "[" "huNm [" "huNm [" ja-kanji)
("" "L" "LLm L" "LLm L" ja-kanji)
("" "C" "A}UPm C" "A}UPm C" ja-kanji)
("" "N" "Em N" "Em N" ja-kanji)
("" "W[" "JXm L[W" "JXm L[W" ja-kanji)
("" "L" "q}m L" "q}m L" ja-kanji)
("" "[" "EOXm [" "EOXm [" ja-kanji)
("" "VN" "Xm L[W" "Xm L[W" ja-kanji)
("" "" "ITm " "ITm " ja-kanji)
("" "g[" "Jlwj J^i" "Jlwj J^i" ja-kanji)
("" "nc" "Jlwj n`" "Jlwj n`" ja-kanji)
("" "t" "J}m CW" "J}m CW" ja-kanji)
("" "RN" "Jlwj `J" "Jlwj `J" ja-kanji)
("" "P" "cMm CW" "cMm CW" ja-kanji)
("" "T" "JUVm T" "JUVm T" ja-kanji)
("" "V" "Jlwj i" "Jlwj i" ja-kanji)
("" "L" "qgVCm L" "qgVCm L" ja-kanji)
("" "L" "^`Lm L" "^`Lm L" ja-kanji)
("" "V[" "XNCgm V[" "XNCgm V[" ja-kanji)
("" "^N" "^Nm NW" "^Nm NW" ja-kanji)
("" "`[" "{^m `[" "{^m `[" ja-kanji)
("" "o" "C^Klm o" "C^Klm o" ja-kanji)
("" "Gc" "}TJm Gc" "}TJm Gc" ja-kanji)
("" "J" "NrJZm J" "NrJZm J" ja-kanji)
("" "L" "IILCm L" "IILCm L" ja-kanji)
("" "Q" "Jlwj Q" "Jlwj Q" ja-kanji)
("" "N" "JMm N" "JMm N" ja-kanji)
("" "V" "i^m V" "i^m V" ja-kanji)
("" "ec" "NKlm ec" "NKlm ec" ja-kanji)
("" "e" "Jlwj ^" "Jlwj ^" ja-kanji)
("" "z[" "Jim z[" "Jim z[" ja-kanji)
("" "ZL" "Jlwj CV" "Jlwj CV" ja-kanji)
("" "J" "Ncm J" "Ncm J" ja-kanji)
("" "V" "Jlwj V" "Jlwj V" ja-kanji)
("" "Z" "nJm Z" "nJm Z" ja-kanji)
("" "Z" "mm Z" "mm Z" ja-kanji)
("" "[" "Jlwj E" "Jlwj E" ja-kanji)
("" "L[" "nT~m L[" "nT~m L[" ja-kanji)
("" "V[" "Trm V[" "Trm V[" ja-kanji)
("" "V[" "gPm V[" "gPm V[" ja-kanji)
("" "{[" "LbTLm {[" "LbTLm {[" ja-kanji)
("" "A" "VRm A" "VRm A" ja-kanji)
("" "G" "Ji}m G" "Ji}m G" ja-kanji)
("" "L[" "Jlwj L[" "Jlwj L[" ja-kanji)
("" "R" "tTOm R" "tTOm R" ja-kanji)
("@" "V" "J^m ^C" "J^m ^C" ja-kanji)
("A" "Z" "[jm L[W" "[jm L[W" ja-kanji)
("B" "\[" "hm \[" "hm \[" ja-kanji)
("C" "eC" "VRm eC" "VRm eC" ja-kanji)
("D" "L`" "Jlwj V" "Jlwj V" ja-kanji)
("E" "J" "Jlwj ni" "Jlwj ni" ja-kanji)
("F" "u" "Jlwj u" "Jlwj u" ja-kanji)
("G" "J" "Jlwj J" "Jlwj J" ja-kanji)
("H" "R[" "Jlwj R[" "Jlwj R[" ja-kanji)
("I" "V" "VL[m n" "VL[m n" ja-kanji)
("J" "`[" "V`[m `[" "V`[m `[" ja-kanji)
("K" "`" "Jlwj W" "Jlwj W" ja-kanji)
("L" "Cc" "Jlwj }X" "Jlwj }X" ja-kanji)
("M" "R[" "ixVmMm R[" "ixVmMm R[" ja-kanji)
("N" "`" "VYm L[W" "VYm L[W" ja-kanji)
("O" "[" "CK^m [" "CK^m [" ja-kanji)
("P" "\[" "ecwj IN" "ecwj IN" ja-kanji)
("Q" "I[" "~iSVm I[" "~iSVm I[" ja-kanji)
("R" "R[" "Jlwj J^C" "Jlwj J^C" ja-kanji)
("S" "T" "^Klm T" "^Klm T" ja-kanji)
("T" "V[" "Jlwj h" "Jlwj h" ja-kanji)
("U" "V[" "Jlwj V[" "Jlwj V[" ja-kanji)
("V" "\N" "Wm \N" "Wm \N" ja-kanji)
("W" "}" "Rem }" "Rem }" ja-kanji)
("X" "[" "CV~m w" "CV~m w" ja-kanji)
("Y" "" "NTm " "NTm " ja-kanji)
("Z" "[" "zm [" "zm [" ja-kanji)
("[" "A" "VRm A" "VRm A" ja-kanji)
("\" "^" "com ^" "com ^" ja-kanji)
("]" "^C" "CVYLm ^C" "CVYLm ^C" ja-kanji)
("^" "h[" "hm h[" "hm h[" ja-kanji)
("_" "n" "Jlwj o" "Jlwj o" ja-kanji)
("`" "[" "NTm [" "NTm [" ja-kanji)
("a" "J" "rm J" "rm J" ja-kanji)
("b" "Z" "m~m Z" "m~m Z" ja-kanji)
("c" "ec" "ecm L[W" "ecm L[W" ja-kanji)
("d" "ec" "ecm CW" "ecm CW" ja-kanji)
("e" "g[" "RWm g[" "RWm g[" ja-kanji)
("f" "o" "E}m NrJU" "E}m NrJU" ja-kanji)
("g" "J" "JK~m CW" "JK~m CW" ja-kanji)
("h" "`[" "Cm L[W" "Cm L[W" ja-kanji)
("i" "R[" "AKlm L[W" "AKlm L[W" ja-kanji)
("j" "VN" "gJXm VN" "gJXm VN" ja-kanji)
("k" "" "Xm " "Xm " ja-kanji)
("l" "[" "XYm [" "XYm [" ja-kanji)
("m" "" "Cm " "Cm " ja-kanji)
("n" "" "Cm NW" "Cm NW" ja-kanji)
("o" "N" "JMm N" "JMm N" ja-kanji)
("p" "J" "uLJm J" "uLJm J" ja-kanji)
("q" "W[" "PkLm W[" "PkLm W[" ja-kanji)
("r" "T" "Lm L[W" "Lm L[W" ja-kanji)
("s" "T" "Lm T" "Lm T" ja-kanji)
("t" "" "hm " "hm " ja-kanji)
("u" "" "XYm " "XYm " ja-kanji)
("v" "JN" "Nm JN" "Nm JN" ja-kanji)
("w" "TN" "EKcm TN" "EKcm TN" ja-kanji)
("x" "T" "JkLm T" "JkLm T" ja-kanji)
("y" "wC" "gWm CW" "gWm CW" ja-kanji)
("z" "Z" "K}Gj }" "K}Gj }" ja-kanji)
("{" "r" "EGj r" "EGj r" ja-kanji)
("|" "XC" "K}Gj ~Y" "K}Gj ~Y" ja-kanji)
("}" "R[" "XCm R[" "XCm R[" ja-kanji)
("~" "h[" "TKVCm h[" "TKVCm h[" ja-kanji)
("" "MN" "K}Gj ^}" "K}Gj ^}" ja-kanji)
("" "PC" "PCV[m PC" "PCV[m PC" ja-kanji)
("" "R[" "J`hLm R[" "J`hLm R[" ja-kanji)
("" "" "R[m " "R[m " ja-kanji)
("" "Ac" "tTOm Ac" "tTOm Ac" ja-kanji)
("" "G" "G}m G" "G}m G" ja-kanji)
("" "G" "LZCXm G" "LZCXm G" ja-kanji)
("" "CL" "VLCm CL" "VLCm CL" ja-kanji)
("" "Jc" "Jb^cm Jc" "Jb^cm Jc" ja-kanji)
("" "Jc" "Jcm CW" "Jcm CW" ja-kanji)
("" "PL" "VXJm PL" "VXJm PL" ja-kanji)
("" "g" "K}Gj m" "K}Gj m" ja-kanji)
("" "" "^Pim " "^Pim " ja-kanji)
("" "Pc" "JPm Pc" "JPm Pc" ja-kanji)
("" "R[" "m gr" "m gr" ja-kanji)
("" "`" "`j[m `" "`j[m `" ja-kanji)
("" "J" "ZLm L[W" "ZLm L[W" ja-kanji)
("" "Z" "ALJm Z" "ALJm Z" ja-kanji)
("" "^c" "K}Gj ^c" "K}Gj ^c" ja-kanji)
("" "wL" "qNm wL" "qNm wL" ja-kanji)
("" "Z" "~`m Z" "~`m Z" ja-kanji)
("" "N" "tTKm N" "tTKm N" ja-kanji)
("" "Q" "Cm RNC" "Cm RNC" ja-kanji)
("" "V" "Agm V" "Agm V" ja-kanji)
("" "n" "cc~m n" "cc~m n" ja-kanji)
("" "nN" "RUgwj qN" "RUgwj qN" ja-kanji)
("" "^" "RUgwj A" "RUgwj A" ja-kanji)
("" "[" "CVCm [" "CVCm [" ja-kanji)
("" "J" "I`Cm CW" "I`Cm CW" ja-kanji)
("" "L[" "}ACm L[" "}ACm L[" ja-kanji)
("" "V[" "m{m V[" "m{m V[" ja-kanji)
("" "Z" "ZZCV[" "ZZCV[" ja-kanji)
("" "`N" "m{m `N" "m{m `N" ja-kanji)
("" "g[" "RUgwj W" "RUgwj W" ja-kanji)
("" "XC" "zgm XC" "zgm XC" ja-kanji)
("" "X[" "X~m X[" "X~m X[" ja-kanji)
("" "R[" "zm R[" "zm R[" ja-kanji)
("" "AC" "Z}Cm AC" "Z}Cm AC" ja-kanji)
("" "C" "CZLm C" "CZLm C" ja-kanji)
("" "JC" "RUgwj Ij" "RUgwj Ij" ja-kanji)
("" "P" "PVCm L[W" "PVCm L[W" ja-kanji)
("" "XC" "glm YC" "glm YC" ja-kanji)
("" "C" "JNm L[W" "JNm L[W" ja-kanji)
("" "Vc" "T_m Vc" "T_m Vc" ja-kanji)
("" "Vc" "T`m Vc" "T`m Vc" ja-kanji)
("" "[" "IJm [" "IJm [" ja-kanji)
("" "^C" "Ium ^C" "Ium ^C" ja-kanji)
("" "C" "V^KEm L[W" "V^KEm L[W" ja-kanji)
("" "XC" "gm XC" "gm XC" ja-kanji)
("" "V" "~nm V" "~nm V" ja-kanji)
("" "V" "XO^m V" "XO^m V" ja-kanji)
("" "`" "LWm `" "LWm `" ja-kanji)
("" "[" "Om [" "Om [" ja-kanji)
("" "][" "Rwj V[" "Rwj V[" ja-kanji)
("" "Uc" "}Wm L[W" "}Wm L[W" ja-kanji)
("" "JN" "jJm JN" "jJm JN" ja-kanji)
("" "`[" "`[nm `[" "`[nm `[" ja-kanji)
("" "nN" "q[m nN" "q[m nN" ja-kanji)
("" "V[" "\m V[" "\m V[" ja-kanji)
("" "eC" "CJY`m eC" "CJY`m eC" ja-kanji)
("" "nC" "IIAm nC" "IIAm nC" ja-kanji)
("" "QC" "jWm QC" "jWm QC" ja-kanji)
("" "V[" "VoVm \[" "VoVm \[" ja-kanji)
("" "e" "EIXm e" "EIXm e" ja-kanji)
("" "q" "m q" "m q" ja-kanji)
("" "" "iKAm " "iKAm " ja-kanji)
("" "GC" "~]m GC" "~]m GC" ja-kanji)
("" "[" "A}_m [" "A}_m [" ja-kanji)
("" "C" "iKAm C" "iKAm C" ja-kanji)
("" "T" "Am T" "Am T" ja-kanji)
("" "wL" "CJY`m wL" "CJY`m wL" ja-kanji)
("" "ZC" "nm ZC" "nm ZC" ja-kanji)
("" "oC" "c`Om oC" "c`Om oC" ja-kanji)
("" "AC" "m AC" "m AC" ja-kanji)
("" "^C" "AC^Cm ^C" "AC^Cm ^C" ja-kanji)
("" "C" "^}m L[W" "^}m L[W" ja-kanji)
("" "L" "Nj L" "Nj L" ja-kanji)
("" "AC" "AC^Cm AC" "AC^Cm AC" ja-kanji)
("" "ZC" "VYJm L[W" "VYJm L[W" ja-kanji)
("" "R[" "^KEm R[" "^KEm R[" ja-kanji)
("" "z[" "jLrm z[" "jLrm z[" ja-kanji)
("" "e" "wj ~" "wj ~" ja-kanji)
("" "[" "GN{m [" "GN{m [" ja-kanji)
("" "N" "Ncm N" "Ncm N" ja-kanji)
("" "W" "Wm CW" "Wm CW" ja-kanji)
("" "W" "ViJm W" "ViJm W" ja-kanji)
("" "iC" "Jwj E`" "Jwj E`" ja-kanji)
("" "I[" "iKCm I[" "iKCm I[" ja-kanji)
("" "^" "iVKm ^" "iVKm ^" ja-kanji)
("" "q" "Jwj J" "Jwj J" ja-kanji)
("" "}c" "Jwj XG" "Jwj XG" ja-kanji)
("" "wC" "Jwj wC" "Jwj wC" ja-kanji)
("" "AC" "Wm AC" "Wm AC" ja-kanji)
("" "L[" "^olm L[" "^olm L[" ja-kanji)
("" "Rn[" "Jwj W[Q" "Jwj W[Q" ja-kanji)
("" "g[" "Ncm g[" "Ncm g[" ja-kanji)
("" "Jc" "JOcm Jc" "JOcm Jc" ja-kanji)
("" "V[" "VKCm V[" "VKCm V[" ja-kanji)
("" "W[" "iXm W[" "iXm W[" ja-kanji)
("" "g[" "wCLm g[" "wCLm g[" ja-kanji)
("" "q" "tCSm q" "tCSm q" ja-kanji)
("" "_c" "_^Wm _c" "_^Wm _c" ja-kanji)
("" "Z" "Jwj Z" "Jwj Z" ja-kanji)
("" "oc" "NcV^m oc" "NcV^m oc" ja-kanji)
("" "C" "iVKm C" "iVKm C" ja-kanji)
("" "g[" "g[JCm g[" "g[JCm g[" ja-kanji)
("" "LN" "jm LN" "jm LN" ja-kanji)
("" "ZC" "i}Xm ZC" "i}Xm ZC" ja-kanji)
("" "ZC" "i}Xm CW" "i}Xm CW" ja-kanji)
("" "L[" "cCjm L[" "cCjm L[" ja-kanji)
("" "V[" "EcNVCm V[" "EcNVCm V[" ja-kanji)
("" "C" "qrNm CW" "qrNm CW" ja-kanji)
("" "R[" "mhm R[" "mhm R[" ja-kanji)
("" "V[" "zm V[" "zm V[" ja-kanji)
("" "PC" "PCYCm PC" "PCYCm PC" ja-kanji)
("" "C" "ASm C" "ASm C" ja-kanji)
("" "Lc" "NrXWm Lc" "NrXWm Lc" ja-kanji)
("" "J" "EiYNm J" "EiYNm J" ja-kanji)
("" "^C" "NYm ^C" "NYm ^C" ja-kanji)
("" "J" "cum J" "cum J" ja-kanji)
("" "K" "JIm ZCW" "JIm ZCW" ja-kanji)
("" "TC" "Gm TC" "Gm TC" ja-kanji)
("" "Z" "tGm Z" "tGm Z" ja-kanji)
("" "P" "ALm L[W" "ALm L[W" ja-kanji)
("" "r" "VJm r" "VJm r" ja-kanji)
("@" "" "`[Rcm " "`[Rcm " ja-kanji)
("A" "J" "zI{lm J" "zI{lm J" ja-kanji)
("B" "V[" "RJ~m V[" "RJ~m V[" ja-kanji)
("C" "J" "V^j J[" "V^j J[" ja-kanji)
("D" "Tc" "nem Tc" "nem Tc" ja-kanji)
("E" "^C" "^Ct[m ^C" "^Ct[m ^C" ja-kanji)
("F" "N" "cWJ[m N" "cWJ[m N" ja-kanji)
("G" "q[" "qKGm q[" "qKGm q[" ja-kanji)
("H" "q[" "q[m CW" "q[m CW" ja-kanji)
("I" "q[" "{[t[m q[" "{[t[m q[" ja-kanji)
("J" "g" "V}W[" "V}W[" ja-kanji)
("K" "I" "ALm I" "ALm I" ja-kanji)
("L" "R[" "M[Um R[" "M[Um R[" ja-kanji)
("M" "Q" "Vm Q" "Vm Q" ja-kanji)
("N" "_C" "EGm _C" "EGm _C" ja-kanji)
("O" "z" "[Vm z" "[Vm z" ja-kanji)
("P" "" "A}m L[W" "A}m L[W" ja-kanji)
("Q" "A" "ARm A" "ARm A" ja-kanji)
("R" "VN" "JUm VN" "JUm VN" ja-kanji)
("S" "Z" "niPm Z" "niPm Z" ja-kanji)
("T" "^" "XXm ^" "XXm ^" ja-kanji)
("U" "wC" "`m wC" "`m wC" ja-kanji)
("V" "R" "Jm R" "Jm R" ja-kanji)
("W" "ec" "T{^x" "T{^x" ja-kanji)
("X" "L" "LKm L" "LKm L" ja-kanji)
("Y" "[" "VNwj [" "VNwj [" ja-kanji)
("Z" "E" "Ehm E" "Ehm E" ja-kanji)
("[" "L" "EGm L" "EGm L" ja-kanji)
("\" "}" "}W[m }" "}W[m }" ja-kanji)
("]" "C" "XGm C" "XGm C" ja-kanji)
("^" "L" "INmm L" "INmm L" ja-kanji)
("_" "L" "EGm L" "EGm L" ja-kanji)
("`" "W[" "^Jm W[" "^Jm W[" ja-kanji)
("a" "Z" "\iGm Z" "\iGm Z" ja-kanji)
("b" "g[" "T{m g[" "T{m g[" ja-kanji)
("c" "L" "zI{lm L" "zI{lm L" ja-kanji)
("d" "JN" "NrLm JN" "NrLm JN" ja-kanji)
("e" "tN" "JoVCm tN" "JoVCm tN" ja-kanji)
("f" "M" "Acm M" "Acm M" ja-kanji)
("g" "q[" "Vm Om q[" "Vm Om q[" ja-kanji)
("h" "u" "E}wj t~" "E}wj t~" ja-kanji)
("i" "V" "E}wj " "E}wj " ja-kanji)
("j" "V" "nCm V" "nCm V" ja-kanji)
("k" "_" "N_m _" "N_m _" ja-kanji)
("l" "^C" "juCm ^C" "juCm ^C" ja-kanji)
("m" "h" "juCm h" "juCm h" ja-kanji)
("n" "JC" "IhNm JC" "IhNm JC" ja-kanji)
("o" "nN" "}_m nN" "}_m nN" ja-kanji)
("p" "N" "N_m N" "N_m N" ja-kanji)
("q" "V[" "E}wj V[" "E}wj V[" ja-kanji)
("r" "J" "AE}m J" "AE}m J" ja-kanji)
("s" "V" "nCm V" "nCm V" ja-kanji)
("t" "eC" "nZm eC" "nZm eC" ja-kanji)
("u" "L" "Lm L" "Lm L" ja-kanji)
("v" "XC" "AVQm XC" "AVQm XC" ja-kanji)
("w" "w" "ium w" "ium w" ja-kanji)
("x" "w" "_}Xm w" "_}Xm w" ja-kanji)
("y" "P" "A}`m P" "A}`m P" ja-kanji)
("z" "\[" "TOm L[W" "TOm L[W" ja-kanji)
("{" "N" "JPm L[W" "JPm L[W" ja-kanji)
("|" "T" "\GE}m T" "\GE}m T" ja-kanji)
("}" "oN" "}bVOm oN" "}bVOm oN" ja-kanji)
("~" "q[" "VJQm q[" "VJQm q[" ja-kanji)
("" "" "om " "om " ja-kanji)
("" "L[" "ISm L[" "ISm L[" ja-kanji)
("" "M[" "cCm M[" "cCm M[" ja-kanji)
("" "GL" "GLm L[W" "GLm L[W" ja-kanji)
("" "P" "VVm L[W" "VVm L[W" ja-kanji)
("" "V[" "jJm V[" "jJm V[" ja-kanji)
("" "" "om " "om " ja-kanji)
("" "L" "Zm E}m L" "Zm E}m L" ja-kanji)
("" "W[" "E}wj W[" "E}wj W[" ja-kanji)
("" "J" "Rum J" "Rum J" ja-kanji)
("" "q[" "E} TrL" "E} TrL" ja-kanji)
("" "" "ixm " "ixm " ja-kanji)
("" "J" "zlwj zX" "zlwj zX" ja-kanji)
("" "TC" "TCRm TC" "TCRm TC" ja-kanji)
("" "JN" "NTb^ zlJN" "NTb^ zlJN" ja-kanji)
("" "q" "m q" "m q" ja-kanji)
("" "" "hNm " "hNm " ja-kanji)
("" "hN" "hNm hN" "hNm hN" ja-kanji)
("" "YC" "Xlm L[W" "Xlm L[W" ja-kanji)
("" "^C" "J_m L[W" "J_m L[W" ja-kanji)
("" "\[" "^JCj \[" "^JCj \[" ja-kanji)
("" "q[" "J~KVm q[" "J~KVm q[" ja-kanji)
("" "eC" "JWm eC" "JWm eC" ja-kanji)
("" "z[" "je Cm z[" "je Cm z[" ja-kanji)
("" "{[" "^K~m {[" "^K~m {[" ja-kanji)
("" "[" "qQm [" "qQm [" ja-kanji)
("" "`[" "EiCm `[" "EiCm `[" ja-kanji)
("" "nc" "J~m L[W" "J~m L[W" ja-kanji)
("" "tc" "jeCm tc" "jeCm tc" ja-kanji)
("" "z[" "^om z[" "^om z[" ja-kanji)
("" "LN" "}Qm LN" "}Qm LN" ja-kanji)
("" "Lc" "^uTm Lc" "^uTm Lc" ja-kanji)
("" "V[" "Xm V[" "Xm V[" ja-kanji)
("" "}" "Jcm }" "Jcm }" ja-kanji)
("" "V" "qQm V" "qQm V" ja-kanji)
("" "J" "ghm J" "ghm J" ja-kanji)
("" "q" "rdm q" "rdm q" ja-kanji)
("" "[" "^eK~m [" "^eK~m [" ja-kanji)
("" "g[" "g[K}Gm g[" "g[K}Gm g[" ja-kanji)
("" "h[" "TOm h[" "TOm h[" ja-kanji)
("" "R[" "J`hLm R[" "J`hLm R[" ja-kanji)
("" "PL" "ZOm PL" "ZOm PL" ja-kanji)
("" "g[" "^^JEm L[W" "^^JEm L[W" ja-kanji)
("" "L[" "NWm L[" "NWm L[" ja-kanji)
("" "`[" "uNm `[" "uNm `[" ja-kanji)
("" "JN" "J}m JN" "J}m JN" ja-kanji)
("" "nN" "^}VCm nN" "^}VCm nN" ja-kanji)
("" "oc" "qfm oc" "qfm oc" ja-kanji)
("" "M" "MVm M" "MVm M" ja-kanji)
("" "[" "[[m [" "[[m [" ja-kanji)
("" "[" "[[m [" "[[m [" ja-kanji)
("" "`" "`~m `" "`~m `" ja-kanji)
("" "G" "EiTm G" "EiTm G" ja-kanji)
("" "z[" "TJim z[" "TJim z[" ja-kanji)
("" "T" "XVm T" "XVm T" ja-kanji)
("" "q[" "TJim q" "TJim q" ja-kanji)
("" "z[" "TJim Ar" "TJim Ar" ja-kanji)
("" "V[" "TJim JWJ" "TJim JWJ" ja-kanji)
("" "g[" "TJim RmV" "TJim RmV" ja-kanji)
("" "A" "TJim AR[" "TJim AR[" ja-kanji)
("" "KC" "TJim n" "TJim n" ja-kanji)
("" "V" "XVm V" "XVm V" ja-kanji)
("" "L[" "TJim o" "TJim o" ja-kanji)
("" "R" "TJiwj PC" "TJiwj PC" ja-kanji)
("" "V" "TJim n[" "TJim n[" ja-kanji)
("" "V[" "TJim ^R" "TJim ^R" ja-kanji)
("" "z" "TJim CJ" "TJim CJ" ja-kanji)
("" "o" "TJim EOC" "TJim EOC" ja-kanji)
("" "L" "TJim JYmR" "TJim JYmR" ja-kanji)
("" "" "TJim R`" "TJim R`" ja-kanji)
("" "GL" "TJim X" "TJim X" ja-kanji)
("" "QC" "TV[EI" "TV[EI" ja-kanji)
("" "R" "TJim nS" "TJim nS" ja-kanji)
("" "V" "TJim {" "TJim {" ja-kanji)
("" "q" "TJim jV" "TJim jV" ja-kanji)
("" "\[" "TJim AW" "TJim AW" ja-kanji)
("" "I" "TJim hW[" "TJim hW[" ja-kanji)
("" "R" "TJim V`" "TJim V`" ja-kanji)
("" "l" "TJim i}Y" "TJim i}Y" ja-kanji)
("" "J" "TJim Gr" "TJim Gr" ja-kanji)
("" "J" "TJim ^" "TJim ^" ja-kanji)
("" "R[" "TJim qKC" "TJim qKC" ja-kanji)
("" "TC" "Gm TC" "Gm TC" ja-kanji)
("" "V[" "TJim hW[" "TJim hW[" ja-kanji)
("" "V" "TJim T" "TJim T" ja-kanji)
("" "`[" "TJim JC" "TJim JC" ja-kanji)
("" "tN" "TJim Ar" "TJim Ar" ja-kanji)
("" "" "TJim jV" "TJim jV" ja-kanji)
("" "C" "TJim C" "TJim C" ja-kanji)
("" "I" "TJim CV" "TJim CV" ja-kanji)
("" "I" "CVm CW" "CVm CW" ja-kanji)
("" "J" "TJim }" "TJim }" ja-kanji)
("" "V" "TJim u" "TJim u" ja-kanji)
("" "[" "TJij gh}" "TJij gh}" ja-kanji)
("" "W" "TJim n^n^" "TJim n^n^" ja-kanji)
("" "R[" "TJim AR[" "TJim AR[" ja-kanji)
("" "S[" "h[ucm Xb|" "h[ucm Xb|" ja-kanji)
("" "V[" "TJim ^R" "TJim ^R" ja-kanji)
("" "q[" "ELuNm q[" "ELuNm q[" ja-kanji)
("" "L" "TJim LX" "TJim LX" ja-kanji)
("" "JC" "TJim i}X" "TJim i}X" ja-kanji)
("" "C" "TJim n" "TJim n" ja-kanji)
("" "V[" "TJim tJ" "TJim tJ" ja-kanji)
("" "" "TJim XYL" "TJim XYL" ja-kanji)
("" "t" "gm J" "gm J" ja-kanji)
("" "t" "Jm ]NW" "Jm ]NW" ja-kanji)
("" "j" "gm jI" "gm jI" ja-kanji)
("" "A" "gm JX" "gm JX" ja-kanji)
("" "K" "Jm CW" "Jm CW" ja-kanji)
("" "K" "Jm NW" "Jm NW" ja-kanji)
("" "PL" "gm Y" "gm Y" ja-kanji)
("" "`" "gm `" "gm `" ja-kanji)
("" "Cc" "nCm Cc" "nCm Cc" ja-kanji)
("" "I[" "gm IVh" "gm IVh" ja-kanji)
("" "I[" "EOCXm L[W" "EOCXm L[W" ja-kanji)
("" "R" "gm VR" "gm VR" ja-kanji)
("" "V" "gm tN[" "gm tN[" ja-kanji)
("" "V" "LVm V" "LVm V" ja-kanji)
("" "_" "_`[m _" "_`[m _" ja-kanji)
("" "C" "gm ZLC" "gm ZLC" ja-kanji)
("" "R[" "gm SCTM" "gm SCTM" ja-kanji)
("" "R[" "gm CGog" "gm CGog" ja-kanji)
("" "{[" "gm gL" "gm gL" ja-kanji)
("" "`" "gm `h" "gm `h" ja-kanji)
("" "`[" "~~wj g" "~~wj g" ja-kanji)
("@" "K" "gm K`[" "gm K`[" ja-kanji)
("A" "K" "K`[m K" "K`[m K" ja-kanji)
("B" "JN" "Jhjg" "Jhjg" ja-kanji)
("C" "P" "^IThm P" "^IThm P" ja-kanji)
("D" "u" "tiVEYm u" "tiVEYm u" ja-kanji)
("E" "PL" "gm Y" "gm Y" ja-kanji)
("F" "WN" "gm JTTM" "gm JTTM" ja-kanji)
("G" "V" "gm EY" "gm EY" ja-kanji)
("H" "g[" "gm cO~" "gm cO~" ja-kanji)
("I" "g[" "cO~m CW" "cO~m CW" ja-kanji)
("J" "q" "gm qh" "gm qh" ja-kanji)
("K" "" "Gm " "Gm " ja-kanji)
("L" "KN" "gm ~TS" "gm ~TS" ja-kanji)
("M" "E" "g[}m E" "g[}m E" ja-kanji)
("N" "u" "gm Aq" "gm Aq" ja-kanji)
("O" "I[" "gm I[" "gm I[" ja-kanji)
("P" "PC" "jgm L[W" "jgm L[W" ja-kanji)
("Q" "QL" "~Yhm i}G" "~Yhm i}G" ja-kanji)
("R" "Rc" "gm Jh" "gm Jh" ja-kanji)
("S" "WN" "gm q" "gm q" ja-kanji)
("T" "ZL" "gm ZLC" "gm ZLC" ja-kanji)
("U" "e" "^Jm L[W" "^Jm L[W" ja-kanji)
("V" "e" "gm ^J" "gm ^J" ja-kanji)
("W" "[" "gm nV^J" "gm nV^J" ja-kanji)
("X" "V" "gm Ah" "gm Ah" ja-kanji)
("Y" "V" "gm VR" "gm VR" ja-kanji)
("Z" "Cc" "gm VM" "gm VM" ja-kanji)
("[" "V[" "gm ~\TUC" "gm ~\TUC" ja-kanji)
("\" "o" "gm n" "gm n" ja-kanji)
("]" "[" "gm VL" "gm VL" ja-kanji)
("^" "KN" "gm E\" "gm E\" ja-kanji)
("_" "I[" "gm I[" "gm I[" ja-kanji)
("`" "J" "gm R[mg" "gm R[mg" ja-kanji)
("a" "" "C`[m " "C`[m " ja-kanji)
("b" "" "VIm " "VIm " ja-kanji)
("c" "J" "VIPm J" "VIPm J" ja-kanji)
("d" "G" "VIm ZCW" "VIm ZCW" ja-kanji)
("e" "\" "ACm \" "ACm \" ja-kanji)
("f" "V" "IIVJm V" "IIVJm V" ja-kanji)
("g" "r" "giJCm r" "giJCm r" ja-kanji)
("h" "S" "IWJm S" "IWJm S" ja-kanji)
("i" "L" "Lm L" "Lm L" ja-kanji)
("j" "L" "LV[m L" "LV[m L" ja-kanji)
("k" "QC" "JmRm QC" "JmRm QC" ja-kanji)
("l" "W" "WR[WJm W" "WR[WJm W" ja-kanji)
("m" "oN" "Mm L[W" "Mm L[W" ja-kanji)
("n" "t" "tX}m t" "tX}m t" ja-kanji)
("o" "t" "tX}m CW" "tX}m CW" ja-kanji)
("p" "" "\om " "\om " ja-kanji)
("q" "z[" "Ri`m z[" "Ri`m z[" ja-kanji)
("r" "q" "irNm q" "irNm q" ja-kanji)
("s" "R[" "}irm R[" "}irm R[" ja-kanji)
("t" "C" "CCm C" "CCm C" ja-kanji)
("u" "l" "lom l" "lom l" ja-kanji)
("v" "`" "g`m `" "g`m `" ja-kanji)
("w" "P" "NCm P" "NCm P" ja-kanji)
("x" "`c" "V]Pm `c" "V]Pm `c" ja-kanji)
("y" "e" "em L[W" "em L[W" ja-kanji)
("z" "[" "AIOCm [" "AIOCm [" ja-kanji)
("{" "Jc" "KVRC" "KVRC" ja-kanji)
("|" "PC" "CY~m PC" "CY~m PC" ja-kanji)
("}" "g[" "g[m L[W" "g[m L[W" ja-kanji)
("~" "A" "NCm A" "NCm A" ja-kanji)
("" "oC" "Jrm oC" "Jrm oC" ja-kanji)
("" "A" "zNm A" "zNm A" ja-kanji)
("" "gN" "PKm gN" "PKm gN" ja-kanji)
("" "`" "kCgm `" "kCgm `" ja-kanji)
("" "tc" "qUJPm tc" "qUJPm tc" ja-kanji)
("" "z" "CtNm z" "CtNm z" ja-kanji)
("" "{[" "cgm {[" "cgm {[" ja-kanji)
("" "S[" "IIXb|m S[" "IIXb|m S[" ja-kanji)
("" "xc" "xbR[m xc" "xbR[m xc" ja-kanji)
("" "R" "cd~m CW" "cd~m CW" ja-kanji)
("" "g[" "cd~m Ig" "cd~m Ig" ja-kanji)
("" "\" "lY~m NW" "lY~m NW" ja-kanji)
("" "" "C^`m " "C^`m " ja-kanji)
("" "J" "CrLm J" "CrLm J" ja-kanji)
("" "ZC" "\Em L[W" "\Em L[W" ja-kanji)
("" "V" "nm L[W" "nm L[W" ja-kanji)
("" "V" "nKm V" "nKm V" ja-kanji)
("" "ZL" "R}m ZL" "R}m ZL" ja-kanji)
("" "\" "\VNm \" "\VNm \" ja-kanji)
("" "`[" "nKm `[" "nKm `[" ja-kanji)
("" "C" "Cm L[W" "Cm L[W" ja-kanji)
("" "R" "nOLm R" "nOLm R" ja-kanji)
("" "Pc" "JWm Pc" "JWm Pc" ja-kanji)
("" "S" "NC`KCm S" "NC`KCm S" ja-kanji)
("" "TN" "R_m TN" "R_m TN" ja-kanji)
("" "AN" "ANTNm AN" "ANTNm AN" ja-kanji)
("" "E" "Vom E" "Vom E" ja-kanji)
("" "KN" "nOLm KN" "nOLm KN" ja-kanji)
("" "J" "YVm K" "YVm K" ja-kanji)
("" "L" "Jm L[W" "Jm L[W" ja-kanji)
("" "N" "tGm N" "tGm N" ja-kanji)
("" "ME" "^JVm ME" "^JVm ME" ja-kanji)
("" "}L" "RYGm }L" "RYGm }L" ja-kanji)
("" "E" "EnCm E" "EnCm E" ja-kanji)
("" "E" "EcNVC ^}m E" "EcNVC ^}m E" ja-kanji)
("" "" "qLV}m " "qLV}m " ja-kanji)
("" "L" "ALJm L" "ALJm L" ja-kanji)
)))
;;}}}

(emacspeak-m17n-ja-setup-auditory-display-table)
(emacspeak-m17n-ja-setup-character-to-speech-table)

(provide 'emacspeak-ja-tbl)
